/*******************************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestLogEventsLazyPathContentProvider.java,v 1.4 2009/05/15 18:58:26 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.forms.util;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFInvocationEvent;
import org.eclipse.hyades.test.ui.internal.editor.form.base.Partition;
import org.eclipse.jface.viewers.ILazyTreePathContentProvider;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeViewer;

/**
 * <p>This is the content provider used by TPTP Test Log Viewer Events tree.</p>
 
 * <p>It keeps the elements in the tree updated and also allows to show many elements in the tree</p>
 * 
 * 
 * @author      Bianca Xue Jiang
 * @author      Marcelo Paternostro
 * @author      Blazej Kroll
 * @author      Paul Slauenwhite
 * @version     May 15, 2009
 * @since       March 16, 2005
 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
 */
public class TestLogEventsLazyPathContentProvider extends EObjectTreeContentProvider implements ILazyTreePathContentProvider{
	
	/**
	 * List keeps the partitions created for a particular index in the tree. 
	 */
	private List parts = new ArrayList();

	/**
	 * Number of items on a tree to keep without creating partitions.
	 */
	private static final int MAX_ITEMS = 30000;

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.forms.util.EObjectContainerContentProvider#getRegisteredParentChild(java.lang.Object)
	 */
	protected Object getRegisteredParentChild(Object parent)
	{		

		if(parent instanceof Partition)
		{
			return (((Partition)(parent)).getChildren());
		}
		else if(parent instanceof TPFExecutionResult)
		{
			TPFExecutionResult executionResult = (TPFExecutionResult)parent;

			if(executionResult.getExecutionHistory() != null){
				return executionResult.getExecutionHistory().getExecutionEvents();
			}

			return null;
		}
		//added to set the parent/child hierarchy for execution events :BGB(12/2/03)
		else if (parent instanceof TPFExecutionEvent)
		{
			if ( parent instanceof TPFInvocationEvent )
			{
				return (((TPFInvocationEvent)(parent)).getInvokedExecutionResult() );
			}
			else
			{
				TPFExecutionEvent   executionEvent  = (TPFExecutionEvent)parent;

				if ( ( executionEvent.getChildren() != null ) &&
						( executionEvent.getChildren().isEmpty() == false )){
					return( executionEvent.getChildren());
				}
			}
		}
		return null;
	}

	/**
	 * Content's provider constructor
	 * @param eStructuralFeature
	 */
	public TestLogEventsLazyPathContentProvider(EStructuralFeature eStructuralFeature)
	{
		super(eStructuralFeature);

	}

	/**
	 * Return the possible parent paths for the given element. An empty array
	 * is returned if the paths cannot be computed.
	 * @param element
	 *            the element
	 * @return the possible parent paths for the given element
	 */
	public TreePath[] getParents(Object element)
	{
		return null;
	}

	/**
	 * Called when a previously-blank item becomes visible in the TreeViewer.    
	 */
	public void updateElement(final TreePath parentPath, final int index)
	{

		 Object child = null;
		 final Object parent = getElement(parentPath);

			if (parent instanceof Partition){					
				child = ((Partition)(parent)).getChild(index);
 
				
		 }
			else if (parent instanceof TPFExecutionHistory) {

				final TPFExecutionHistory eventLog = (TPFExecutionHistory) parent;
				final int partitionSize = computePartitionSize(eventLog.getExecutionEvents().size());
				
				if (partitionSize > 1) {	
					child = createPartition(eventLog.getExecutionEvents(), partitionSize, index);
				}
				else
				{ 
					child = (TPFExecutionEvent) eventLog.getExecutionEvents().get(index);
				}
			} 

			else if(parent instanceof TPFExecutionResult)
			{ 	

				final TPFExecutionResult executionResult = (TPFExecutionResult)parent;

				if(executionResult.getExecutionHistory() != null){

					int  partitionSize = computePartitionSize(executionResult.getExecutionHistory().getExecutionEvents().size());
					
					if (partitionSize > 1) {
						child = createPartition(executionResult.getExecutionHistory().getExecutionEvents(), partitionSize, index);
					} 
					else {
						child = executionResult.getExecutionHistory().getExecutionEvents().get(index);
					}       	
				}
			}

			else if ( parent instanceof TPFExecutionEvent ) 
			{
				if ( parent instanceof TPFInvocationEvent )
				{            	
					final TPFInvocationEvent invocationEvent  = (TPFInvocationEvent)parent;

					int partitionSize1 = computePartitionSize(invocationEvent.getChildren().size());
					if (partitionSize1 > 1) {
						child = createPartition(invocationEvent.getChildren(), partitionSize1, index);
					}
					else
					{
						child = invocationEvent.getInvokedExecutionResult();
					}                     
				}
				else
				{
					final TPFExecutionEvent  executionEvent  = (TPFExecutionEvent)parent;
					if ( ( executionEvent.getChildren() != null ) &&
							( executionEvent.getChildren().isEmpty() == false ))
					{
						int partitionSize = computePartitionSize(executionEvent.getChildren().size());

						if (partitionSize > 1) {
							child = createPartition(executionEvent.getChildren(), partitionSize, index);
						}
						else
						{
							child = (TPFExecutionEvent)executionEvent.getChildren().get(index);  
						}
					} 
				}
			}

			updateTreeViewer(parentPath, index, child);	  
	  	
	}

	/**
	 * Update the treeViewer with gathered child at specified parent's path index
	 * @author  Blazej Kroll
	 * @param parentPath the path to parent object
	 * @param index the index (position) of the item in the tree
	 * @param child - child element associated to the index
	 */
	private void updateTreeViewer(final TreePath parentPath, final int index, final Object child) {

		if (child != null){
			
			final TreeViewer treeViewer = (TreeViewer)getViewer();										
			
			if (index == 0) {

				final Partition loadingPartition = new Partition(-1,-1,null); // "Loading..." Partition node								
				treeViewer.replace(parentPath, index, (Object)loadingPartition);
			}
								
			int count = 0;
			Object child1 = getRegisteredParentChild(child);
			if(child1 != null)
			{
			  if(child1 instanceof Collection) {
				 
				  	count = ((Collection)child1).size();
			
					if (count > MAX_ITEMS){
						count = computeNumberOfParitions(((Collection) child1).size());
					}
			  } else 
					{
						count = 1;
					}

			}
			
			final int count_in_display = count;							
			
			treeViewer.replace(parentPath, index, child);
			treeViewer.setChildCount(child, count_in_display);
	}
 			
}

	/**
	 * Creates a partition from list of elements with a given partitionSize and starting with a given index within the list
	 * @param list elements from which the partition is to be created (list)
	 * @param partitionSize quantity of the partition
	 * @param index the index of the list from which the partition should take the first element
	 * @return
	 */
	public Partition createPartition(EList list, int partitionSize, int index) {
		int offset = 0 + partitionSize * index; //initial offset is zero;
		int length = list.size();
		int numPartitions = length / partitionSize;
		int remainder = length % partitionSize;
		if (remainder > 0) {
			numPartitions++;
		}

		if (index <= numPartitions - 1) {

			Partition part = new Partition( offset, partitionSize, list);
			if (index == numPartitions - 1)
				part = new Partition( offset, length - offset, list);

			parts.add(part);                                   
			return ((Partition)(parts.get(index)));
		}
		return null;

	}

	/**
	 *  Compute the actual size of a partition, take into consideration the preferred constant value MAX_ITEMS
	 * @param valueQuantity number of items in the list of values to be placed in partitions
	 * @return partitionsSize size of a single partition
	 */
	public static int computePartitionSize(int valueQuantity) {
		int partitionSize = 1;
		try {
			int length = valueQuantity;
			int partitionDepth = 0;
			int preferredSize = MAX_ITEMS; 
			int remainder = length % preferredSize;
			length = length / preferredSize;
			while (length > 0) {
				if (remainder == 0 && length == 1) {
					break;
				}
				partitionDepth++;
				remainder = length % preferredSize;
				length = length / preferredSize;
			}

			for (int i = 0; i < partitionDepth; i++) {
				partitionSize = partitionSize * preferredSize;
			}
		} catch (Exception e) {
		}

		return partitionSize;
	}

	/**
	 * Compute the number of partitions needed to show the whole list of values
	 * @param child1 - Collection of values to be placed in partitions
	 * @return number of partitions needed to place all elements within the limits of a single partition size
	 */
	protected int computeNumberOfParitions(int childrenQuantity) {

		int length = childrenQuantity;
		try {
			int preferredSize = MAX_ITEMS;
			int remainder = length % preferredSize;
			length = length / preferredSize;
			if (remainder > 0) length++;         
		} catch (Exception e) {
		}
		return length;
	}


	/**
	 * Called when the TreeViewer needs an up-to-date child count for the given tree path
	 */
	public void updateChildCount(TreePath treePath, int currentChildCount)
	{

		Object element = getElement(treePath);

		int count = currentChildCount;

		if(element instanceof Partition)
		{
			count = ((Partition)element).getSize();
		}
		else if(element instanceof TPFExecutionHistory)
		{
			count = ((TPFExecutionHistory)element).getExecutionEvents().size();

			if (count > MAX_ITEMS){
				count = computeNumberOfParitions(((TPFExecutionHistory)element).getExecutionEvents().size());
			}
		}	
		else if(element instanceof TPFExecutionResult)
		{
			TPFExecutionResult executionResult = (TPFExecutionResult)element;
			if(executionResult.getExecutionHistory() != null)
			{
				count = executionResult.getExecutionHistory().getExecutionEvents().size();

				if (count > MAX_ITEMS){	                	
					count = computeNumberOfParitions(((TPFExecutionHistory)element).getExecutionEvents().size());
				}
			}
		}
		else if (element instanceof TPFExecutionEvent) {

			if (!( element instanceof TPFInvocationEvent) )
			{

				TPFExecutionEvent   executionEvent  = (TPFExecutionEvent)element;
				if ( ( executionEvent.getChildren() != null ) &&
						( executionEvent.getChildren().isEmpty() == false )){
					count = executionEvent.getChildren().size();
				}

				if (count > MAX_ITEMS){	                	
					count = computeNumberOfParitions(executionEvent.getChildren().size());
				}
			}

		}

		if(count != currentChildCount){
			((TreeViewer)getViewer()).setChildCount(treePath, count);
		}
	}

	/**
	 * Called when the TreeViewer needs up-to-date information whether the node
	 * at the given tree path can be expanded. If the content provider knows the
	 * element at the given tree path, it should respond by calling
	 * {@link TreeViewer#setHasChildren(Object, boolean)}. The content provider
	 * may also choose to call {@link TreeViewer#setChildCount(Object, int)}
	 * instead if it knows the number of children.
	 */	
	public void updateHasChildren(TreePath path)
	{
		Object element = getElement(path);
		TreeViewer viewer = (TreeViewer)getViewer();

		if(element instanceof Partition)
		{
			
			//Partition always has children:
			viewer.setHasChildren(path, true);
		}
		else if(element instanceof TPFExecutionHistory)
		{
			if (((TPFExecutionHistory)element).getExecutionEvents().size() > MAX_ITEMS){
				viewer.setChildCount(path, ((TPFExecutionHistory)element).getExecutionEvents().size() / computePartitionSize(((TPFExecutionHistory)element).getExecutionEvents().size()));					
			}
			else{
				viewer.setChildCount(path, ((TPFExecutionHistory)element).getExecutionEvents().size());
			}
		}	
		else if(element instanceof TPFExecutionResult)
		{
			TPFExecutionResult executionResult = (TPFExecutionResult)element;
			if(executionResult.getExecutionHistory() != null)
			{	
				if (executionResult.getExecutionHistory().getExecutionEvents().size() > MAX_ITEMS){
					viewer.setChildCount(path, (executionResult.getExecutionHistory().getExecutionEvents().size() / computePartitionSize(executionResult.getExecutionHistory().getExecutionEvents().size())));						
				}
				else{
					viewer.setChildCount(path, executionResult.getExecutionHistory().getExecutionEvents().size());	            	
				}
			}
		}
		else if (element instanceof TPFExecutionEvent) {

			if ( element instanceof TPFInvocationEvent) 
			{
				viewer.setHasChildren(path, false);	            	
			}
			else
			{
				TPFExecutionEvent   executionEvent  = (TPFExecutionEvent)element;

				if (executionEvent.getChildren().size() > MAX_ITEMS){
					viewer.setChildCount(path, (executionEvent.getChildren().size() / computePartitionSize(executionEvent.getChildren().size())));						
				}
				else{
					viewer.setHasChildren(path, executionEvent.getChildren() != null && executionEvent.getChildren().size() > 0);
				}	            
			}				
		}

	}

	/**
	 * Returns the element corresponding to the given tree path.
	 * 
	 * @param path tree path
	 * @return model element
	 */
	protected Object getElement(TreePath path) {
		if (path.getSegmentCount() > 0) {
			return path.getLastSegment();
		}
		return getViewer().getInput();
	}


}
