/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ComparablePropertyInputDialog.java,v 1.16 2010/04/02 16:05:32 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.dialog;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.hyades.models.common.configuration.CFGCategory;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.configuration.HyadesClasspathCategory;
import org.eclipse.hyades.models.common.configuration.HyadesRootDirectroyCategory;
import org.eclipse.hyades.models.common.configuration.util.Common_ConfigurationCategoryUtil;
import org.eclipse.hyades.models.common.configuration.util.ICategory;
import org.eclipse.hyades.models.common.configuration.util.ICategoryEnumeration;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.AccessibleAdapter;
import org.eclipse.swt.accessibility.AccessibleEvent;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

/**
 * <p>Dialog to add a new {@link CFGComparableProperty} to
 * {@link org.eclipse.hyades.models.common.configuration.CFGConfigurableObject} or
 * edit an existing {@link CFGComparableProperty}.</p>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Paul Slauenwhite
 * @version April 2, 2010
 * @since	February 1, 2005
 */
public class ComparablePropertyInputDialog extends Dialog {

	private String title = null;
	private CCombo[] comboBoxes = null;
	private ICategory[] categories = null;
	private CFGComparableProperty property = null;
	
	public ComparablePropertyInputDialog(Shell shell, String title){
		
		super(shell);
		
		setShellStyle(getShellStyle() | SWT.RESIZE);
	
		this.title = title;
		this.comboBoxes = new CCombo[3];
		this.categories = Common_ConfigurationCategoryUtil.getInstance().getCategories();
	}
	
	public CFGComparableProperty getProperty()
	{
		return property;
	}
	
	public void setProperty(CFGComparableProperty property)
	{
		this.property = property;
	}
		
	protected Control createDialogArea(Composite parent){

		//Create reusable layouts:
		GridLayout zeroMarginsLayout = new GridLayout();
		zeroMarginsLayout.marginWidth = 0;
		zeroMarginsLayout.marginHeight = 0;

		GridData fixedWidthLayoutData = new GridData(GridData.FILL_BOTH);
		fixedWidthLayoutData.widthHint = 200;

		//Outer composite:		
		GridLayout compositeLayout = new GridLayout(3, false);
		compositeLayout.marginWidth = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_MARGIN);
		compositeLayout.marginTop = convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_MARGIN);
		compositeLayout.marginBottom = 0; //Required for uniform spacing between composites in the dialog.
		compositeLayout.horizontalSpacing = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_SPACING);
		compositeLayout.verticalSpacing = convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_SPACING);
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(compositeLayout);
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        
		//Property name composite:
		Composite propertyNameComposite = new Composite(composite, SWT.NONE);
		propertyNameComposite.setLayout(zeroMarginsLayout);
		propertyNameComposite.setLayoutData(fixedWidthLayoutData);

		//Property name label:
		Label propertyNameLabel = new Label(propertyNameComposite, SWT.WRAP);
		propertyNameLabel.setText(UiPluginResourceBundle.DLG_PROPERTY_NAME); 
		propertyNameLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));

		//Property name combo:
		String[] categoryDisplayNames = new String[categories.length];
		int selectionIndex = -1;
		String propertyName = null;
		
		if((property != null) && (property.getName() != null)){
			propertyName = property.getName();
		}
				
		for(int counter = 0; counter < categories.length; counter++){

			categoryDisplayNames[counter] = categories[counter].getDisplayName();
			
			if((propertyName != null) && (selectionIndex == -1) && (propertyName.equals(categories[counter].getName()))){
				selectionIndex = counter;
			}			
		}
		
		comboBoxes[0]= new CCombo(propertyNameComposite, SWT.BORDER);
		comboBoxes[0].setItems(categoryDisplayNames);
		
		if(selectionIndex != -1){
			comboBoxes[0].select(selectionIndex);
		}
		else if(propertyName != null){
			comboBoxes[0].setText(propertyName);
		}
		
		comboBoxes[0].setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
		comboBoxes[0].addSelectionListener(new SelectionAdapter() {
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {
				
				int index = comboBoxes[0].getSelectionIndex();
	            if (index < 0) return;

	            ICategory catg = categories[index];
	            String[] names = getEnumerationDisplayNames(catg);
	            List items = new LinkedList(Arrays.asList(names));
	            items.addAll(getSelector(catg));
	            comboBoxes[2].setItems((String[]) items.toArray(new String[items.size()]));
			}
		});
		comboBoxes[0].addModifyListener(new ModifyListener() {
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
			 */
			public void modifyText(ModifyEvent e) {
				validateInput();
			}
		});
		comboBoxes[0].getAccessible().addAccessibleListener(new AccessibleAdapter() {				
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.accessibility.AccessibleAdapter#getName(org.eclipse.swt.accessibility.AccessibleEvent)
			 */
			public void getName(AccessibleEvent e) {
				e.result = UiPluginResourceBundle.DLG_PROPERTY_NAME; 
			}
		});
		
		//Operator composite:
		Composite operatorComposite = new Composite(composite, SWT.NONE);
		operatorComposite.setLayout(zeroMarginsLayout);
		operatorComposite.setLayoutData(new GridData(GridData.FILL_VERTICAL));
		
		//Operator label:
		Label operatorLabel = new Label(operatorComposite, SWT.WRAP);
		operatorLabel.setText(UiPluginResourceBundle.DLG_OPERATOR); 
		operatorLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));

		//Operator combo:
		comboBoxes[1]= new CCombo(operatorComposite, (SWT.BORDER | SWT.READ_ONLY));
		comboBoxes[1].setItems(new String[]{"=", "!=", ">", "<", ">=", "<="}); //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$//$NON-NLS-5$ //$NON-NLS-6$
		
		if((property != null) && (property.getOperator() != null)){
			comboBoxes[1].setText(property.getOperator());
		}
		else{
			comboBoxes[1].select(0);
		}
		
		comboBoxes[1].setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
		comboBoxes[1].addModifyListener(new ModifyListener() {
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
			 */
			public void modifyText(ModifyEvent e) {
				validateInput();
			}
		});
		comboBoxes[1].getAccessible().addAccessibleListener(new AccessibleAdapter() {				
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.accessibility.AccessibleAdapter#getName(org.eclipse.swt.accessibility.AccessibleEvent)
			 */
			public void getName(AccessibleEvent e) {
				e.result = UiPluginResourceBundle.DLG_OPERATOR; 
			}
		});
		
		//Property value composite:
		Composite propertyValueComposite = new Composite(composite, SWT.NONE);
		propertyValueComposite.setLayout(zeroMarginsLayout);
		propertyValueComposite.setLayoutData(fixedWidthLayoutData);
		
		//Property value label:
		Label propertyValueLabel = new Label(propertyValueComposite, SWT.WRAP);
		propertyValueLabel.setText(UiPluginResourceBundle.DLG_PROPERTY_VALUE); 
		propertyValueLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
		
		//Property value combo:
		comboBoxes[2]= new CCombo(propertyValueComposite, SWT.BORDER);
		
		if((property != null) && (property.getValue() != null)){
			comboBoxes[2].setText(property.getValue());
		}

		comboBoxes[2].setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
        comboBoxes[2].addSelectionListener(new SelectionAdapter() {
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {
				
				String value = comboBoxes[2].getText();
	            String selectedPath = null;
	            if (DialogMessages.SELECT_FILE.equals(value)) {
	                FileDialog dialog = new FileDialog(getShell(), SWT.OPEN);
	                selectedPath = dialog.open();
	            } else if (DialogMessages.SELECT_DIR.equals(value)) {
	                DirectoryDialog dialog = new DirectoryDialog(getShell());
	                selectedPath = dialog.open();
	            }
	            if (selectedPath != null) {
	                selectedPath = selectedPath.trim();
	                if (selectedPath.length() > 0) {
	                    comboBoxes[2].add(selectedPath, 0);
	                    comboBoxes[2].setText(selectedPath);
	                }
	            }
			}
		});
		comboBoxes[2].addModifyListener(new ModifyListener() {
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
			 */
			public void modifyText(ModifyEvent e) {
				validateInput();
			}
		});
		comboBoxes[2].getAccessible().addAccessibleListener(new AccessibleAdapter() {				
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.accessibility.AccessibleAdapter#getName(org.eclipse.swt.accessibility.AccessibleEvent)
			 */
			public void getName(AccessibleEvent e) {
				e.result = UiPluginResourceBundle.DLG_PROPERTY_VALUE; 
			}
		});
		
		return composite;
	}
	
	 /* (non-Javadoc)
	 * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
	 */
	protected void configureShell(Shell shell) {
		
		super.configureShell(shell);
		
		if (title != null) {
			shell.setText(title);
		}	     
	}
	 
	/* (non-Javadoc)
	 * Method declared on Dialog.
	 */
	protected void buttonPressed(int buttonId) 
	{
		if (buttonId == IDialogConstants.OK_ID) {
			String name = comboBoxes[0].getText();
			ICategory category = null;
			int index = comboBoxes[0].indexOf(name);
			if(index > -1)
			{
				category = categories[index];
			    name = category.getName();
			}
			
			String operator = comboBoxes[1].getText();
			String value = comboBoxes[2].getText();
			if(property == null)
				property = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
			property.setName(name);
			if(category instanceof CFGCategory)
				property.setType((CFGCategory)category);
			if(operator != null && operator.length() > 0)
				property.setOperator(operator);
			if(value != null && value.length() > 0)
				property.setValue(value);
		} else {
			property= null;
		}
		super.buttonPressed(buttonId);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
	 */
	protected void createButtonsForButtonBar(Composite parent) {

		super.createButtonsForButtonBar(parent);
		
		getButton(IDialogConstants.OK_ID).setEnabled(false);
	}	
	
	private void validateInput() 
	{
		String name = comboBoxes[0].getText();
		String operator = comboBoxes[1].getText();
		String value = comboBoxes[2].getText();
		if(name == null || name.length() < 1 ||
		   operator == null || operator.length() < 1 ||
		   value == null || value.length() < 1)
			getButton(IDialogConstants.OK_ID).setEnabled(false);
		else
			getButton(IDialogConstants.OK_ID).setEnabled(true);
	}
	
	private String[] getEnumerationDisplayNames(ICategory category)
	{
		ICategoryEnumeration[] cat = Common_ConfigurationCategoryUtil.getInstance().getCategoryEnumeration(category);
		if(cat == null)
			return new String[0];
		ArrayList names = new ArrayList();
		for(int i = 0; i < cat.length; i++)
		{
			List values = cat[i].getIEnumerationValues();
			for(int j = 0; j < values.size(); j++)
			{
			 	names.add(values.get(j));	
			}
		}
		return (String[])names.toArray(new String[names.size()]);
	}
    
    /**
     * Some categories may have a file/directory selector to improve the usability.
     * 
     * @param category
     * @return the list of additionnal combo items that allows the user to select a file or/and a directory

     * @author pnedelec
     * @since 4.3
     */
    private List getSelector(ICategory category) {
        List names = new LinkedList();
        if (category instanceof HyadesClasspathCategory || category instanceof HyadesRootDirectroyCategory) {
            names.add(DialogMessages.SELECT_DIR);
        }
        if (category instanceof HyadesClasspathCategory) {
            names.add(DialogMessages.SELECT_FILE);
        }
        return names;
    }
}
