/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BaseEditorPart.java,v 1.16 2010/04/16 17:48:07 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor;

import java.io.File;
import java.io.InputStream;
import java.util.Collection;
import java.util.Iterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.common.facades.behavioral.impl.FacadeResourceImpl;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.util.EObjectEditorInput;
import org.eclipse.hyades.test.ui.util.RCPFileEditorInput;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.internal.editor.HyadesEditorPart;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.INavigationLocation;
import org.eclipse.ui.INavigationLocationProvider;
import org.eclipse.ui.IStorageEditorInput;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.editors.text.ILocationProvider;

/**
 * Abstract base class for test editors.
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @author  Jerome Bozier
 * @version April 16, 2010
 * @since   February 1, 2005
 */
public abstract class BaseEditorPart
extends HyadesEditorPart implements INavigationLocationProvider
{
	private ResourceSet resourceSet;
	private Class editorObjectClass;
	
	/**
	 * Constructor for BaseEditorPart
	 * @param editoObjectClass
	 */
	public BaseEditorPart(Class editoObjectClass)
	{
		super();
		this.editorObjectClass = editoObjectClass;
	}
	
	/**
	 * @see org.eclipse.ui.IWorkbenchPart#dispose()
	 */
	public void dispose()
	{
		if(resourceSet != null)
		{
			EMFUtil.removeAdapters(resourceSet.getAllContents());
			resourceSet.eAdapters().clear();

			Iterator iterator = getResourceSet().getResources().iterator();
			while (iterator.hasNext()) {
				Resource resource = (Resource) iterator.next();
				if ( resource instanceof FacadeResourceImpl )
					((FacadeResourceImpl)resource).cleanup();
			}
			
			resourceSet.getResources().clear();
			resourceSet = null;
		}
		
		super.dispose();
	}

	/**
	 * Returns this editor resource set.
	 * @return ResourceSet
	 */
	protected ResourceSet getResourceSet()
	{
		return resourceSet;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.editor.HyadesEditorPart#identifyEditorDescriptor(org.eclipse.ui.IEditorInput)
	 */
	protected IAssociationDescriptor identifyEditorDescriptor(IEditorInput input)
	throws PartInitException
	{		
        //- first init the resource set used by this editor
        initResourceSet();
		IAssociationDescriptor editorDescriptor = null;
		EObject eObject = null;
		if(input instanceof EObjectEditorInput){
			eObject = getEObjectContent(((EObjectEditorInput)(input)).getEObject());
		}
		else if(input instanceof RCPFileEditorInput)
			eObject = getFileContent(((RCPFileEditorInput)input).getIOFile());
		else if(input instanceof IFileEditorInput)
			eObject = getFileContent(((IFileEditorInput)input).getFile());
		else if (input instanceof ILocationProvider)
		{
			ILocationProvider javaFileEditorInput = (ILocationProvider)input;
			eObject = getFileContent(javaFileEditorInput.getPath(javaFileEditorInput).toFile());
		}
		else if(input instanceof IStorageEditorInput)
		{
			try {
                IStorage storage = ((IStorageEditorInput)input).getStorage();
                FacadeResourceImpl res = new FacadeResourceImpl();
                res.setUseZip(true);
                res.setEncoding("UTF-8"); //$NON-NLS-1$
                InputStream inputStream = storage.getContents();
                try {
                    res.load(inputStream, EMFUtil.RESOURCE_OPTIONS);
                } finally {
                    inputStream.close();
                }                
                EList content = res.getContents();
                if((content.size() == 1) && (editorObjectClass.isInstance(content.get(0)))) {
                    eObject = (EObject)content.get(0);
                    resourceSet.getResources().add(res);
                }
			}
			catch (Exception e) {
			    UiPlugin.logError(e);
            }
		}
		if(eObject == null) {
			return null; // if unrecognized, editor will close automatically
		}
		setEditorObject(eObject);
		editorDescriptor = identifyEditorDescriptor(eObject);

		return editorDescriptor;
	}

	/**
	 * Identifies the editor descriptor that should be used with the eObject.
	 * 
	 * <p>Implementors of this method may store the identified extension as a 
	 * preference for the next time this editor is opened.
	 * 
	 * <p><b>Important</b>: This method is supposed to return a valid 
	 * IAssociationDescriptor - ie. not null and related to an editor.
	 * 
	 * @param input
	 * @return IAssociationDescriptor
	 */
	protected abstract IAssociationDescriptor identifyEditorDescriptor(EObject eObject);

	/**
	 * Returns the "meaningful" content of a given IFile.
	 * @param file
	 * @return Object
	 */
	protected EObject getFileContent(IFile file)
	{
		EObject[] eObjects = EMFUtil.load(resourceSet, file);
		if((eObjects.length == 1) && (editorObjectClass.isInstance(eObjects[0])))
			return eObjects[0];
		return null;
	}

    /**
     * 
     */
    private void initResourceSet() {
        resourceSet = new ResourceSetImpl();
		AdapterImpl adapterImpl = new AdapterImpl()
		{
			public void notifyChanged(Notification msg)
			{
				switch(msg.getEventType())
				{
					case Notification.ADD:
						if(msg.getNewValue() instanceof Resource)
							((Resource)msg.getNewValue()).setTrackingModification(true);
						break;
					
					case Notification.ADD_MANY:
						Collection collection = (Collection)msg.getNewValue();
						for (Iterator i = collection.iterator(); i.hasNext();)
						{
							Object element = i.next();
							if(element instanceof Resource)
								((Resource)element).setTrackingModification(true);							
						}
						break;	
				}
			}
		};
		resourceSet.eAdapters().add(adapterImpl);
    }

    protected EObject getEObjectContent(EObject eObject){

    	if(editorObjectClass.isInstance(eObject)){

    		//Add the eObject's resource to the shared resource set:
    		resourceSet.getResources().add(eObject.eResource());

    		return eObject;
    	}

    	return null;
	}
    
	protected EObject getFileContent(File file)
	{
		EObject[] eObjects = EMFUtil.load(resourceSet, URI.createFileURI(file.getAbsolutePath()));
		if((eObjects.length == 1) && (editorObjectClass.isInstance(eObjects[0])))
			return eObjects[0];
		return null;
	}
	
	protected EObject getFileContent(IPath path)
	{
		EObject[] eObjects = EMFUtil.load(resourceSet, path.toOSString());
		if((eObjects.length == 1) && (editorObjectClass.isInstance(eObjects[0])))
			return eObjects[0];
		return null;
	}

	
	/**
	 * @see org.eclipse.jface.viewers.ISelectionProvider#setSelection(org.eclipse.jface.viewers.ISelection)
	 */
	public void setSelection(ISelection selection)
	{
		selection = adjustSelection(selection);
		if(!selection.isEmpty())
			super.setSelection(selection);
	}
	
	public ISelection adjustSelection(ISelection selection)
	{
		if(selection instanceof IStructuredSelection)
		{
			Object[] objects = ((IStructuredSelection)selection).toArray(); 
			for (int i = 0, maxi = objects.length; i < maxi; i++)
			{
				Object adjusted = null;
				if(objects[i] instanceof EObject)
				{
					adjusted = EMFUtil.getEObject(getResourceSet(), (EObject)objects[i], false);
				}
				else if(objects[i] instanceof Resource)
				{
					adjusted = getResourceSet().getResource(((Resource)objects[i]).getURI(), false);
				}
				
				if(adjusted == null)
					return StructuredSelection.EMPTY;
					
				objects[i] = adjusted;				
			}
			selection = new StructuredSelection(objects);
		}
		return selection;
	}
	/**
	 * @return Returns the editoObjectClass.
	 */
	protected Class getEditorObjectClass() {
		return editorObjectClass;
	}

    /** 
     * @see org.eclipse.ui.INavigationLocationProvider#createEmptyNavigationLocation()
     */
    public INavigationLocation createEmptyNavigationLocation() {
        if (getEditorExtension() instanceof INavigationLocationProvider) {
            return ((INavigationLocationProvider)getEditorExtension()).createEmptyNavigationLocation();
        }
        return null;
    }

    /** 
     * @see org.eclipse.ui.INavigationLocationProvider#createNavigationLocation()
     */
    public INavigationLocation createNavigationLocation() {
        if (getEditorExtension() instanceof INavigationLocationProvider) {
            return ((INavigationLocationProvider)getEditorExtension()).createNavigationLocation();
        }
        return null;
    }
}
