/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionEditorPart.java,v 1.5 2010/04/12 12:38:50 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor;

import java.util.Collection;
import java.util.Iterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.models.common.util.ResourceCache;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;

/**
 * <p>Concrete implementation of IHyadesEditorPart customized for
 * handling Hyades objects stored in resources with file extension equals to 
 * {@link org.eclipse.hyades.test.ui.TestUI#EXECUTION_FILE_EXTENSION}.</p>
 * 
 * <p>Starting with version 4.0, this implementation is using the shared version
 * of execution models held by the ResourceCache singleton. This allows to have
 * only one occurrence of an execution model in memory, while the execution model
 * is being populated by the model loaders. This also allows the execution editor
 * to be notified of changes (additions) to the execution model by EMF notifications,
 * rather than eclipse resource notifications.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul Slauenwhite
 * @author 	Jerome Bozier
 * @version April 12, 2010
 * @since   February 1, 2005
 */
public class ExecutionEditorPart extends BaseEditorPart {
	
	private ResourceSet sharedResourceSet;
	private Adapter thisAdapter;
	private URI cachedURI;
	
	/**
	 * Constructor for ExecutionEditorPart
	 */
	public ExecutionEditorPart()
	{
		super(TPFExecutionResult.class);
	}

	/**
	 * @see org.eclipse.hyades.ui.editor.IHyadesEditorPart#getFileExtension()
	 */
	public String getFileExtension()
	{
		return ICommonConstants.EXECUTION_FILE_EXTENSION;
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.EMFEditorPart#identifyEditorDescriptor(org.eclipse.emf.ecore.EObject)
	 */
	protected IAssociationDescriptor identifyEditorDescriptor(EObject eObject)
	{
		IAssociationMapping editorAssociationMapping = TestUIExtension.getExecutionMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_EDITOR_EXTENSIONS);
		return editorAssociationMapping.getDefaultAssociationDescriptor();
	}
	
	/**
	 * @see org.eclipse.ui.IWorkbenchPart#dispose()
	 */
	public void dispose()
	{
		if(sharedResourceSet != null)
		{
			EMFUtil.removeAdapters(sharedResourceSet.getAllContents(), thisAdapter);
			sharedResourceSet.eAdapters().remove(thisAdapter);
			sharedResourceSet = null;
			ResourceCache.getInstance().releaseSharedResource(cachedURI);
		}
		
		super.dispose();
	}

	/**
	 * Returns this editor resource set.
	 * @return ResourceSet
	 */
	protected ResourceSet getResourceSet()
	{
		return sharedResourceSet;
	}

	/**
	 * Returns the "meaningful" content of a given IFile.
	 * @param file
	 * @return Object
	 */
	protected EObject getFileContent(IFile file)
	{
		cachedURI = URI.createPlatformResourceURI(file.getFullPath().toString(), false);
		Resource resource = ResourceCache.getInstance().getSharedResource(cachedURI);
		if (resource.getContents().size() != 1
			|| !getEditorObjectClass().isInstance(resource.getContents().get(0))) {
			return null;
		}
		
		sharedResourceSet = resource.getResourceSet();
		thisAdapter = new AdapterImpl()
		{
			public void notifyChanged(Notification msg)
			{
				switch(msg.getEventType())
				{
					case Notification.ADD:
						if(msg.getNewValue() instanceof Resource)
							((Resource)msg.getNewValue()).setTrackingModification(true);
						break;
					
					case Notification.ADD_MANY:
						Collection collection = (Collection)msg.getNewValue();
						for (Iterator i = collection.iterator(); i.hasNext();)
						{
							Object element = i.next();
							if(element instanceof Resource)
								((Resource)element).setTrackingModification(true);							
						}
						break;	
				}
			}
		};
		sharedResourceSet.eAdapters().add(thisAdapter);
		
		return (EObject)resource.getContents().get(0);
	}

}
