/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ConfigurableObjectEditorExtension.java,v 1.13 2008/04/17 02:21:06 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.extension;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGConfigurableObject;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.editor.extension.BaseEditorExtension;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.ConfigurableObjectOverviewForm;
import org.eclipse.hyades.test.ui.internal.editor.form.IPropertyGroupForm;
import org.eclipse.hyades.test.ui.internal.editor.form.PropertyGroupForm;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;

/**
 * This is the base class for Hyades configuration editor extension.
 * <p/>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Paul E. Slauenwhite
 * @version April 16, 2008
 * @since   February 1, 2005
 */
public class ConfigurableObjectEditorExtension extends BaseEditorExtension implements IExtensibleEditorExtension
{	
	IConfigurationElement[] propertyGroupExtensions;
	private Hashtable extensionPagesMap = new Hashtable();

	protected static final int PAGE_OVERVIEW = 0;	
	private ConfigurableObjectOverviewForm overviewForm;	
	private WidgetFactory widgetFactory;
	
	public ConfigurableObjectEditorExtension()
	{
		super();	
		propertyGroupExtensions = collectPropertyGroupExtensions();		
	}

	/**
	 * Retrieves property group extensions that extends <i>"org.eclipse.hyades.test.ui.configurableObjectExtension"</i>
	 * @return
	 */
	protected IConfigurationElement[] collectPropertyGroupExtensions() 
	{
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + ".configurableObjectExtension"); //$NON-NLS-1$
		if( extPoint != null )
			return extPoint.getConfigurationElements();
		else
			return new IConfigurationElement[0];
	}
	
	/**
	 * Returns property group extensions that extends <i>"org.eclipse.hyades.test.ui.configurableObjectExtension"</i>
	 * @return
	 */
	public IConfigurationElement[] getPropertyGroupExtensions()
	{
		return this.propertyGroupExtensions;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		overviewForm.dispose();
		
		Enumeration elements = extensionPagesMap.elements();
		while(elements.hasMoreElements())
		{
			((EditorForm)elements.nextElement()).dispose();
		}		
		extensionPagesMap.clear();		
		
		if(widgetFactory != null)
			widgetFactory.dispose();
		
		super.dispose();
	}

	/**
	 * Returns the data object that is manipulated by this editor extension.
	 * @return EObject 
	 */
	public EObject getEditorObject()
	{
		return (EObject)getHyadesEditorPart().getEditorObject();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#createPages()
	 */
	public void createPages()
	{
		widgetFactory = new WidgetFactory();
		
		this.overviewForm = createOverviewPage(widgetFactory);		
		getHyadesEditorPart().addPage(overviewForm.createControl());
		getHyadesEditorPart().setPageText(PAGE_OVERVIEW, UiPluginResourceBundle.W_OVERVIEW); 
		overviewForm.updateTitle();
		
		createOtherStaticPages(widgetFactory);		
		
		// create forms for existing property groups.
		createPropertyGroupPages(widgetFactory);
	}
	
	/**
	 * Creates the overview page of this editor.
	 * @param widgetFactory
	 * @return
	 */
	protected ConfigurableObjectOverviewForm createOverviewPage(WidgetFactory widgetFactory)
	{
		return new ConfigurableObjectOverviewForm(this, widgetFactory);
	}

	/**
	 * Creates any pages of this editor that comes in front of the property group extension pages.
	 * Sub classes should override this method to add their own pages. This method does nothing by default.
	 * @param widgetFactory
	 */
	protected void createOtherStaticPages(WidgetFactory widgetFactory)
	{
	}
	
	/**
	 * Create pages for property group extensions that extends
	 * <i>"org.eclipse.hyades.test.ui.configurableObjectExtension"</i>.
	 * @param widgetFacotory
	 */
	protected void createPropertyGroupPages(WidgetFactory widgetFacotory)
	{
		EObject editorObject = getEditorObject();
		EList propertyGroups = null;
		if(editorObject instanceof CFGConfigurableObject)
		{
			propertyGroups = ((CFGConfigurableObject)editorObject).getPropertyGroups();
		}
		if(propertyGroups == null || propertyGroups.isEmpty())
			return;
			
		Iterator it = propertyGroups.iterator();
		CFGPropertyGroup propGroup = null;
		String propGroupID = null;
		boolean extensionFound = false;
		while( it.hasNext() )
		{		
			propGroup = (CFGPropertyGroup)it.next();
			if(propGroup.getProperties().isEmpty())
				continue;
				
			propGroupID = propGroup.getPropertyGroupID();			
			for(int i = 0; i < propertyGroupExtensions.length; i++)
			{
				if( propertyGroupExtensions[i].getAttribute("propertyGroupID").equals(propGroupID) ) //$NON-NLS-1$
				{
					addExtensionPage(propertyGroupExtensions[i]);
					extensionFound = true;
					break;
				}
			}
			
			// if there is an existing property group 
			// but no extension form registered for it
			// create a default form 
			if(extensionFound == false)
			{
				PropertyGroupForm defaultForm = new PropertyGroupForm();
				defaultForm.init(propGroup, this, widgetFactory);

				int index = getHyadesEditorPart().addPage(defaultForm.createControl());
				getHyadesEditorPart().setPageText(index, propGroup.getName());					
				extensionPagesMap.put(new Integer(index), defaultForm);
				overviewForm.addSection(defaultForm, index); 
			}
		}
	}
	
	protected ConfigurableObjectOverviewForm getOverviewForm()
	{
		return this.overviewForm;
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IRefreshable#refreshContent(java.lang.Object)
	 */
	public void refreshContent(Object data)
	{
		overviewForm.load();
		overviewForm.updateTitle();
	}

	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#getSelection()
	 */
	public IStructuredSelection getSelection()
	{
		int index = getHyadesEditorPart().getActivePage();
		if(index == PAGE_OVERVIEW) // overview page, just return the data object
			return new StructuredSelection(getEditorObject());
		else if(extensionPagesMap.containsKey(new Integer(index)))
			return ((IPropertyGroupForm)extensionPagesMap.get(new Integer(index))).getSelection();
		
		return StructuredSelection.EMPTY;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.editor.EditorExtension#setSelection(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void setSelection(IStructuredSelection structuredSelection){

		Object object = structuredSelection.getFirstElement();

		if(object instanceof CFGComparableProperty){
			
			EObject selectionPropertyEContainer = ((CFGComparableProperty)(object)).eContainer();
			
			if(selectionPropertyEContainer instanceof CFGPropertyGroup){
			
				String selectionPropertyGroupId = ((CFGPropertyGroup)(selectionPropertyEContainer)).getPropertyGroupID();
				Iterator pageIndexes = extensionPagesMap.keySet().iterator();
				
				while(pageIndexes.hasNext()){
					
					int pageIndex = ((Integer)(pageIndexes.next())).intValue();
					IPropertyGroupForm propertyGroupForm = ((IPropertyGroupForm)(extensionPagesMap.get(new Integer(pageIndex))));
					
					if(propertyGroupForm.getPropertyGroup().getPropertyGroupID().equals(selectionPropertyGroupId)){
						
						if(pageIndex < getHyadesEditorPart().getPageCount()){
			
							getHyadesEditorPart().setActivePage(pageIndex);
							propertyGroupForm.selectReveal(structuredSelection);
						}
						
						break;
					}
				}
			}
		}		
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#pageActivated(int)
	 */
	public boolean pageActivated(int index)
	{
		if(index == PAGE_OVERVIEW)
			return overviewForm.activated();
		else if(extensionPagesMap.containsKey(new Integer(index)))
			return ((EditorForm)extensionPagesMap.get(new Integer(index))).activated();
		
		return false;
	}
		
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.editor.IExtensibleEditorExtension#addExtensionPage(org.eclipse.core.runtime.IConfigurationElement)
	 */
	public int addExtensionPage(IConfigurationElement extensionElement) 
	{
		// check if this page is already activated.
		String propertyGroupID = extensionElement.getAttribute("propertyGroupID"); //$NON-NLS-1$
		Enumeration keys = extensionPagesMap.keys();
		Integer key;
		IPropertyGroupForm page = null;
		while( keys.hasMoreElements() )
		{
			key = (Integer)keys.nextElement();
			page = (IPropertyGroupForm)extensionPagesMap.get(key);
			if( page.getPropertyGroup().getPropertyGroupID().equals( propertyGroupID ) )
				return key.intValue();
		}
			 
		IPropertyGroupForm formExt = null;
		try
		{
			formExt = (IPropertyGroupForm)extensionElement.createExecutableExtension("class"); //$NON-NLS-1$
		}
		catch(Throwable e)
		{
			e.printStackTrace();
			// TODO properly handle exception when failed to instantiate extension.
		}
		
		if(formExt instanceof IPropertyGroupForm)
		    ((IPropertyGroupForm)formExt).init(getCFGPropertyGroup(extensionElement), this, widgetFactory);
		formExt.setExtension(extensionElement);
		
		int index = getHyadesEditorPart().addPage(formExt.createControl());
		getHyadesEditorPart().setPageText(index, extensionElement.getAttribute("name"));					
		extensionPagesMap.put(new Integer(index), formExt);
		overviewForm.addSection(formExt, index); 
		
		return index;
	}
	
	private CFGPropertyGroup getCFGPropertyGroup(IConfigurationElement extensionElement)
	{
		EObject editorObject = getEditorObject();
		EList propertyGroups = null; 
		if(editorObject instanceof CFGConfigurableObject)
			propertyGroups = ((CFGConfigurableObject)editorObject).getPropertyGroups();
		if(propertyGroups == null)
			return null;
			
		CFGPropertyGroup propertyGroup = null;
		String propertyGroupID = extensionElement.getAttribute("propertyGroupID"); //$NON-NLS-1$
		for( Iterator i = propertyGroups.iterator(); i.hasNext(); )
		{			
			propertyGroup = (CFGPropertyGroup)i.next();
			if(propertyGroupID.equals(propertyGroup.getPropertyGroupID()))
				break;
			else
				propertyGroup = null;			
		}

		if( propertyGroup == null ) 
		{
			String name = extensionElement.getAttribute("name"); //$NON-NLS-1$
			String description = extensionElement.getAttribute("description"); //$NON-NLS-1$
			propertyGroup = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
			propertyGroup.setPropertyGroupID(propertyGroupID);
			propertyGroup.setName(name);
			propertyGroup.setDescription(description);
			((CFGConfigurableObject)editorObject).getPropertyGroups().add(propertyGroup);
		}
	
		return propertyGroup;
	}

}


