/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolEditorExtension.java,v 1.25 2010/03/24 16:18:41 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.extension;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolCell;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.edit.datapool.IDatapoolListener;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.datapool.util.DatapoolEncryptManager;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.control.DatapoolActionHandlerListener;
import org.eclipse.hyades.test.ui.datapool.internal.control.DatapoolTable;
import org.eclipse.hyades.test.ui.datapool.internal.control.DatapoolTableUtil;
import org.eclipse.hyades.test.ui.datapool.internal.interfaces.IDatapoolPartExtended;
import org.eclipse.hyades.test.ui.editor.extension.BaseEditorExtension;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.DataTableForm;
import org.eclipse.hyades.test.ui.internal.editor.form.DatapoolForm;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.editor.IEditActionsExtension;
import org.eclipse.hyades.ui.editor.IHyadesEditorPart;
import org.eclipse.jface.text.IFindReplaceTarget;
import org.eclipse.jface.text.IFindReplaceTargetExtension;
import org.eclipse.jface.text.IFindReplaceTargetExtension3;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.Region;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.custom.TableCursor;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.tptp.platform.common.ui.internal.util.UIUtil;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;

/**
 * <p>Datapool editor extension.</p>
 * 
 * 
 * @author  Peter Sun
 * @author  Paul E. Slauenwhite
 * @version March 24, 2010
 * @since   February 1, 2005
 */
public class DatapoolEditorExtension extends BaseEditorExtension implements IDatapoolPartExtended, IEditActionsExtension {
    
    public static final String EXTENSION_ID = "org.eclipse.hyades.test.ui.editor.extension.Datapool"; //$NON-NLS-1$
	
    private DatapoolForm datapoolForm;
    private ArrayList equivalenceClassPages = new ArrayList();
    private int activePageIndex = -1;
    private IFindReplaceTarget findReplaceTarget;
    private IDatapoolListener datapoolListener = new DatapoolListener();
    
    private static final int PAGE_OVERVIEW = 0;

    protected class DatapoolListener implements IDatapoolListener {

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#cellChanged(org.eclipse.hyades.edit.datapool.IDatapool, int, int, int)
         */
        public void cellChanged(IDatapool datapool, int equivalenceClassIndex, int recordIndex, int variableIndex) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#equivalenceClassAdded(org.eclipse.hyades.edit.datapool.IDatapool, int)
         */
        public void equivalenceClassAdded(IDatapool datapool, int newEquivalenceClassIndex) {
            addEquivalenceClassPage(newEquivalenceClassIndex);
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#equivalenceClassChanged(org.eclipse.hyades.edit.datapool.IDatapool, int, java.lang.String)
         */
        public void equivalenceClassChanged(IDatapool datapool, int equivalenceClassIndex, String oldName) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#equivalenceClassChanged(org.eclipse.hyades.edit.datapool.IDatapool, int)
         */
        public void equivalenceClassChanged(IDatapool datapool, int equivalenceClassIndex) {
        	getHyadesEditorPart().setPageText(getEquivalenceClassPageIndex(equivalenceClassIndex), ((IDatapoolEquivalenceClass)(datapool.getEquivalenceClass(equivalenceClassIndex))).getName());
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#equivalenceClassMoved(org.eclipse.hyades.edit.datapool.IDatapool, int, int)
         */
        public void equivalenceClassMoved(IDatapool datapool, int sourceEquivalenceClassIndex, int targetEquivalenceClassIndex) {
            moveEquivalenceClassPage(sourceEquivalenceClassIndex, targetEquivalenceClassIndex);
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#equivalenceClassRemoved(org.eclipse.hyades.edit.datapool.IDatapool, int)
         */
        public void equivalenceClassRemoved(IDatapool datapool, int equivalenceClassIndex) {
            removeEquivalenceClassPage(equivalenceClassIndex);
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#equivalenceClassReordered(org.eclipse.hyades.edit.datapool.IDatapool, int)
         */
        public void equivalenceClassReordered(IDatapool datapool, int equivalenceClassIndex) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#recordAdded(org.eclipse.hyades.edit.datapool.IDatapool, int, int)
         */
        public void recordAdded(IDatapool datapool, int EquivClassIndex, int newRecordIndex) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#recordChanged(org.eclipse.hyades.edit.datapool.IDatapool, int, int)
         */
        public void recordChanged(IDatapool datapool, int EquivClassIndex, int recordIndex) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#recordMoved(org.eclipse.hyades.edit.datapool.IDatapool, int, int, int)
         */
        public void recordMoved(IDatapool datapool, int EquivClassIndex, int sourceRecordIndex, int targetRecordIndex) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#recordRemoved(org.eclipse.hyades.edit.datapool.IDatapool, int, int)
         */
        public void recordRemoved(IDatapool datapool, int EquivClassIndex, int recordIndex) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#save(org.eclipse.hyades.edit.datapool.IDatapool)
         */
        public void save(IDatapool datapool) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#variableAdded(org.eclipse.hyades.edit.datapool.IDatapool, int)
         */
        public void variableAdded(IDatapool datapool, int newVariableIndex) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#variableChanged(org.eclipse.hyades.edit.datapool.IDatapool, int, java.lang.String)
         */
        public void variableChanged(IDatapool datapool, int variableIndex, String oldName) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#variableChanged(org.eclipse.hyades.edit.datapool.IDatapool, int)
         */
        public void variableChanged(IDatapool datapool, int variableIndex) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#variableMoved(org.eclipse.hyades.edit.datapool.IDatapool, int, int)
         */
        public void variableMoved(IDatapool datapool, int sourceVariableIndex, int targetVariableIndex) {
            //No-operation.
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.edit.datapool.IDatapoolListener#variableRemoved(org.eclipse.hyades.edit.datapool.IDatapool, int)
         */
        public void variableRemoved(IDatapool datapool, int variableIndex) {
            //No-operation.
        }
    }

    /**
     * <p>Class implementing {@link IFindReplaceTarget}, {@link IFindReplaceTargetExtension}, and 
     * {@link IFindReplaceTargetExtension3} to provide Find/Replace functionality.</p>
     * 
     * <p>Note: A row is equated to a line and a cell (smallest unit of selection) is equated to a character.</p>
     * 
     * 
     * @author  Bianca Xue Jiang
     * @author  Paul Slauenwhite
     * @version March 24, 2010
     * @since   February 1, 2005
     */
    class FindReplaceTarget implements IFindReplaceTarget, IFindReplaceTargetExtension, IFindReplaceTargetExtension3 {
        
    	private int previousActivePage = -1;
        private DatapoolTable datapoolTable = null;
        private Table table = null;
        private Pattern pattern = null;
        private Matcher matcher = null;
    	private boolean isGlobalScope = true;
    	
        //Copied from org.eclipse.jface.text.FindReplaceDocumentAdapter (see Bugzilla 275911).
    	private int fRetainCaseMode = RC_MIXED;
    	
        //Copied from org.eclipse.jface.text.FindReplaceDocumentAdapter (see Bugzilla 275911).
    	private static final int RC_MIXED = 0;
    	private static final int RC_UPPER = 1;
    	private static final int RC_LOWER = 2;
    	private static final int RC_FIRSTUPPER = 3;

        public FindReplaceTarget(int activePage) {
            updatePage(activePage);
        }

        public void updatePage(int activePage) {
            
        	if (previousActivePage != activePage) {
                
        		previousActivePage = activePage;
                
                datapoolTable = ((DataTableForm)(equivalenceClassPages.get(getEquivalenceClassIndexFromPageIndex(activePage)))).getDatapoolTable();
                
                table = datapoolTable.getViewer().getTable();
            }
        }

		/* (non-Javadoc)
    	 * @see org.eclipse.jface.text.IFindReplaceTargetExtension#beginSession()
    	 */
    	public void beginSession() {
    		isGlobalScope = true;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.text.IFindReplaceTargetExtension#endSession()
		 */
		public void endSession() {
			isGlobalScope = true;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.text.IFindReplaceTargetExtension#getScope()
		 */
		public IRegion getScope() {	

			if(!isGlobalScope){
				
				//Since this Find/Replace target must support multiple selected rows that are not contiguous, 
				//we will return all of the rows/lines and handle searching the actual selected rows in the 
				//findAndSelect() method:
				return (new Region(0, table.getItemCount()));   
			}
			
			return null;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.text.IFindReplaceTargetExtension#setScope(org.eclipse.jface.text.IRegion)
		 */
		public void setScope(IRegion scope) {
			isGlobalScope = (scope == null);
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.text.IFindReplaceTargetExtension#getLineSelection()
		 */
		public Point getLineSelection() {

			//Since this Find/Replace target must support multiple selected rows that are not contiguous, 
			//we will return all of the rows/lines to cause the setScope() method to be invoked and handle 
			//searching the actual selected rows in the findAndSelect() method:
            return (new Point(0, table.getItemCount()));            
		}

		/* (non-Javadoc)
         * @see org.eclipse.jface.text.IFindReplaceTarget#getSelection()
         */
        public Point getSelection() {

        	//Calculate the offset (zero-based row by zero-based column count) of the current selected cell:
            Point tableCursorPosition = datapoolTable.getCursorPosition();
            
            //Note: The length is 1 since a cell is the smallest unit of selection.
            return (new Point(((tableCursorPosition.x * table.getColumnCount()) + tableCursorPosition.y + 1), 1));
        }

        /* (non-Javadoc)
         * @see org.eclipse.jface.text.IFindReplaceTarget#getSelectionText()
         */
        public String getSelectionText() {
            
        	TableCursor tableCursor = datapoolTable.getTableCursor();
            
        	if ((tableCursor != null) && (!tableCursor.isDisposed())){
                return (tableCursor.getRow().getText(tableCursor.getColumn()));
            }
            
        	return null;
        }

		/* (non-Javadoc)
		 * @see org.eclipse.jface.text.IFindReplaceTargetExtension#setSelection(int, int)
		 */
		public void setSelection(int offset, int length) {
			//No-operation.			
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.text.IFindReplaceTargetExtension#setScopeHighlightColor(org.eclipse.swt.graphics.Color)
		 */
		public void setScopeHighlightColor(Color color) {
			//No-operation.			
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.text.IFindReplaceTargetExtension#setReplaceAllMode(boolean)
		 */
		public void setReplaceAllMode(boolean replaceAll) {
			//No-operation.			
		}

        /* (non-Javadoc)
         * @see org.eclipse.jface.text.IFindReplaceTarget#canPerformFind()
         */
        public boolean canPerformFind() {
            return ((datapoolTable.getEquivalenceClass() != null) && (datapoolTable.getEquivalenceClass().getRecordCount() > 0) && (getDatapool().getVariableCount() > 0));
        }
        
        /* (non-Javadoc)
         * @see org.eclipse.jface.text.IFindReplaceTarget#isEditable()
         */
        public boolean isEditable() {
            return (!getHyadesEditorPart().isReadOnly());
        }

        /* (non-Javadoc)
         * @see org.eclipse.jface.text.IFindReplaceTarget#findAndSelect(int, java.lang.String, boolean, boolean, boolean)
         */
        public int findAndSelect(int widgetOffset, String findString, boolean searchForward, boolean caseSensitive, boolean wholeWord) {
            return findAndSelect(widgetOffset, findString, searchForward, caseSensitive, wholeWord, false);
        }
        
        /* (non-Javadoc)
         * @see org.eclipse.jface.text.IFindReplaceTargetExtension3#findAndSelect(int, java.lang.String, boolean, boolean, boolean, boolean)
         */
        public int findAndSelect(int offset, String findString, boolean searchForward, boolean caseSensitive, boolean wholeWord, boolean regExSearch) {

            if ((findString != null) && (findString.length() > 0)){
	
	            int patternFlags = 0;
	
	            if(regExSearch){
	            	patternFlags |= Pattern.MULTILINE;
	            }
	
	            if(!caseSensitive){
	            	
	            	patternFlags |= Pattern.CASE_INSENSITIVE;
	            	patternFlags |= Pattern.UNICODE_CASE;
	            }
	
	            //Search using a word boundary:
	            if(wholeWord){
	            	findString = ("\\b" + findString + "\\b"); //$NON-NLS-1$ //$NON-NLS-2$
	            }
	            else if(!regExSearch){
	            	findString = asRegPattern(findString);
	            }
	
	            if((pattern == null) || (!pattern.pattern().equals(findString)) || (pattern.flags() != patternFlags)){
	                pattern = Pattern.compile(findString, patternFlags);
	            }
	
	            if(searchForward){

	            	//Case 1: Offset is -1 (wrap search) so the starting row and column defaults to the first column in the first row.
	            	int startingRow = 0;
	 	            int startingColumn = 0;
	 	            	         
            		//Case 2: Offset is >0 so the starting row and column defaults to the current column in the current row.
	            	if (offset > 0){

	            		startingRow = ((int)((offset - 1) / table.getColumnCount()));
	            		startingColumn = ((offset - 1) - ((startingRow * table.getColumnCount())));	    
	            	}

	            	//Iterate forward over the rows starting at the starting row:
	            	for (int row = startingRow; row < table.getItemCount(); row++) {

	            		 if((isGlobalScope) || (table.isSelected(row))){

	            			 //Iterate forward over the columns starting at the starting column (starting row only) or first column:
	            			 for (int column = ((row == startingRow) ? startingColumn: 0); column < table.getColumnCount(); column++) {

	            				 if (matchTableCellText(row, column)){
	            					 return ((row * table.getColumnCount()) + column + 1);
	            				 }
	            			 }
	            		 }
	                }
	            } 
	            
	            //Backward direction:
	            else{
	                
	            	//Case 1: Offset is -1 (wrap search) so the starting row and column defaults to the last column in the last row.
	            	int startingRow = (table.getItemCount() - 1);
	 	            int startingColumn = (table.getColumnCount() - 1);
	 	            	      
            		//Case 2: Offset is >0 so the starting row and column defaults to the current column in the current row.
	            	if (offset > 0){

	            		startingRow = ((int)((offset - 1) / table.getColumnCount()));
	            		startingColumn = ((offset - 1) - ((startingRow * table.getColumnCount())));	     
	            	}
	            		            	
	            	//Iterate backward over the rows starting at the starting row:
	                for(int row = startingRow; row >= 0; row--) {
	                    
	                	 if((isGlobalScope) || (table.isSelected(row))){

	                		 //Iterate backward over the columns starting at the starting column (starting row only) or last column:
	                		 for(int column = ((row == startingRow) ? startingColumn: (table.getColumnCount() - 1)); column >= 0; column--){ 

	                			 if (matchTableCellText(row, column)){
	                				 return ((row * table.getColumnCount()) + column + 1);
	                			 }
	                		 }
	                	 }
	                }
	            }
            }
            
            return -1;
        }
        
        /* (non-Javadoc)
         * @see org.eclipse.jface.text.IFindReplaceTarget#replaceSelection(java.lang.String)
         */
        public void replaceSelection(String text) {
            replaceSelection(text, false);
        }

        /* (non-Javadoc)
         * @see org.eclipse.jface.text.IFindReplaceTargetExtension3#replaceSelection(java.lang.String, boolean)
         */
        public void replaceSelection(String text, boolean regExReplace) {

            String replaceWithText = text;
        	
        	if(regExReplace){
        		
				try{
					replaceWithText = interpretReplaceEscapes(text, matcher.group());
				} 
				catch (Exception e) {
					UiPlugin.logError(e);
				}
        	}

            String replacedText = null;

            //Note: Since a datapool cell is the smallest unit of selection, the replace operation
            //will replace all occurrences within the cell.
            if (matcher != null){
            	replacedText = matcher.replaceAll(replaceWithText);
            }
            else{
            	replacedText = replaceWithText;
            }

            TableCursor tableCursor = datapoolTable.getTableCursor();
            TableItem selectedTableItem = tableCursor.getRow();
            
            if((selectedTableItem != null) && (!selectedTableItem.isDisposed())){
	            	
	            IDatapoolCell selectedCell = datapoolTable.getSelectedCell();
	            
	            if (selectedCell != null){
	            	
	            	selectedCell.setCellValue(replacedText);
		            
		            selectedTableItem.setText(tableCursor.getColumn(), replacedText);
		            
		            tableCursor.redraw();
		            
		            datapoolTable.getIDatapoolPart().markDirty();
	            }
            }
        }
        
        /**
         * <p>Matches the table cell text at a given row and column.</p>
         * 
         * <p>Note: The first column (index 0) and columns containing encrypted cells are not matched.</p>
         * 
         * @param row The zero-based row index.
         * @param column The zero-based column index.
         * @return True if the table cell text at a given row and column matches, otherwise false.
         */
        private boolean matchTableCellText(int row, int column) {

        	//Skip the first/row number column:
        	if(column != 0){
        		
	        	TableItem tableItem = table.getItem(row);
				
				if((tableItem != null) && (!tableItem.isDisposed())){
					
					IDatapoolCell[] cells = ((IDatapoolCell[])(tableItem.getData(DatapoolTableUtil.TAG_DATA)));
					
					//Only match decrypted cells:
					if((cells != null) && (column <= cells.length) && (cells[column - 1] != null) && (!DatapoolEncryptManager.isVariableEncrypted(cells[column - 1].getCellVariable()))){
			        	
						matcher = pattern.matcher(tableItem.getText(column));
			            
			        	if (matcher.find()) {
			                
			            	//Select the table row if it is not selected:
			                if (!Arrays.asList((table.getSelection())).contains(tableItem)){
			                	table.setSelection(row);
			                }
	
			                TableCursor cursor = datapoolTable.getTableCursor();
			                cursor.setSelection(row, column);
			                cursor.redraw();
	
			                return true;
			            }
					}
				}
        	}
        	
            return false;
        }
        
        /**
    	 * Converts a non-regex string to a pattern
    	 * that can be used with the regex search engine.
    	 *
    	 * @param string the non-regex pattern
    	 * @return the string converted to a regex pattern
    	 */
        //Copied from org.eclipse.jface.text.FindReplaceDocumentAdapter (see Bugzilla 275911).
        private String asRegPattern(String string) {
    		StringBuffer out= new StringBuffer(string.length());
    		boolean quoting= false;
    		
    		for (int i= 0, length= string.length(); i < length; i++) {
    			char ch= string.charAt(i);
    			if (ch == '\\') {
    				if (quoting) {
    					out.append("\\E"); //$NON-NLS-1$
    					quoting= false;
    				}
    				out.append("\\\\"); //$NON-NLS-1$
    				continue;
    			}
    			if (!quoting) {
    				out.append("\\Q"); //$NON-NLS-1$
    				quoting= true;
    			}
    			out.append(ch);
    		}
    		if (quoting)
    			out.append("\\E"); //$NON-NLS-1$

    		return out.toString();
    	}
        
        /**
    	 * Interprets escaped characters in the given replace pattern.
    	 *
    	 * @param replaceText the replace pattern
    	 * @param foundText the found pattern to be replaced
    	 * @return a replace pattern with escaped characters substituted by the respective characters
    	 */
        //Copied from org.eclipse.jface.text.FindReplaceDocumentAdapter (see Bugzilla 275911).
    	private String interpretReplaceEscapes(String replaceText, String foundText) {
    		int length= replaceText.length();
    		boolean inEscape= false;
    		StringBuffer buf= new StringBuffer(length);

    		/* every string we did not check looks mixed at first
    		 * so initialize retain case mode with RC_MIXED
    		 */
    		fRetainCaseMode= RC_MIXED;

    		for (int i= 0; i < length; i++) {
    			final char ch= replaceText.charAt(i);
    			if (inEscape) {
    				i= interpretReplaceEscape(ch, i, buf, replaceText, foundText);
    				inEscape= false;

    			} else if (ch == '\\') {
    				inEscape= true;

    			} else if (ch == '$') {
    				buf.append(ch);

    				/*
    				 * Feature in java.util.regex.Matcher#replaceFirst(String):
    				 * $00, $000, etc. are interpreted as $0 and
    				 * $01, $001, etc. are interpreted as $1, etc. .
    				 * If we support \0 as replacement pattern for capturing group 0,
    				 * it would not be possible any more to write a replacement pattern
    				 * that appends 0 to a capturing group (like $0\0).
    				 * The fix is to interpret \00 and $00 as $0\0, and
    				 * \01 and $01 as $0\1, etc.
    				 */
    				if (i + 2 < length) {
    					char ch1= replaceText.charAt(i + 1);
    					char ch2= replaceText.charAt(i + 2);
    					if (ch1 == '0' && '0' <= ch2 && ch2 <= '9') {
    						buf.append("0\\"); //$NON-NLS-1$
    						i++; // consume the 0
    					}
    				}
    			} else {
    				interpretRetainCase(buf, ch);
    			}
    		}

    		if (inEscape) {
    			// '\' as last character is invalid, but we still add it to get an error message
    			buf.append('\\');
    		}
    		return buf.toString();
    	}
    	
    	/**
    	 * Interprets the escaped character <code>ch</code> at offset <code>i</code>
    	 * of the <code>replaceText</code> and appends the interpretation to <code>buf</code>.
    	 *
    	 * @param ch the escaped character
    	 * @param i the offset
    	 * @param buf the output buffer
    	 * @param replaceText the original replace pattern
    	 * @param foundText the found pattern to be replaced
    	 * @return the new offset
    	 */
    	//Copied from org.eclipse.jface.text.FindReplaceDocumentAdapter (see Bugzilla 275911).
    	private int interpretReplaceEscape(final char ch, int i, StringBuffer buf, String replaceText, String foundText) {
    		int length= replaceText.length();
    		switch (ch) {
    			case 'r':
    				buf.append('\r');
    				break;
    			case 'n':
    				buf.append('\n');
    				break;
    			case 't':
    				buf.append('\t');
    				break;
    			case 'f':
    				buf.append('\f');
    				break;
    			case 'a':
    				buf.append('\u0007');
    				break;
    			case 'e':
    				buf.append('\u001B');
    				break;
    			case 'R': //see http://www.unicode.org/unicode/reports/tr18/#Line_Boundaries
    				buf.append(System.getProperty("line.separator", "\n")); //$NON-NLS-1$ //$NON-NLS-2$
    				break;
    			/*
    			 * \0 for octal is not supported in replace string, since it
    			 * would conflict with capturing group \0, etc.
    			 */
    			case '0':
    				buf.append('$').append(ch);
    				/*
    				 * See explanation in "Feature in java.util.regex.Matcher#replaceFirst(String)"
    				 * in interpretReplaceEscape(String) above.
    				 */
    				if (i + 1 < length) {
    					char ch1= replaceText.charAt(i + 1);
    					if ('0' <= ch1 && ch1 <= '9') {
    						buf.append('\\');
    					}
    				}
    				break;

    			case '1':
    			case '2':
    			case '3':
    			case '4':
    			case '5':
    			case '6':
    			case '7':
    			case '8':
    			case '9':
    				buf.append('$').append(ch);
    				break;

    			case 'c':
    				if (i + 1 < length) {
    					char ch1= replaceText.charAt(i + 1);
    					interpretRetainCase(buf, (char)(ch1 ^ 64));
    					i++;
    				} else {
    					throw new PatternSyntaxException("Illegal control escape sequence \\c", replaceText, i); //$NON-NLS-1$
    				}
    				break;

    			case 'x':
    				if (i + 2 < length) {
    					int parsedInt;
    					try {
    						parsedInt= Integer.parseInt(replaceText.substring(i + 1, i + 3), 16);
    						if (parsedInt < 0)
    							throw new NumberFormatException();
    					} catch (NumberFormatException e) {
    						throw new PatternSyntaxException("Illegal hexadecimal escape sequence " + replaceText.substring(i - 1, i + 3), replaceText, i); //$NON-NLS-1$
    					}
    					interpretRetainCase(buf, (char) parsedInt);
    					i+= 2;
    				} else {
    					throw new PatternSyntaxException("Illegal hexadecimal escape sequence " + replaceText.substring(i - 1, length), replaceText, i); //$NON-NLS-1$
    				}
    				break;

    			case 'u':
    				if (i + 4 < length) {
    					int parsedInt;
    					try {
    						parsedInt= Integer.parseInt(replaceText.substring(i + 1, i + 5), 16);
    						if (parsedInt < 0)
    							throw new NumberFormatException();
    					} catch (NumberFormatException e) {
    						throw new PatternSyntaxException("Illegal Unicode escape sequence " + replaceText.substring(i - 1, i + 5), replaceText, i); //$NON-NLS-1$
    					}
    					interpretRetainCase(buf, (char) parsedInt);
    					i+= 4;
    				} else {
    					throw new PatternSyntaxException("Illegal Unicode escape sequence " + replaceText.substring(i - 1, length), replaceText, i); //$NON-NLS-1$
    				}
    				break;

    			case 'C':
    				if(foundText.toUpperCase().equals(foundText)) // is whole match upper-case?
    					fRetainCaseMode= RC_UPPER;
    				else if (foundText.toLowerCase().equals(foundText)) // is whole match lower-case?
    					fRetainCaseMode= RC_LOWER;
    				else if(Character.isUpperCase(foundText.charAt(0))) // is first character upper-case?
    					fRetainCaseMode= RC_FIRSTUPPER;
    				else
    					fRetainCaseMode= RC_MIXED;
    				break;

    			default:
    				// unknown escape k: append uninterpreted \k
    				buf.append('\\').append(ch);
    				break;
    		}
    		return i;
    	}
    	
    	/**
    	 * Interprets current Retain Case mode (all upper-case,all lower-case,capitalized or mixed)
    	 * and appends the character <code>ch</code> to <code>buf</code> after processing.
    	 *
    	 * @param buf the output buffer
    	 * @param ch the character to process
    	 */
    	//Copied from org.eclipse.jface.text.FindReplaceDocumentAdapter (see Bugzilla 275911).
    	private void interpretRetainCase(StringBuffer buf, char ch) {
    		if (fRetainCaseMode == RC_UPPER)
    			buf.append(Character.toUpperCase(ch));
    		else if (fRetainCaseMode == RC_LOWER)
    			buf.append(Character.toLowerCase(ch));
    		else if (fRetainCaseMode == RC_FIRSTUPPER) {
    			buf.append(Character.toUpperCase(ch));
    			fRetainCaseMode= RC_MIXED;
    		} else
    			buf.append(ch);
    	}
    }
    
    /**
     * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
     */
    public void dispose() {
        datapoolForm.getWidgetFactory().dispose();
        datapoolForm.dispose();

        Object page;
        for (Iterator it = equivalenceClassPages.iterator(); it.hasNext();) {
            page = it.next();
            if (page != null) ((DataTableForm) page).dispose();
        }

        equivalenceClassPages.clear();

        super.dispose();
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.editor.IEditorExtension#createPages()
     */    
    public void createPages() {
    	IHyadesEditorPart hyadesEditorPart = getHyadesEditorPart();
        WidgetFactory widgetFactory = new WidgetFactory();

        datapoolForm = new DatapoolForm(this, widgetFactory);
        hyadesEditorPart.addPage(datapoolForm.createControl());
        hyadesEditorPart.setPageText(PAGE_OVERVIEW, UiPluginResourceBundle.W_OVERVIEW); 
        datapoolForm.updateTitle();

        IDatapool dp = getDatapool(); 
        for (int i = 0; i < dp.getEquivalenceClassCount(); i++) {
            addEquivalenceClassPage(i);
        }

        if (dp.getEquivalenceClassCount() > 0) {
            // bugzilla 74864 difficult discovering data table -- show data table of default EC first. 
            int defaultIndex = dp.getDefaultEquivalenceClassIndex();
            defaultIndex = (defaultIndex == -1) ? 0 : defaultIndex;
            hyadesEditorPart.setActivePage(getEquivalenceClassPageIndex(defaultIndex));
        }

        dp.addDatapoolListener(this.datapoolListener);
    }

    public void addEquivalenceClassPage(int newECIndex) {
        IDatapoolEquivalenceClass eqClass = (IDatapoolEquivalenceClass) getDatapool().getEquivalenceClass(newECIndex);
        IHyadesEditorPart hyadesEditorPart = getHyadesEditorPart();
        DataTableForm dataTableForm = new DataTableForm(this, new WidgetFactory(), eqClass);
        dataTableForm.initialize(eqClass);

        int index = hyadesEditorPart.addPage(dataTableForm.createControl());
        hyadesEditorPart.setPageText(index, eqClass.getName());
        equivalenceClassPages.add(newECIndex, dataTableForm);
    }

    public void removeEquivalenceClassPage(int equivalenceClassIndex) {
        DataTableForm tableForm = (DataTableForm) equivalenceClassPages.remove(equivalenceClassIndex);
        if (tableForm != null) tableForm.dispose();
        IHyadesEditorPart hyadesEditorPart = getHyadesEditorPart();
        int pageIndex = getEquivalenceClassPageIndex(equivalenceClassIndex);
        hyadesEditorPart.removePage(pageIndex);
    }
    
    private void moveEquivalenceClassPage(int sourceECIndex, int targetECIndex) {
        // Because IHyadesEditorPart can only append pages, but not insert a page in the middle
        // have to remove pages and re-add them in new order.
        int start = Math.min(sourceECIndex, targetECIndex);
        for (int i = (getDatapool().getEquivalenceClassCount() - 1); i >= start; i--)
            removeEquivalenceClassPage(i);

        for (int i = start; i < getDatapool().getEquivalenceClassCount(); i++)
            addEquivalenceClassPage(i);
    }

    public int getEquivalenceClassPageIndex(int equivalenceClassIndex) {
        return (PAGE_OVERVIEW + equivalenceClassIndex + 1);
    }

    private int getEquivalenceClassIndexFromPageIndex(int pageIndex) {
        return (pageIndex - 1);
    }

    /**
     * Returns the datapool that is manipulated by this editor
     * extension.
     * @return DPLDatapool 
     */
    public DPLDatapool getDatapool() {
        return (DPLDatapool) getHyadesEditorPart().getEditorObject();
    }

    /**
     * @see org.eclipse.hyades.ui.editor.IEditorExtension#getSelection()
     */
    public IStructuredSelection getSelection() {
        switch (getHyadesEditorPart().getActivePage()) {
            case PAGE_OVERVIEW:
                return new StructuredSelection(getDatapool());

        }

        return StructuredSelection.EMPTY;
    }

    /**
     * @see org.eclipse.hyades.ui.editor.IEditorExtension#pageActivated(int)
     */
    public boolean pageActivated(int index) {

    	//Optimization: Only activate a different page.
    	if(index != activePageIndex){
    		
	    	activePageIndex = index;
	
	        IWorkbenchPage activeWorkbenchPage = UIUtil.getActiveWorkbenchPage();
	        
	        //Attempt to show the Properties view:
	        if (activeWorkbenchPage != null) {
	            
	        	try {
	                activeWorkbenchPage.showView(IPageLayout.ID_PROP_SHEET, null, IWorkbenchPage.VIEW_VISIBLE);
	            } 
	        	catch (PartInitException p) {
	        		//Ignore since the Properties view could not be initialized.
	        	}
	        }
	
	        if (index == PAGE_OVERVIEW) {
	            return (datapoolForm.activated());
	        } 
	        else if (index > PAGE_OVERVIEW){
	            
	        	//Resolve the index of the equivalence class page:
	        	int equivalenceClassIndex = getEquivalenceClassIndexFromPageIndex(index);

	        	//Resolve the equivalence class page:
	        	DataTableForm dataTableForm = ((DataTableForm)(equivalenceClassPages.get(equivalenceClassIndex)));
	            
	        	//Create the control for the equivalence class page if it does not exist:
	            if (getHyadesEditorPart().getControl(index) == null){ 
	            	getHyadesEditorPart().setControl(index, dataTableForm.createControl());
	            }
	            
	            //Refresh without reloading the datapool table:
	            dataTableForm.getDatapoolTable().refresh(false);
	
	            //Update find replace target with the index of the equivalence class page:
	            ((DatapoolEditorExtension.FindReplaceTarget)(getFindReplaceTarget())).updatePage(index);
	
	            //Select the equivalence class in the Overview page to display its attributes in Properties view:
	            IDatapoolEquivalenceClass equivalenceClass = ((IDatapoolEquivalenceClass)(getDatapool().getEquivalenceClass(equivalenceClassIndex)));

	            if (equivalenceClass != null) {
	                datapoolForm.setSelection(new StructuredSelection(equivalenceClass));
	            }
	
	            return (dataTableForm.activated());
	        }
	
	        return false;
    	}
    	
    	return true;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.editor.extension.BaseEditorExtension#reloadEditorObject()
     */
    protected boolean reloadEditorObject() {
        // take care of this datapool listener.
        getDatapool().removeDatapoolListener(this.datapoolListener);
        boolean ret = super.reloadEditorObject();
        getDatapool().addDatapoolListener(this.datapoolListener);
        return ret;
    }

    /**
     * @see org.eclipse.hyades.ui.util.IRefreshable#refreshContent(java.lang.Object)
     */
    public void refreshContent(Object data) {
        if (!(data instanceof IDatapool)) return;

        // optimization: only refresh pages without recreating them
        if (data == getDatapool() && ((IDatapool) data).getEquivalenceClassCount() == equivalenceClassPages.size()) {
            datapoolForm.load();
            datapoolForm.updateTitle();

            for (int i = 0; i < ((IDatapool) data).getEquivalenceClassCount(); i++) {
                Object page = equivalenceClassPages.get(i);
                if (page != null) {
                    IDatapoolEquivalenceClass ec = (IDatapoolEquivalenceClass) ((IDatapool) data).getEquivalenceClass(i);
                    ((DataTableForm) page).load(ec);
                    getHyadesEditorPart().setPageText(getEquivalenceClassPageIndex(i), ec.getName());
                }
            }
        } else {
            getHyadesEditorPart().setEditorObject(data);
            int previousActivePage = activePageIndex;
            datapoolForm.load();
            datapoolForm.updateTitle();

            for (int i = (equivalenceClassPages.size() - 1); i >= 0; i--) {
                Object page = equivalenceClassPages.get(i);
                if (page != null) removeEquivalenceClassPage(i);
            }

            int count = ((IDatapool) data).getEquivalenceClassCount();
            for (int i = 0; i < count; i++) {
                addEquivalenceClassPage(i);
            }

            if ((previousActivePage > -1) && (previousActivePage < (count + 1))) getHyadesEditorPart().setActivePage(previousActivePage);
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.datapool.internal.interfaces.IDatapoolPart#isReadOnly()
     */
    public boolean isReadOnly() {
        return this.getHyadesEditorPart().isReadOnly();
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.datapool.internal.interfaces.IDatapoolPart#getActionBars()
     */
    public IActionBars getActionBars() {
        return this.getHyadesEditorPart().getEditorPart().getEditorSite().getActionBars();
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.datapool.internal.interfaces.IDatapoolPart#notifyEdit()
     */
    public void notifyEdit() {
    }

    public IEditorPart getEditorPart() {
        return this.getHyadesEditorPart().getEditorPart();
    }

    public IFindReplaceTarget getFindReplaceTarget() {
        int activePage = getHyadesEditorPart().getActivePage();
        if (activePage <= PAGE_OVERVIEW) return null;

        if (findReplaceTarget == null) {
            findReplaceTarget = new FindReplaceTarget(activePage);
        }

        return findReplaceTarget;
    }

    /**
     * 
     * @return the datapool table used in on the current page
     * @author pnedelec
     * @since 4.4
     */
    public DatapoolTable getCurrentDatapoolTable() {
        int equivalenceClassIndex = getEquivalenceClassIndexFromPageIndex(activePageIndex);
        if (equivalenceClassIndex > -1 && equivalenceClassIndex < equivalenceClassPages.size()) {
            DataTableForm dataTable = (DataTableForm) equivalenceClassPages.get(equivalenceClassIndex);
            return dataTable.getDatapoolTable();
        } else {
            return null;
        }
    }

    /** 
     * @see org.eclipse.hyades.ui.internal.editor.HyadesEditorPart#enableEditActions()
     * @see org.eclipse.hyades.ui.editor.IEditActionsExtension#connectPart(org.eclipse.ui.IWorkbenchPart)
     * 
     * @author pnedelec
     * @since 4.3
     */
    public void connectPart(IWorkbenchPart part) {
        DatapoolActionHandlerListener.INSTANCE.connectPart(part);
    }
    /**
     * used for DatapoolEditorPart to save
     * deal with the encrypted datapool
     * */
    public void save(){
    	if(super.getProgressMonitor() != null){
    		try{
    			super.save(super.getProgressMonitor());
    		}catch(Exception e){
    			UiPlugin.logError(e);
    		}
		}
    }
}
