/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ArtifactLocationMapForm.java,v 1.18 2010/03/29 13:52:35 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.configuration.CFGArtifact;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.action.model.MoveFeatureChildrenDownAction;
import org.eclipse.hyades.test.ui.action.model.MoveFeatureChildrenUpAction;
import org.eclipse.hyades.test.ui.editor.form.util.EObjectTableContentProvider;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.ExtensibleEObjectTableSection;
import org.eclipse.hyades.test.ui.editor.form.util.ExtensibleEObjectTreeSection;
import org.eclipse.hyades.test.ui.editor.form.util.TableLabelProviderAdapter;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.action.AddArtifactAction;
import org.eclipse.hyades.test.ui.internal.action.AddLocationAction;
import org.eclipse.hyades.test.ui.internal.action.OpenArtifactAction;
import org.eclipse.hyades.test.ui.internal.action.OpenLocationAction;
import org.eclipse.hyades.test.ui.internal.action.RemoveDeploymentPairElementAction;
import org.eclipse.hyades.test.ui.internal.editor.extension.ConfigurableObjectEditorExtension;
import org.eclipse.hyades.test.ui.internal.model.ui.ChildrenSelectionAction;
import org.eclipse.hyades.test.ui.internal.model.ui.MoveChildrenDownAction;
import org.eclipse.hyades.test.ui.internal.model.ui.MoveChildrenUpAction;
import org.eclipse.hyades.test.ui.internal.model.ui.RemoveChildrenAction;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.editor.IHyadesEditorPartExtended;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.AccessibleAdapter;
import org.eclipse.swt.accessibility.AccessibleEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.model.IWorkbenchAdapter;

/**
 * <p>This class implements the map tab of the deployment editor for the editing of 
 * {@link org.eclipse.hyades.models.common.configuration.CFGArtifact} - {@link org.eclipse.hyades.models.common.configuration.CFGLocation} pairs.</p>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Paul E. Slauenwhite
 * @version March 29, 2010
 * @since   February 1, 2005
 */
public class ArtifactLocationMapForm extends EditorForm implements ISelectionChangedListener, SelectionListener {
	
    public static final String PROPERTY_NAME_CLASSPATH = Common_ConfigurationFactory.eINSTANCE.createHyadesClasspathCategory().getName(); //"CLASSPATH"
    public static final String PROPERTY_NAME_ROOTDIR = Common_ConfigurationFactory.eINSTANCE.createHyadesRootDirectroyCategory().getName(); //"ROOTDIR"    
    
	private ExtensibleEObjectTreeSection artifactsSection;
	private ExtensibleEObjectTreeSection locationsSection;
	private ExtensibleEObjectTableSection mapSection;
	private AddLocationAction addLocationAction;
	private AddArtifactAction addArtifactAction;
	private Button createPairButton;
	
	/**
	 * Constructor for ArtifactLocationMapForm
	 * @param ConfigurableObjectEditorExtension
	 * @param widgetFactory
	 */
	public ArtifactLocationMapForm(ConfigurableObjectEditorExtension deploymentEditorExtension, WidgetFactory widgetFactory)
	{
		super(deploymentEditorExtension, widgetFactory);
		setHeadingText(UiPluginResourceBundle.W_DEPLOYMENT);
	}
	
	/**
	 * @see org.eclipse.update.ui.forms.internal.IForm#dispose()
	 */
	public void dispose()
	{
		if(artifactsSection != null)
		{
			artifactsSection.dispose();
			artifactsSection = null;
		}
		
		if(locationsSection != null)
		{
			locationsSection.dispose();
			locationsSection = null;
		}
		
		if(mapSection != null)
		{
			mapSection.dispose();
			mapSection = null;
		}
			
		if(createPairButton != null)
		{
			createPairButton.dispose();
			createPairButton = null;
		}								

		super.dispose();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorForm#createEditorFormContents(org.eclipse.swt.widgets.Composite)
	 */
	protected void createEditorFormContents(Composite parent)
	{
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.makeColumnsEqualWidth = true;
		gridLayout.marginWidth = 10;
		gridLayout.horizontalSpacing = 15;
		parent.setLayout(gridLayout);
		Control control;

		Composite artifactsColumn = createColumn(parent);
		Composite locationsColumn = createColumn(parent);	
		IAction[] artifactButtonActions = createArifactSectionButtonActions();
		artifactsSection = new ExtensibleEObjectTreeSection(this, Common_TestprofilePackage.eINSTANCE.getTPFDeployment_Artifacts(), artifactButtonActions);
		registerSection(artifactsSection);		
		artifactsSection.setHeaderText(UiPluginResourceBundle.W_ARTIFACTS);
		artifactsSection.setCollapsable(true);
		control = artifactsSection.createControl(artifactsColumn, getWidgetFactory());
		control.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING));
		artifactsSection.setEditable(true);		
		for(int i = 0; i < artifactButtonActions.length; i++)
		{
		    IAction action = artifactButtonActions[i];
		    if(action != null)
		    {
		        if(action instanceof ChildrenSelectionAction)		    
		            ((ChildrenSelectionAction)action).setStructuredViewer(artifactsSection.getTreeViewer());
		        else if(action instanceof OpenArtifactAction)
		            ((OpenArtifactAction)action).setStructuredViewer(artifactsSection.getTreeViewer());
		    }
		}
		
		IAction[] locationButtonActions = createLocationSectionButtonActions();
		locationsSection = new ExtensibleEObjectTreeSection(this, Common_TestprofilePackage.eINSTANCE.getTPFDeployment_RefLocations(), locationButtonActions);
		registerSection(locationsSection);
		locationsSection.setHeaderText(UiPluginResourceBundle.W_LOCS);
		locationsSection.setCollapsable(true);
		control = locationsSection.createControl(locationsColumn, getWidgetFactory());
		control.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING));
		locationsSection.setEditable(true);
		for(int i = 0; i < locationButtonActions.length; i++)
		{
		    IAction action = locationButtonActions[i];
		    if(action != null)
		    {
		        if(action instanceof ChildrenSelectionAction)		    
		            ((ChildrenSelectionAction)action).setStructuredViewer(locationsSection.getTreeViewer());
		        else if(action instanceof OpenLocationAction)
		            ((OpenLocationAction)action).setStructuredViewer(locationsSection.getTreeViewer());
		    }
		}
		    
		createPairButton = new Button(parent, SWT.PUSH); //SWT.ARROW | SWT.DOWN);
		GridData buttonData = new GridData();
		buttonData.horizontalAlignment = GridData.END;
		createPairButton.setLayoutData(buttonData);
		createPairButton.setToolTipText(UiPluginResourceBundle.BTN_CREATE_PAIR);
		createPairButton.setImage(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_DOWNARROW).createImage());
		createPairButton.setEnabled(false);
		createPairButton.addSelectionListener(this);
		createPairButton.getAccessible().addAccessibleListener(new AccessibleAdapter() {
			public void getName(AccessibleEvent e) {
				e.result = createPairButton.getToolTipText();
			}
		});
		
		//blank area to take up the second column of the createPairButtonRow
		Composite blankCell = new Composite(parent, SWT.NO_FOCUS);
		blankCell.setBackground(parent.getBackground());
		
		Composite southComposite = getFactory().createComposite(parent);
		GridData gridData = new GridData(GridData.FILL_BOTH);
		gridData.horizontalSpan = 2;
		southComposite.setLayoutData(gridData);
		
		GridLayout southLayout = new GridLayout();
		southLayout.numColumns = 1;
		southComposite.setLayout(southLayout);
		
		IAction[] mapButtonActions = createMapSectionButtonActions();
		mapSection = new ExtensibleEObjectTableSection(this, Common_TestprofilePackage.eINSTANCE.getTPFDeployment_ArtifactLocations(), 2, mapButtonActions);
		mapSection.setContentProvider(new EObjectTableContentProvider(this, Common_TestprofilePackage.eINSTANCE.getTPFDeployment_ArtifactLocations())
			{
				/* (non-Javadoc)
				 * @see org.eclipse.hyades.test.ui.editor.form.util.EObjectContainerContentProvider#getElements(java.lang.Object)
				 */
				public Object[] getElements(Object inputElement) 
				{
					List resolvedEls = new ArrayList();
					Object[] elements = super.getElements(inputElement);					
					for(int i = 0; i < elements.length; i++)
					{
						if(elements[i] instanceof CFGArtifactLocationPair)
						{
							CFGArtifactLocationPair pair = (CFGArtifactLocationPair)elements[i];
							if((pair.getArtifact() == null) || (pair.getLocation() == null)) {
								continue;
							}
							if(pair.getArtifact().eIsProxy() || pair.getLocation().eIsProxy())
								continue;
							resolvedEls.add(pair);
						}
					}
					return resolvedEls.toArray();
				}
			}
		);        
		mapSection.setLabelProvider(new TableLabelProviderAdapter()
	        {		    	
		    	public Image getColumnImage(Object element, int columnIndex)
		    	{
		    	    return null;		    	    
		    	}
		    	
		    	public String getColumnText(Object element, int columnIndex)
		    	{
		    		CMNNamedElement namedElement = null;

		    		if(element instanceof CFGArtifactLocationPair){

		    			if(columnIndex == 0){
		    				namedElement = ((CFGArtifactLocationPair)(element)).getArtifact();
		    			}
		    			else if(columnIndex == 1){
		    				namedElement = ((CFGArtifactLocationPair)(element)).getLocation();
		    			}		    	     
		    		}
		    		
		    		if(namedElement != null){
		    			return (namedElement.getName());
		    		}
		    		
		    	    return "";
		    	}
	        });
		registerSection(mapSection);
		mapSection.setHeaderText(UiPluginResourceBundle.EDT_DEP_PAIR);
		mapSection.setCollapsable(true);
		control = mapSection.createControl(southComposite, getWidgetFactory());
		GridData mapData = new GridData(GridData.FILL_BOTH);
		mapData.grabExcessHorizontalSpace = true;
		mapData.grabExcessVerticalSpace =   true;
		control.setLayoutData(mapData);
		for(int i = 0; i < mapButtonActions.length; i++)
		{
		    IAction action = mapButtonActions[i];
		    if(action != null && action instanceof ChildrenSelectionAction)
		        ((ChildrenSelectionAction)action).setStructuredViewer(mapSection.getTableViewer());
		}
	}
	
	/**
	 * Creates the actions of the buttons that the location tree sections will use. 
	 * @return
	 */
	protected IAction[] createLocationSectionButtonActions()
	{
		IAction[] buttonActions = new IAction[5];
		
		buttonActions[0] = addLocationAction = new AddLocationAction();
		buttonActions[0].setText(UiPluginResourceBundle.EDT_DEP_LOC_LBL_ADD);
		
		buttonActions[1] = new RemoveDeploymentPairElementAction(getEditorObject(), Common_TestprofilePackage.eINSTANCE.getTPFDeployment_RefLocations(), Common_TestprofilePackage.eINSTANCE.getTPFDeployment_ArtifactLocations(), false);	
		buttonActions[1].setText(UiPluginResourceBundle.EDT_DEP_LOC_LBL_REMOVE);
		
		buttonActions[2] = new MoveFeatureChildrenUpAction(getEditorObject(), Common_TestprofilePackage.eINSTANCE.getTPFDeployment_RefLocations());
		buttonActions[2].setText(UiPluginResourceBundle.EDT_DEP_LOC_LBL_UP);
		
		buttonActions[3] = new MoveFeatureChildrenDownAction(getEditorObject(), Common_TestprofilePackage.eINSTANCE.getTPFDeployment_RefLocations());
		buttonActions[3].setText(UiPluginResourceBundle.EDT_DEP_LOC_LBL_DOWN);
		
		buttonActions[4] = new OpenLocationAction();		
		buttonActions[4].setText(UiPluginResourceBundle.EDT_DEP_LOC_LBL_OPEN);
		
		return buttonActions;
	}	
	
	/**
	 * Creates the actions of the buttons that the location tree sections will use. 
	 * @return
	 */
	protected IAction[] createArifactSectionButtonActions()
	{
		IAction[] buttonActions = new IAction[5];
				
 		buttonActions[0] = addArtifactAction = new AddArtifactAction();
		buttonActions[0].setText(UiPluginResourceBundle.EDT_DEP_ART_LBL_ADD);

 		buttonActions[1] = new RemoveDeploymentPairElementAction(getEditorObject(), Common_TestprofilePackage.eINSTANCE.getTPFDeployment_Artifacts(), Common_TestprofilePackage.eINSTANCE.getTPFDeployment_ArtifactLocations(), false);	
		buttonActions[1].setText(UiPluginResourceBundle.EDT_DEP_ART_LBL_REMOVE);

 		buttonActions[2] = new MoveFeatureChildrenUpAction(getEditorObject(), Common_TestprofilePackage.eINSTANCE.getTPFDeployment_Artifacts());
		buttonActions[2].setText(UiPluginResourceBundle.EDT_DEP_ART_LBL_UP);

 		buttonActions[3] = new MoveFeatureChildrenDownAction(getEditorObject(), Common_TestprofilePackage.eINSTANCE.getTPFDeployment_Artifacts());
		buttonActions[3].setText(UiPluginResourceBundle.EDT_DEP_ART_LBL_DOWN);

 		buttonActions[4] = new OpenArtifactAction();		
		buttonActions[4].setText(UiPluginResourceBundle.EDT_DEP_ART_LBL_OPEN);

		return buttonActions;
	}		
	
	protected IAction[] createMapSectionButtonActions()
	{
		IAction[] buttonActions = new IAction[3];
		buttonActions[0] = new RemoveChildrenAction(false);
		buttonActions[0].setText(UiPluginResourceBundle.EDT_DEP_PAIR_LBL_REMOVE);

		buttonActions[1] = new MoveChildrenUpAction();
		buttonActions[1].setText(UiPluginResourceBundle.EDT_DEP_PAIR_LBL_UP);
		((MoveChildrenUpAction)buttonActions[1]).setEStructuralFeature(Common_TestprofilePackage.eINSTANCE.getTPFDeployment_ArtifactLocations());

		buttonActions[2] = new MoveChildrenDownAction();
		buttonActions[2].setText(UiPluginResourceBundle.EDT_DEP_PAIR_LBL_DOWN);
		((MoveChildrenDownAction)buttonActions[2]).setEStructuralFeature(Common_TestprofilePackage.eINSTANCE.getTPFDeployment_ArtifactLocations());

		return buttonActions;
	}

	/**
	 * Returns the {@link org.eclipse.hyades.models.common.testprofile.TPFDeployment} object that is edited by this form.
	 * @return EObject 
	 */
	protected EObject getEditorObject()
	{
		return ((ConfigurableObjectEditorExtension)getBaseEditorExtension()).getEditorObject();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorForm#load()
	 */
	public void load()
	{
		TPFDeployment deployment = (TPFDeployment)getEditorObject();
		addArtifactAction.setDeployment(deployment);
		addLocationAction.setDeployment(deployment);		

		artifactsSection.getTreeViewer().removeSelectionChangedListener(this);
		artifactsSection.setInput(deployment);
		artifactsSection.getTreeViewer().addSelectionChangedListener(this);
		
		locationsSection.getTreeViewer().removeSelectionChangedListener(this);
		locationsSection.setInput(deployment);
		locationsSection.getTreeViewer().addSelectionChangedListener(this);
		
		mapSection.setInput(deployment);
	}
	
	public ISelection getSelection()
	{
		return mapSection.getTableViewer().getSelection();
	}
		
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#updateTitle()
	 */
	public void updateTitle()
	{
		String title = ((IWorkbenchAdapter)((IAdaptable)getEditorObject()).getAdapter(IWorkbenchAdapter.class)).getLabel(getEditorObject());
		if(title != null)
			((IHyadesEditorPartExtended)getBaseEditorExtension().getHyadesEditorPart()).setEditorTitle(title);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event)
	{
		IStructuredSelection artifactSelection = artifactsSection.getStructuredSelection();
		IStructuredSelection locationSelection = locationsSection.getStructuredSelection();
		if(artifactSelection.size() == 1 && locationSelection.size() == 1)
		{
		    if(pairExists((CFGArtifact)artifactSelection.getFirstElement(), (CFGLocation)locationSelection.getFirstElement()))
		    {
		        createPairButton.setEnabled(false);
		        return;
		    }
			createPairButton.setEnabled(true);
		}
		else
			createPairButton.setEnabled(false);		
		
		if(event.getSource() == locationsSection.getTreeViewer() ||
		   event.getSource() == artifactsSection.getTreeViewer())
		{
			UIUtil.getActiveWorkbenchPage().activate((IWorkbenchPart)getBaseEditorExtension().getHyadesEditorPart());
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e)
	{
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e){
		
		if(e.widget == createPairButton){
			
			//Cancel this selection event if the editor is opened in read-only mode:
			if(getBaseEditorExtension().getHyadesEditorPart().isReadOnly()){
				e.doit = false;
			}
			else{
				
				IStructuredSelection artifactSelection = artifactsSection.getStructuredSelection();
				IStructuredSelection locationSelection = locationsSection.getStructuredSelection();

				if((artifactSelection.size() == 1) && (locationSelection.size() == 1) && (!pairExists((CFGArtifact)artifactSelection.getFirstElement(), (CFGLocation)locationSelection.getFirstElement()))){

					CFGArtifactLocationPair pair = Common_ConfigurationFactory.eINSTANCE.createCFGArtifactLocationPair();
					pair.setArtifact(((CFGArtifact)(artifactSelection.getFirstElement())));
					pair.setLocation(((CFGLocation)(locationSelection.getFirstElement())));

					((TPFDeployment)(getEditorObject())).getArtifactLocations().add(pair);

					createPairButton.setEnabled(false);
				}
			}
		}
	}
	
	private boolean pairExists(CFGArtifact art, CFGLocation loc)
	{
	    TPFDeployment deployment = (TPFDeployment)getEditorObject();
		EList list = deployment.getArtifactLocations();
		if(!list.isEmpty())
		{
			for(Iterator it = list.iterator(); it.hasNext();)
			{
				CFGArtifactLocationPair pair = (CFGArtifactLocationPair)it.next();
				CFGArtifact artifact = pair.getArtifact();
				CFGLocation location = pair.getLocation();
				if(artifact == art && location == loc)
				{
				    return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * @see org.eclipse.ui.part.ISetSelectionTarget#selectReveal(org.eclipse.jface.viewers.ISelection)
	 */
	public void selectReveal(ISelection selection)
	{
	    if(selection.isEmpty())	
	        return;
	    
	    Object obj = ((IStructuredSelection)selection).getFirstElement();
	    if(obj instanceof CFGArtifact)
	        artifactsSection.selectReveal(selection);
	    else if(obj instanceof CFGLocation)
	        locationsSection.selectReveal(selection);
	}
}
