/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ConfigurableObjectOverviewForm.java,v 1.15 2007/05/03 20:51:08 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form;

import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.configuration.CFGCategory;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationPackage;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.NamedElementsSection;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.extension.ConfigurableObjectEditorExtension;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormSection;
import org.eclipse.hyades.test.ui.internal.editor.form.util.GeneralInfoSection;
import org.eclipse.hyades.test.ui.internal.editor.form.util.RegisteredExtensionsSection;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.editor.IHyadesEditorPartExtended;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.model.IWorkbenchAdapter;

/**
 * This class provides a base implementation for {@link org.eclipse.hyades.models.common.configuration.CFGConfigurableObject} editor overview page.
 * 
 * @author bjiang
 * @since  1.3
 */
public class ConfigurableObjectOverviewForm extends EditorForm 
{	
	private FormSection generalInfoSection;
	private RegisteredExtensionsSection propertyGroupsSection;
	private Hashtable propGroupSectionsMap = new Hashtable();
	private Composite leftColumnContainer;
	private Composite rightColumnContainer;
	
	/**
	 * Constructor for ConfigurableObjectOverviewForm
	 * @param editorExtension
	 * @param widgetFactory
	 */
	public ConfigurableObjectOverviewForm(ConfigurableObjectEditorExtension editorExtension, WidgetFactory widgetFactory)
	{
		super(editorExtension, widgetFactory);
	}
	
	/**
	 * @see org.eclipse.update.ui.forms.internal.IForm#dispose()
	 */
	public void dispose()
	{
		if(generalInfoSection != null)
		{
			generalInfoSection.dispose();
			generalInfoSection = null;
		}
		Enumeration elements = propGroupSectionsMap.elements();
		NamedElementsSection section;
		while(elements.hasMoreElements())
		{
			section = (NamedElementsSection)elements.nextElement();
			if(section != null)
			{
				section.dispose();
				section = null;
			}
		}
		propGroupSectionsMap.clear();
		propGroupSectionsMap = null;
	
		super.dispose();
	}
	
	/**
	 * Returns the left column composite which is one of the two columns made up this overview page.
	 * @return
	 */
	protected Composite getLeftColumnContainer()
	{
		return this.leftColumnContainer;
	}
	
	/**
	 * Returns the right column composite which is one of the two columns made up this overview page.
	 * @return
	 */
	protected Composite getRightColumnContainer()
	{
		return this.rightColumnContainer;
	}
	
	protected FormSection getGeneralInfoSection()
	{
		return this.generalInfoSection;
	}
	
	protected void setGeneralInfoSection(FormSection section)
	{
		this.generalInfoSection = section;
	}
	
	/**
	 * Returns the data object that is manipulated by this editor.
	 * @return EObject 
	 */
	protected EObject getDataObject()
	{
		return ((ConfigurableObjectEditorExtension)getBaseEditorExtension()).getEditorObject();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorForm#createEditorFormContents(org.eclipse.swt.widgets.Composite)
	 */
	protected void createEditorFormContents(Composite parent) 
	{
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.makeColumnsEqualWidth = true;
		gridLayout.marginWidth = 10;
		gridLayout.horizontalSpacing = 15;
		parent.setLayout(gridLayout);

		leftColumnContainer = createColumn(parent);
		rightColumnContainer = createColumn(parent);  
		
		createGeneralInfoSection(leftColumnContainer);
		IConfigurationElement[] propertyGroupExtensions = ((ConfigurableObjectEditorExtension)getBaseEditorExtension()).getPropertyGroupExtensions();
		if(propertyGroupExtensions != null && propertyGroupExtensions.length > 0)
			createPropertyGroupExtensionsSection(leftColumnContainer);
	}

	/**
	 * Method used to create the general information section on this overview page.
	 * Sub classes may use or override this method to custimize it. 
	 * @param parent
	 */
	protected void createGeneralInfoSection(Composite parent) 
	{
		Control control;
		generalInfoSection = new GeneralInfoSection(this, true);
		((GeneralInfoSection)generalInfoSection).setEditable(true);
		registerSection(generalInfoSection);		
		control = generalInfoSection.createControl(parent, getWidgetFactory());
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING);
		control.setLayoutData(gridData);		
		setGeneralInfoSectionHelp(control);
	}
	
	/**
	 * Sub classes should override this method to add their own help.
	 * @param control
	 */
	protected void setGeneralInfoSectionHelp(Control control)
	{
	}
	
	/**
	 * Method used to create a section to list property group extensions <i>"org.eclipse.hyades.test.ui.configurableObjectExtension"</i>
	 * on this overview page.
	 * This section is not created by default, so sub classes should call this method to add this section at will. 
	 * It can also be overridden for customization. 
	 * @param parent
	 */
	protected RegisteredExtensionsSection createPropertyGroupExtensionsSection(Composite parent)
	{
		IConfigurationElement[] propertyGroupExtensions = ((ConfigurableObjectEditorExtension)getBaseEditorExtension()).getPropertyGroupExtensions();
		if(propertyGroupExtensions == null)
			return null;
		propertyGroupsSection = new RegisteredExtensionsSection((ConfigurableObjectEditorExtension)getBaseEditorExtension(), propertyGroupExtensions);
		registerSection(propertyGroupsSection);
		propertyGroupsSection.setHeaderText( UiPluginResourceBundle.EDT_REGISTERED_PROP_GROUPS ); 
		propertyGroupsSection.setDescription( UiPluginResourceBundle.EDT_REG_PROP_GRP_DSC ); 
		Control control = propertyGroupsSection.createControl(parent, getWidgetFactory());
		control.setLayoutData(new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING));
		return propertyGroupsSection;
	}
	
	/**
	 * Returns the section for property group extensions or <code>null</code> if it is not created.
	 * @return
	 */
	protected RegisteredExtensionsSection getPropertyGroupsSection()
	{
		return propertyGroupsSection;
	}
	
	/**
	 * Adds a new section on this overview page to link to the specified property group page. 
	 * A new section will be added for each property group page created. 
	 * Sub classes can override this method to customize this section.
	 * @param propGroupExtensionForm
	 * @param pageIndex
	 */
	public void addSection(IPropertyGroupForm propGroupExtensionForm, int pageIndex)
	{				
		NamedElementsSection section = new NamedElementsSection(getBaseEditorExtension(), pageIndex);
		registerSection(section);
		section.createControl(rightColumnContainer, getWidgetFactory(), propGroupExtensionForm.getName(), propGroupExtensionForm.getDescription());
	
		CFGPropertyGroup propertyGroup = propGroupExtensionForm.getPropertyGroup(); 
		propGroupSectionsMap.put(propertyGroup, section);
		section.initialize(propertyGroup, Common_ConfigurationPackage.eINSTANCE.getCFGPropertyGroup_Properties(), null, new LabelProvider()
		{
			public String getText(Object element)
			{
				if(element instanceof CFGComparableProperty)
				{
					CFGCategory type = ((CFGComparableProperty)element).getType();
					if(type != null)
						return type.getDisplayName();
				}
				return ((CMNNamedElement)element).getName();
			}
		});
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorForm#load()
	 */
	public void load() 
	{
		((GeneralInfoSection)generalInfoSection).setInput(getDataObject());
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#updateTitle()
	 */
	public void updateTitle()
	{
		String title = ((IWorkbenchAdapter)((IAdaptable)getDataObject()).getAdapter(IWorkbenchAdapter.class)).getLabel(getDataObject());
		if(title != null)
			((IHyadesEditorPartExtended)getBaseEditorExtension().getHyadesEditorPart()).setEditorTitle(title);
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#activated()
	 */
	public boolean activated()
	{
		return true;
	}
}
