/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionForm.java,v 1.17 2008/01/21 17:19:20 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form;

import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFInvocationEvent;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.editor.form.base.IHyperlinkListener;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.NamedElementSection;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.extension.ExecutionEditorExtension;
import org.eclipse.hyades.test.ui.internal.editor.form.base.SelectableFormLabel;
import org.eclipse.hyades.test.ui.internal.editor.form.util.DetailSection;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.hyades.ui.editor.IHyadesEditorPartExtended;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.model.IWorkbenchAdapter;

/**
 * @author marcelop, jcanches
 * @since 0.3.0
 */
public class ExecutionForm extends EditorForm
{	
	private class ExecutionSection extends NamedElementSection implements IHyperlinkListener {
		private StyledText fileText;
		private SelectableFormLabel gotoFirstErrorLink;
		private TPFVerdictEvent firstFailEvent;
		public ExecutionSection(EditorForm editorForm) {
			super(editorForm);
		}
		protected void addSouthControls(Composite parent, WidgetFactory widgetFactory) {
			Composite detailComposite = getWidgetFactory().createComposite(parent);
			GridLayout gridLayout = new GridLayout(2, false);
			gridLayout.marginHeight = 0;
			gridLayout.marginWidth = 0;
			detailComposite.setLayout(gridLayout);
			detailComposite.setLayoutData(GridDataUtil.createHorizontalFill());

			getWidgetFactory().createLabel(detailComposite, UiPluginResourceBundle.L_FILE); 
			fileText = getWidgetFactory().createStyledText(detailComposite, SWT.FULL_SELECTION | SWT.SINGLE);
			fileText.setEnabled(false);
			fileText.setLayoutData(GridDataUtil.createHorizontalFill());
			UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(fileText, UiPlugin.getID() + ContextIds.DEPL_TEXT_FORM);		
			
			gotoFirstErrorLink = widgetFactory.createSelectableLabel(parent, ""); //$NON-NLS-1$
			widgetFactory.turnIntoHyperlink(gotoFirstErrorLink, this);
		}
		public void linkActivated(Control linkLabel) {
			if (firstFailEvent == null) {
				// Activate the second page				
				getEditorForm().getBaseEditorExtension().getHyadesEditorPart().setActivePage(1);
			} else {
				getEditorForm().getBaseEditorExtension().setSelection(new StructuredSelection(firstFailEvent));
			}
		}
		public void linkEntered(Control linkLabel) {
			// NOP
		}
		public void linkExited(Control linkLabel) {
			// NOP
		}
		public void setInput(Object object)
		{
			super.setInput(object);
			TPFExecutionResult execResult = getExecution();
			if (execResult.eResource() != null) {
				fileText.setText(EMFUtil.getFilePath(getExecution()));
			} else {
				// This should not happen normally, but has already happened
				// in buggy situations due to the loaders. Handle it properly.
				fileText.setText("<not saved>");
			}
			TPFVerdictEvent event = getFirstFailEvent(execResult);
			setFirstFailEvent(event);
		}
		public void setFirstFailEvent(TPFVerdictEvent event) {
			firstFailEvent = event;
			if (firstFailEvent == null) {
				gotoFirstErrorLink.setText(UiPluginResourceBundle.ExecutionForm_goToEventsPageLink); 
				gotoFirstErrorLink.setToolTipText(UiPluginResourceBundle.ExecutionForm_goToEventsPageTooltip); 
			} else {
				gotoFirstErrorLink.setText(UiPluginResourceBundle.ExecutionForm_goToFirstFailLink); 
				gotoFirstErrorLink.setToolTipText(UiPluginResourceBundle.ExecutionForm_goToFirstFailTooltip); 
			}
		}
	}
	
	protected TPFVerdictEvent getFirstFailEvent(TPFExecutionResult result) {
		return getFirstFailEvent(result.getExecutionHistory().getExecutionEvents());
	}
	
	protected TPFVerdictEvent getFirstFailEvent(List executionEvents) {
		Iterator it = executionEvents.iterator();
		while (it.hasNext()) {
			TPFExecutionEvent event = (TPFExecutionEvent) it.next();
			TPFVerdictEvent firstFail;
			if (event instanceof TPFInvocationEvent) {				
				TPFInvocationEvent iEvent = (TPFInvocationEvent) event;
				TPFExecutionResult result = iEvent.getInvokedExecutionResult();
				if (result != null && (result.getVerdict() == TPFVerdict.FAIL_LITERAL
					|| result.getVerdict() == TPFVerdict.ERROR_LITERAL)) {
					firstFail = getFirstFailEvent(result);
				} else {
					firstFail = null;
				}
			} else if (event instanceof TPFVerdictEvent) {
				TPFVerdictEvent vEvent = (TPFVerdictEvent)event;
				if (vEvent.getVerdict() == TPFVerdict.FAIL_LITERAL
					|| vEvent.getVerdict() == TPFVerdict.ERROR_LITERAL) {
					firstFail = vEvent;
				} else {
					firstFail = null;
				}
			} else {
				firstFail = getFirstFailEvent(event.getChildren());
			}
			if (firstFail != null) return firstFail;
		}
		return null;
	}
	
	private ExecutionSection executionSection;
	private DetailSection detailSection;
	
	/**
	 * Constructor for ExecutionForm
	 * @param executionEditorExtension
	 * @param widgetFactory
	 */
	public ExecutionForm(ExecutionEditorExtension executionEditorExtension, WidgetFactory widgetFactory)
	{
		super(executionEditorExtension, widgetFactory);
		setHeadingText(UiPluginResourceBundle.W_EXECUTION); 
	}

	/**
	 * @see org.eclipse.update.ui.forms.internal.IForm#dispose()
	 */
	public void dispose()
	{
		executionSection.dispose();
		executionSection = null;

		detailSection.dispose();
		
		super.dispose();
	}

	/**
	 * Returns the execution manipulated by this form.
	 * @return TPFExecutionResult
	 */
	public TPFExecutionResult getExecution()
	{
		return ((ExecutionEditorExtension)getBaseEditorExtension()).getExecution();		
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorForm#createControl()
	 */
	public Control createControl()
	{
		Control control = super.createControl();
		return control;
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#createEditorFormContents(org.eclipse.swt.widgets.Composite)
	 */
	protected void createEditorFormContents(Composite parent)
	{
		UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(parent, UiPlugin.getID() + ContextIds.EXEC_EDIT_OVERVIEW);
		
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.makeColumnsEqualWidth = true;
		gridLayout.marginWidth = 10;
		gridLayout.horizontalSpacing = 15;
		parent.setLayout(gridLayout);

		Composite leftColumn = createColumn(parent);
		Composite rightColumn = createColumn(parent);

		Control control = createGenericInformation(leftColumn);
		control.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING));

		detailSection = new DetailSection(this);
		detailSection.setDetailPageProvider(new ExecutionResultDetailPageProvider());
		detailSection.setHeaderText(UiPluginResourceBundle.W_SUMMARY); 
		registerSection(detailSection);
		control = detailSection.createControl(rightColumn, getWidgetFactory());
		GridData gridData = new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING);
		control.setLayoutData(gridData);
	}
	
	/**
	 * Creates the generic information form section.
	 * @return FormSection
	 */
	protected Control createGenericInformation(Composite parent)
	{
		executionSection = new ExecutionSection(this);
		registerSection(executionSection);		
		executionSection.setHeaderText(UiPluginResourceBundle.EDT_GENERAL_INFO); 
		executionSection.setDescription(UiPluginResourceBundle.EDT_GENERAL_DSC); 
		return executionSection.createControl(parent, getWidgetFactory());
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#load()
	 */
	public void load()
	{
		executionSection.setInput(getExecution());
		detailSection.setInput(getExecution());
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#updateTitle()
	 */
	public void updateTitle()
	{
		String title = ((IWorkbenchAdapter)((IAdaptable)getExecution()).getAdapter(IWorkbenchAdapter.class)).getLabel(getExecution());
		if(title != null)
			((IHyadesEditorPartExtended)getBaseEditorExtension().getHyadesEditorPart()).setEditorTitle(title);
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#activated()
	 */
	public boolean activated()
	{
		return true;
	}
	
}
