/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestSuiteForm.java,v 1.11 2008/01/21 17:19:20 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.editor.extension.TestSuiteEditorExtension;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.NamedElementSection;
import org.eclipse.hyades.test.ui.editor.form.util.NamedElementsSection;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.editor.IHyadesEditorPartExtended;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.model.IWorkbenchAdapter;

/**
 * @author marcelop
 * @since 0.2.0
 */
public class TestSuiteForm 
extends EditorForm
{	
	private int testCasePageIndex;
	private int testComponentPageIndex;
	private int sutPageIndex;

	private NamedElementSection namedElementSection;	
	
	private NamedElementsSection testCasesSection;
	private NamedElementsSection testComponentsSection;
	private NamedElementsSection sutsSection;
	
	/**
	 * Constructor for TestSuiteForm
	 * @param testSuiteEditorExtension
	 * @param widgetFactory
	 */
	public TestSuiteForm(TestSuiteEditorExtension testSuiteEditorExtension, WidgetFactory widgetFactory)
	{
		super(testSuiteEditorExtension, widgetFactory);
		setHeadingText(UiPluginResourceBundle.W_OVERVIEW); 
	}

	/**
	 * @see org.eclipse.update.ui.forms.internal.IForm#dispose()
	 */
	public void dispose()
	{
		namedElementSection.dispose();
		namedElementSection = null;
		
		testCasesSection.dispose();
		testCasesSection = null;
		
		testComponentsSection.dispose();
		testComponentsSection = null;
		
		sutsSection.dispose();
		sutsSection = null;
		
		super.dispose();
	}

	/**
	 * Returns the test case suite manipulated by this form.
	 * @return TPFTestSuite
	 */
	protected TPFTestSuite getTestSuite()
	{
		return ((TestSuiteEditorExtension)getBaseEditorExtension()).getTestSuite();		
	}

	/**
	 * Sets the index of the test case page that is presented when the "more" button
	 * of the test cases section is pressed. 
	 * @param index
	 */
	public void setTestCasePageIndex(int index)
	{
		testCasePageIndex = index;
	}

	/**
	 * Sets the index of the test component page that is presented when the "more" button
	 * of the test components section is pressed. 
	 * @param index
	 */
	public void setTestComponentPageIndex(int index)
	{
		testComponentPageIndex = index;
	}

	/**
	 * Sets the index of the sut page that is presented when the "more" button
	 * of the test cases section is pressed. 
	 * @param index
	 */
	public void setSUTIndex(int index)
	{
		sutPageIndex = index;
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorForm#createControl()
	 */
	public Control createControl()
	{
		Control control = super.createControl();
		UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(getControl(), UiPlugin.getID() + ContextIds.TESTSUITE_FORM);
		return control;
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#createEditorFormContents(org.eclipse.swt.widgets.Composite)
	 */
	protected void createEditorFormContents(Composite parent)
	{
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.makeColumnsEqualWidth = true;
		gridLayout.marginWidth = 10;
		gridLayout.horizontalSpacing = 15;
		parent.setLayout(gridLayout);

		Composite leftColumn = createColumn(parent);
		Composite rightColumn = createColumn(parent);

		Control control = createGenericInformation(leftColumn);
		control.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING));
		UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(control, UiPlugin.getID() + ContextIds.TESTSUITE_GEN_FORM);										

		testCasesSection = new NamedElementsSection(getBaseEditorExtension(), testCasePageIndex);
		registerSection(testCasesSection);
		testCasesSection.createControl(leftColumn, factory, 
		        UiPluginResourceBundle.W_TST_CASES,  
		        UiPluginResourceBundle.EDT_TST_CASE_DSC); 

		testComponentsSection = new NamedElementsSection(getBaseEditorExtension(), testComponentPageIndex);
		registerSection(testComponentsSection);
		testComponentsSection.createControl(rightColumn, factory, 
		        UiPluginResourceBundle.W_TST_COMPS,  
		        UiPluginResourceBundle.EDT_TST_COMP_DSC); 

		sutsSection = new NamedElementsSection(getBaseEditorExtension(), sutPageIndex);
		registerSection(sutsSection);
		sutsSection.createControl(rightColumn, factory, 
		        UiPluginResourceBundle.W_SUTS,  
		        UiPluginResourceBundle.EDT_SUTS_DSC); 
	}
	
	/**
	 * Creates the generic information form section.
	 * @return FormSection
	 */
	protected Control createGenericInformation(Composite parent)
	{
		namedElementSection = new NamedElementSection(this)
		{
			private StyledText typeText;
			private StyledText fileText;
			
			protected void addSouthControls(Composite parent, FormWidgetFactory formWidgetFactory)
			{
				Composite detailComposite = getWidgetFactory().createComposite(parent);
				GridLayout gridLayout = new GridLayout(2, false);
				gridLayout.marginHeight = 0;
				gridLayout.marginWidth = 0;
				detailComposite.setLayout(gridLayout);
				detailComposite.setLayoutData(GridDataUtil.createHorizontalFill());

				getWidgetFactory().createLabel(detailComposite, UiPluginResourceBundle.L_TYPE); 
				typeText = getWidgetFactory().createStyledText(detailComposite, SWT.FULL_SELECTION | SWT.SINGLE);
				typeText.setEnabled(false);
				typeText.setLayoutData(GridDataUtil.createHorizontalFill());

				getWidgetFactory().createLabel(detailComposite, UiPluginResourceBundle.L_FILE); 
				fileText = getWidgetFactory().createStyledText(detailComposite, SWT.FULL_SELECTION | SWT.SINGLE);
				fileText.setEnabled(false);
				fileText.setLayoutData(GridDataUtil.createHorizontalFill());

				UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(typeText, UiPlugin.getID() + ContextIds.TESTSUITE_TYPE_FORM);														
				UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(fileText, UiPlugin.getID() + ContextIds.TESTSUITE_FILE_FORM);										
				
			}
			
			public void setInput(Object object)
			{
				super.setInput(object);
				
				String value = getTestSuite().getType();
				if(value != null)
				{
					IAssociationDescriptor descriptor = TestUIExtension.getTestSuiteMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS).getDefaultAssociationDescriptor(value);
					if((descriptor != null) && (descriptor.getName() != null))
						value = descriptor.getName();
					typeText.setText(value);
				}
				fileText.setText(EMFUtil.getFilePath(getTestSuite()));						
			}
		};
		registerSection(namedElementSection);		
		namedElementSection.setHeaderText(UiPluginResourceBundle.EDT_GENERAL_INFO); 
		namedElementSection.setDescription(UiPluginResourceBundle.EDT_GENERAL_DSC); 
		return namedElementSection.createControl(parent, getWidgetFactory());
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#load()
	 */
	public void load()
	{
		namedElementSection.setInput(getTestSuite());

		testCasesSection.initialize(getTestSuite(), 
		        Common_TestprofilePackage.eINSTANCE.getTPFTestSuite_TestCases(), 
		        UiPluginResourceBundle.W_TST_CASES); 
		testComponentsSection.initialize(getTestSuite(), 
		        Common_TestprofilePackage.eINSTANCE.getTPFTestSuite_TestComponents(), 
		        UiPluginResourceBundle.W_TST_COMPS); 
		sutsSection.initialize(getTestSuite(), 
		        Common_TestprofilePackage.eINSTANCE.getTPFTestSuite_SUTs(), 
		        UiPluginResourceBundle.W_SUTS); 
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#updateTitle()
	 */
	public void updateTitle()
	{
		String title = ((IWorkbenchAdapter)((IAdaptable)getTestSuite()).getAdapter(IWorkbenchAdapter.class)).getLabel(getTestSuite());
		if(title != null)
			((IHyadesEditorPartExtended)getBaseEditorExtension().getHyadesEditorPart()).setEditorTitle(title);
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#activated()
	 */
	public boolean activated()
	{
		return true;
	}
}
