/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FormWidgetFactory.java,v 1.7 2010/03/31 20:56:00 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form.base;

import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.hyades.test.ui.editor.form.base.IHyperlinkListener;
import org.eclipse.jface.resource.JFaceColors;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.actions.ActionFactory;

/**
 * <p>FormWidgetFactory.java</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version March 31, 2010
 * @since   February 1, 2005
 */
public class FormWidgetFactory {
	
	private Hashtable<String, Color> colorRegistry = null;
	private Display display = null;
	private Color borderColor = null;
	private Color backgroundColor = null;
	private Color foregroundColor = null;
	private BorderPainter borderPainter = null;
	private HyperlinkHandler hyperlinkHandler = null;
	private KeyListener deleteListener = null;
	private KeyboardHandler keyboardHandler = null;

	public static final int BORDER_STYLE = SWT.NONE;
	public static final String KEY_DRAW_BORDER = "FormWidgetFactory.drawBorder"; //$NON-NLS-1$
	public static final String TREE_BORDER = "treeBorder"; //$NON-NLS-1$
	public static final String DEFAULT_HEADER_COLOR = "__default__header__"; //$NON-NLS-1$
	public static final String COLOR_BORDER = "__border"; //$NON-NLS-1$
	public static final String COLOR_COMPOSITE_SEPARATOR = "__compSep"; //$NON-NLS-1$

	protected class BorderPainter implements PaintListener {
		
		public BorderPainter(){
			super();
		}
		
		public void paintControl(PaintEvent event) {
			
			Control[] children = ((Composite)(event.widget)).getChildren();
			
			for (int counter = 0; counter < children.length; counter++) {
				
				Control control = children[counter];

				if (!(control instanceof SelectableFormLabel)){

					boolean inactiveBorder = false;
					
					Object flag = control.getData(KEY_DRAW_BORDER);
					
					if (flag != null) {
						
						if (flag.equals(Boolean.FALSE)){
							continue;
						}
						else if (flag.equals(TREE_BORDER)){
							inactiveBorder = true;
						}
					}

					if ((!inactiveBorder) && ((control instanceof Text) || (control instanceof Canvas) || (control instanceof CCombo))) {

						Rectangle rectangle = control.getBounds();
						
						GC gc = event.gc;
						gc.setForeground(control.getBackground());
						gc.drawRectangle((rectangle.x - 1), (rectangle.y - 1), (rectangle.width + 1), (rectangle.height + 1));
						gc.setForeground(getForegroundColor());

						if (control instanceof StyledText){
							gc.drawRectangle((rectangle.x - 2), (rectangle.y - 2), (rectangle.width + 2), (rectangle.height + 3));
						}
						else{ 
							gc.drawRectangle((rectangle.x - 1), (rectangle.y - 1), (rectangle.width + 1), (rectangle.height + 1));
						}
					} 
					else if ((inactiveBorder) || (control instanceof Table) || (control instanceof Tree) || (control instanceof TableTree)) {
						
						Rectangle rectangle = control.getBounds();
						
						GC gc = event.gc;
						gc.setForeground(borderColor);
						gc.drawRectangle((rectangle.x - 1), (rectangle.y - 1), (rectangle.width + 2), (rectangle.height + 2));
					}
				}
			}
		}
	}

	class KeyboardHandler extends KeyAdapter {
		public void keyPressed(KeyEvent e) {
			Widget w = e.widget;
			if (w instanceof Control) {
				processKey(e.keyCode, (Control) w);
			}
		}
	}

	public FormWidgetFactory() {
		this(Display.getCurrent());
	}

	public FormWidgetFactory(Display display) {
		
		this.display = display;
		this.colorRegistry = new Hashtable<String, Color>();
		
		initialize();
	}

	public static ScrolledComposite getScrolledComposite(Control c) {
		Composite parent = c.getParent();

		while (parent != null) {
			if (parent instanceof ScrolledComposite) {
				return (ScrolledComposite) parent;
			}
			parent = parent.getParent();
		}
		return null;
	}

	public static void processKey(int keyCode, Control c) {
		ScrolledComposite scomp = getScrolledComposite(c);
		if (scomp != null) {
			switch (keyCode) {
				case SWT.ARROW_DOWN :
					AbstractSectionForm.scrollVertical(scomp, false);
					break;
				case SWT.ARROW_UP :
					AbstractSectionForm.scrollVertical(scomp, true);
					break;
				case SWT.ARROW_LEFT :
					AbstractSectionForm.scrollHorizontal(scomp, true);
					break;
				case SWT.ARROW_RIGHT :
					AbstractSectionForm.scrollHorizontal(scomp, false);
					break;
				case SWT.PAGE_UP :
					AbstractSectionForm.scrollPage(scomp, true);
					break;
				case SWT.PAGE_DOWN :
					AbstractSectionForm.scrollPage(scomp, false);
					break;
			}
		}
	}

	public Button createButton(Composite parent, String text, int style) {

		Button button = new Button(parent, style | SWT.FLAT);
		button.setBackground(backgroundColor);
		button.setForeground(foregroundColor);
		
		if (text != null){
			button.setText(text);
		}
		
		return button;
	}
	public Composite createComposite(Composite parent) {
		return createComposite(parent, SWT.NULL);
	}
	public Composite createComposite(Composite parent, int style) {
		Composite composite = new Composite(parent, style);
		composite.setBackground(backgroundColor);
		composite.addMouseListener(new MouseAdapter() {
			public void mouseDown(MouseEvent e) {
				((Control) e.widget).setFocus();
			}
		});
		composite.setMenu(parent.getMenu());
		return composite;
	}
	public Composite createCompositeSeparator(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setBackground(getColor(COLOR_COMPOSITE_SEPARATOR));
		return composite;
	}

	public Label createHeadingLabel(Composite parent, String text) {
		return createHeadingLabel(parent, text, null, SWT.NONE);
	}

	public Label createHeadingLabel(Composite parent, String text, int style) {
		return createHeadingLabel(parent, text, null, style);
	}

	public Label createHeadingLabel(Composite parent, String text, Color bg) {
		return createHeadingLabel(parent, text, bg, SWT.NONE);
	}

	public Label createHeadingLabel(
		Composite parent,
		String text,
		Color bg,
		int style) {
		Label label = new Label(parent, style);
		if (text != null)
			label.setText(text);
		label.setBackground(backgroundColor);
		label.setForeground(foregroundColor);
		label.setFont(JFaceResources.getFontRegistry().get(JFaceResources.BANNER_FONT));
		return label;
	}

	public Label createHyperlinkLabel(
		Composite parent,
		String text,
		IHyperlinkListener listener) {
		return createHyperlinkLabel(parent, text, listener, SWT.NULL);
	}
	public Label createHyperlinkLabel(
		Composite parent,
		String text,
		IHyperlinkListener listener,
		int style) {
		Label label = createLabel(parent, text, style);
		turnIntoHyperlink(label, listener);
		return label;
	}
	public Label createLabel(Composite parent, String text) {
		return createLabel(parent, text, SWT.NONE);
	}
	public Label createLabel(Composite parent, String text, int style) {
		Label label = new Label(parent, style);
		if (text != null)
			label.setText(text);
		label.setBackground(backgroundColor);
		label.setForeground(foregroundColor);
		return label;
	}

	public SelectableFormLabel createSelectableLabel(
		Composite parent,
		String text) {
		return createSelectableLabel(parent, text, SWT.NONE);
	}

	public SelectableFormLabel createSelectableLabel(
		Composite parent,
		String text,
		int style) {
		SelectableFormLabel label = new SelectableFormLabel(parent, style);
		if (text != null)
			label.setText(text);
		label.setBackground(backgroundColor);
		label.setForeground(foregroundColor);
		label.addKeyListener(keyboardHandler);
		return label;
	}

	public FormEngine createFormEngine(Composite parent) {
		FormEngine engine = new FormEngine(parent, SWT.WRAP);
		engine.setBackground(backgroundColor);
		engine.setForeground(foregroundColor);
		engine.marginWidth = 1;
		engine.marginHeight = 0;
		engine.setHyperlinkSettings(getHyperlinkHandler());
		engine.addKeyListener(keyboardHandler);
		engine.setMenu(parent.getMenu());
		return engine;
	}

	public Label createSeparator(Composite parent, int style) {
		Label label = new Label(parent, SWT.SEPARATOR | style);
		label.setBackground(backgroundColor);
		label.setForeground(borderColor);
		return label;
	}
	public Table createTable(Composite parent, int style) {
		Table table = new Table(parent, BORDER_STYLE | style);
		table.setBackground(backgroundColor);
		table.setForeground(foregroundColor);
		hookDeleteListener(table);
		return table;
	}
	public Text createText(Composite parent, String value) {
		return createText(parent, value, BORDER_STYLE | SWT.SINGLE);
	}
	public Text createText(Composite parent, String value, int style) {
		
		//Note: The background and foreground colors are not set so the text field can be disabled ('grayed') and they are the same colors.
		Text text = new Text(parent, style);
		
		if (value != null){
			text.setText(value);
		}
		
		return text;
	}
	public Tree createTree(Composite parent, int style) {
		Tree tree = new Tree(parent, BORDER_STYLE | style);
		tree.setBackground(backgroundColor);
		tree.setForeground(foregroundColor);
		hookDeleteListener(tree);
		return tree;
	}
	protected void deleteKeyPressed(Widget widget) {
		if (!(widget instanceof Control))
			return;
		Control control = (Control) widget;
		for (Control parent = control.getParent();
			parent != null;
			parent = parent.getParent()) {
			if (parent.getData() instanceof FormSection) {
				FormSection section = (FormSection) parent.getData();
				section.doGlobalAction(ActionFactory.DELETE.getId());
				break;
			}
		}
	}
	public void dispose() {

		Enumeration<Color> colors = colorRegistry.elements();
		
		while (colors.hasMoreElements()) {
			colors.nextElement().dispose();
		}
		
		hyperlinkHandler.dispose();
		
		colorRegistry = null;
	}
	public Color getBackgroundColor() {
		return backgroundColor;
	}
	public Color getBorderColor() {
		return borderColor;
	}
	public Cursor getBusyCursor() {
		return hyperlinkHandler.getBusyCursor();
	}
	public Color getColor(String key) {
		return colorRegistry.get(key);
	}
	public Color getForegroundColor() {
		return foregroundColor;
	}
	public HyperlinkHandler getHyperlinkHandler() {
		return hyperlinkHandler;
	}
	public Cursor getHyperlinkCursor() {
		return hyperlinkHandler.getHyperlinkCursor();
	}
	public Color getHyperlinkColor() {
		return hyperlinkHandler.getForeground();
	}
	public Color getHyperlinkHoverColor() {
		return hyperlinkHandler.getActiveForeground();
	}
	public int getHyperlinkUnderlineMode() {
		return hyperlinkHandler.getHyperlinkUnderlineMode();
	}
	public void hookDeleteListener(Control control) {
		if (deleteListener == null) {
			deleteListener = new KeyAdapter() {
				public void keyPressed(KeyEvent event) {
					if (event.character == SWT.DEL && event.stateMask == 0) {
						deleteKeyPressed(event.widget);
					}
				}
			};
		}
		control.addKeyListener(deleteListener);
	}
	private void initialize() {
		backgroundColor = display.getSystemColor(SWT.COLOR_LIST_BACKGROUND);
		registerColor(COLOR_BORDER, 195, 191, 179);
		registerColor(COLOR_COMPOSITE_SEPARATOR, 152, 170, 203);
		registerColor(DEFAULT_HEADER_COLOR, 0x48, 0x70, 0x98);
		if (isWhiteBackground())
			borderColor = getColor(COLOR_BORDER);
		else
			borderColor = display.getSystemColor(SWT.COLOR_WIDGET_BACKGROUND);
		foregroundColor = display.getSystemColor(SWT.COLOR_LIST_FOREGROUND);
		hyperlinkHandler = new HyperlinkHandler();
		hyperlinkHandler.setBackground(backgroundColor);
		updateHyperlinkColors();
		keyboardHandler = new KeyboardHandler();
	}
	
	public boolean isWhiteBackground() {
		return backgroundColor.getRed()==255 && backgroundColor.getGreen()==255 &&
			backgroundColor.getBlue()==255;
	}

	public KeyboardHandler getKeyboardHandler() {
		return keyboardHandler;
	}

	public void updateHyperlinkColors() {
		Color hyperlinkColor = JFaceColors.getHyperlinkText(display);
		Color activeHyperlinkColor = JFaceColors.getActiveHyperlinkText(display);
		hyperlinkHandler.setForeground(hyperlinkColor);
		hyperlinkHandler.setActiveForeground(activeHyperlinkColor);
	}

	public void paintBordersFor(Composite parent) {

		if (borderPainter == null){
			borderPainter = new BorderPainter();
		}
		
		parent.addPaintListener(borderPainter);
	}
	public Color registerColor(String key, int r, int g, int b) {
		Color c = new Color(display, r, g, b);
		colorRegistry.put(key, c);
		return c;
	}
	public void setHyperlinkColor(Color color) {
		hyperlinkHandler.setForeground(color);
	}
	public void setHyperlinkHoverColor(org.eclipse.swt.graphics.Color hoverColor) {
		hyperlinkHandler.setActiveForeground(hoverColor);
	}
	public void setHyperlinkUnderlineMode(int newHyperlinkUnderlineMode) {
		hyperlinkHandler.setHyperlinkUnderlineMode(newHyperlinkUnderlineMode);
	}
	public void turnIntoHyperlink(Control control, IHyperlinkListener listener) {
		hyperlinkHandler.registerHyperlink(control, listener);
	}
}
