/********************************************************************** 
 * Copyright (c) 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: Partition.java,v 1.1 2009/05/15 18:58:26 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.internal.editor.form.base;

import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.swt.graphics.Image;

/**
 * <p>Partition.java</p>
 * 
 * 
 * @author  Blazej Kroll
 * @author  Paul Slauenwhite
 * @version May 15, 2009
 * @since   May 15, 2009
 */
public class Partition{

	/**
	 * Offset of the partition.
	 */
	private int offset;
	
	/**
	 * Size of the partition (for example, number of elements).
	 */
	private int size;
	
	/**
	 * Children for the partition.
	 */
	private EList children = null;
	
	/**
	 * Label for the partition (for example, <code>[0..4]</code>).
	 */
	private String label = null;
	
	/**
	 * Creates a new partition.
	 * 
	 * @param offset Offset of the partition.
	 * @param size Size of the partition (for example, number of elements).
	 * @param children Children for the partition.
	 */
	public Partition(int offset, int size, EList children) {
		
		this.offset = offset;
		this.size = size;
		this.children = children; 
		this.label = getLabel();
	}
	
	/**
	 * Resolves the offset of the partition.
	 * 
	 * @return The offset of the partition.
	 */
	public int getOffset() {
		return offset;
	}

	/**
	 * Resolves the size of the partition (for example, number of elements).
	 * 
	 * @return The size of the partition (for example, number of elements).
	 */
	public int getSize() {
		return size;
	}
	
	/**
	 * Resolves the children for the partition from <code>{@link #getOffset()}...{@link #getOffset()} + {@link #getSize()}</code>.
	 * 
	 * @return The children for the partition from <code>{@link #getOffset()}...{@link #getOffset()} + {@link #getSize()}</code>.
	 */
	public List getChildren() {
		return (children.subList(offset, offset + size));
	}

	/**
	 * Resolves the child for the partition at <code>{@link #getOffset()} + index</code>.
	 * 
	 * @param index The index of the child. 
	 * @return The child for the partition at <code>{@link #getOffset()} + index</code>.
	 */
	public Object getChild(int index) {
		return (children.get(offset + index));
	}

	/**
	 * Resolves the label for the partition (for example, <code>[{@link #getOffset()}..{@link #getOffset()} + {@link #getSize()} - 1]</code> or <code>Loading...</code>).
	 * 
	 * @return The label for the partition (for example, <code>[{@link #getOffset()}..{@link #getOffset()} + {@link #getSize()} - 1]</code> or <code>Loading...</code>).
	 * @see    UiPluginResourceBundle.PARTITION_LOADING_LABEL
	 */
	public String getLabel() {
		
		if(label == null){

			if ((offset > -1) && (size > -1)){

				StringBuffer buffer = new StringBuffer();
				buffer.append("["); //$NON-NLS-1$
				buffer.append(offset);
				buffer.append("..."); //$NON-NLS-1$
				buffer.append(offset + size - 1);
				buffer.append("]");  //$NON-NLS-1$

				label = buffer.toString();
			}
			else{
				label = UiPluginResourceBundle.PARTITION_LOADING_LABEL;			
			}
		}		

		return label;
	}

	/**
	 * Resolves the image for the partition.
	 * 
	 * @return The image for the partition.
	 * @see    TestUIImages.IMG_EVENT_PARTITION
	 */
	public Image getImage() {
		
		if ((offset > -1) && (size > -1)){
			return (TestUIImages.INSTANCE.getImage(TestUIImages.IMG_EVENT_PARTITION));
		}

		return null;
	}
}
