/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SharedPartWithButtons.java,v 1.4 2005/08/19 21:14:31 hleung Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form.base;

import java.util.LinkedList;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.forms.widgets.FormToolkit;

/**
 * @author marcelop
 * @since 0.2.0
 */
public abstract class SharedPartWithButtons 
extends SharedPart
{
	private String[] buttonLabels;
	private Control[] controls;
    private String[] toolTips;
	private Composite buttonContainer;
	
	private int buttonLength;
	private final int standardButtonLength;
	
	/* To make button insertion cheap, we keep track of index list using a linked list */
	private LinkedList indexList;

	private class SelectionHandler implements SelectionListener
	{
		public void widgetSelected(SelectionEvent e)
		{
			buttonSelected(e);
		}
		public void widgetDefaultSelected(SelectionEvent e)
		{
			buttonSelected(e);
		}
		private void buttonSelected(SelectionEvent e)
		{
			Integer index = (Integer)e.widget.getData();
			SharedPartWithButtons.this.buttonSelected((Button)e.widget, index.intValue());
		}
	}

	public SharedPartWithButtons(String[] buttonLabels)
	{
		/* Add 5 additional spaces in case the client adds more buttons */
		this.buttonLabels = new String[buttonLabels.length + 5];
		buttonLength = buttonLabels.length;
		standardButtonLength = buttonLength;
		System.arraycopy(buttonLabels,0,this.buttonLabels,0,buttonLength);
		
		/* Update the index list */
		indexList = new LinkedList();		
		for (int i = 0; i < buttonLength; i++)
		{
			indexList.add(new Integer(i));
		}
	}

    public SharedPartWithButtons(String[] buttonLabels,String[] toolTips)
    {
    	/* Add 5 additional spaces in case the client adds more buttons */
    	this (buttonLabels);
    	
    	this.toolTips = new String[buttonLength + 5];
		if (toolTips != null)
		{
			System.arraycopy(toolTips,0,this.toolTips,0,buttonLength);
		}		
    }

	public void setButtonEnabled(int index, boolean enabled)
	{
		Button b = getButton (index);
		if (b != null)
			b.setEnabled(enabled);
	}
	
	public String getButtonLabel(int index)
	{
		int inx = ((Integer)indexList.get(index)).intValue();
		if (inx >= 0 && inx < buttonLabels.length)
			return buttonLabels[index];
		return null;
	}
	
	public Menu getButtonMenu(int index)
	{
		Button button = getButton(index);
		if (button == null)
			return null;
		if(button.getMenu() == null)
			button.setMenu(new Menu(button));
		return button.getMenu();
	}
	
	public Button getButton (int index)
	{
		int inx = ((Integer)indexList.get(index)).intValue();
		if (controls != null && inx >= 0 && controls.length > inx)
		{
			Control c = controls[inx];
			if (c instanceof Button)
				return (Button)c;
		}
		return null;
	}

	protected abstract void createMainControl(Composite parent, int style, int span, FormWidgetFactory factory);
	
	protected abstract void createMainControl(Composite parent, int style, int span, FormToolkit factory);
	protected abstract void buttonSelected(Button button, int index);


	public void createControl(Composite parent, int style, int span, FormWidgetFactory factory)
	{
		createControl(parent, style, span, (Object) factory);
	}

	
	public void createControl(Composite parent, int style, int span, FormToolkit toolkit)
	{
		createControl(parent, style, span, (Object) toolkit);
	}
	
	
	/**
	 * A helper method created here to avoid repeated statements.  This method will
	 * need to be taken off and its content transfered to 
	 * createControl(Composite parent, int style, int span, FormToolkit toolkit) once 
	 * createControl(Composite parent, int style, int span, FormWidgetFactory toolkit) is 
	 * taken off
	 */
	
	private void createControl(Composite parent, int style, int span, Object toolkit)
	{
		boolean isToolkit = false;
		if (toolkit instanceof FormToolkit)
			isToolkit = true;
		
		if (isToolkit)
		{
			createMainLabel(parent, span, (FormToolkit)toolkit);
			createMainControl(parent, style, span - 1, (FormToolkit) toolkit);
		}
		else
		{
			createMainLabel(parent, span, (FormWidgetFactory)toolkit);
			createMainControl(parent, style, span - 1, (FormWidgetFactory) toolkit);
		}

		if (buttonLabels != null && buttonLength > 0)
		{
			buttonContainer = (isToolkit ? createComposite(parent, (FormToolkit)toolkit) : createComposite(parent, (FormWidgetFactory)toolkit));
			GridData gd = new GridData(GridData.FILL_VERTICAL);
			buttonContainer.setLayoutData(gd);
			buttonContainer.setLayout(createButtonsLayout());

			controls = new Control[buttonLength];
			SelectionHandler listener = new SelectionHandler();
			
			Object[] indexListArray = indexList.toArray();
			for (int i = 0; i < indexListArray.length; i++)
			{
				int inx = ((Integer)indexListArray[i]).intValue();
				String label = buttonLabels[inx];
				if (label != null)
				{
					Button button = (isToolkit ? createButton(buttonContainer, label, inx, (FormToolkit)toolkit) : createButton(buttonContainer, label, inx, (FormWidgetFactory)toolkit));
					button.addSelectionListener(listener);
                    if ((toolTips!=null) && (toolTips.length > i)) {
                        button.setToolTipText(toolTips[inx]);
                    }

					controls[inx] = button;
				}
				else
				{
					if (isToolkit)
						createEmptySpace(buttonContainer, 1, (FormToolkit)toolkit);
					else
						createEmptySpace(buttonContainer, 1, (FormWidgetFactory)toolkit);
				}
			}
		}
	}
	
	
	/**
	 * This method gives the user the ability to add buttons to the Test
	 * Case form at a particular index.
	 * <br/>
	 * Pre-conditions:
	 * <ul>
	 * 	<li> index must be greater or equalled to zero.  If the index is less than zero,
	 *  then it is set to zero and if it's greater than the total number of buttons, then
	 *  it is set to the last entry. </li>
	 * </ul>
	 * 
	 * @param buttonLable The label of the button
	 * @param buttonToolTip The button's tooltip (can be null)
	 * @param index The index at which it will be inserted.
	 */
	public void addButton (String buttonLabel, String buttonToolTip, int index)
	{
		buttonLength++;
		
		boolean isButtonToolTipRequired = this.toolTips != null || buttonToolTip != null;
		
		/* We need to expand our button labels */
		if (buttonLength >=  buttonLabels.length)
		{
			String[] newButtonLabels = new String[buttonLength + 5];			
			System.arraycopy(this.buttonLabels, 0, newButtonLabels, 0, buttonLength - 1);
			this.buttonLabels = newButtonLabels;	
			
			if (isButtonToolTipRequired)
			{
				String[] newButtonToolTips = new String[buttonLength + 5];
				if (toolTips != null)
					System.arraycopy(this.toolTips, 0, newButtonToolTips, 0, buttonLength - 1);
				this.toolTips = newButtonToolTips;
			}								
		}
			
		if (isButtonToolTipRequired && this.toolTips == null)
			this.toolTips = new String[buttonLength + 5];
		
		this.buttonLabels[buttonLength - 1] = buttonLabel;
		this.toolTips[buttonLength - 1] = buttonToolTip;
		
		if (index < 0) index = 0;
		if (index > indexList.size()) index = indexList.size();
		
		indexList.add(index, new Integer(buttonLength - 1));
	}
	
	protected GridLayout createButtonsLayout()
	{
		GridLayout layout = new GridLayout();
		layout.marginWidth = layout.marginHeight = 0;
		return layout;
	}

	protected Button createButton(Composite parent, String label, int index, FormWidgetFactory factory)
	{
		Button button;

		if (factory != null)
			button = factory.createButton(parent, label, SWT.PUSH);
		else
		{
			button = new Button(parent, SWT.PUSH);
			button.setText(label);
		}
		GridData gd = new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING);
		button.setLayoutData(gd);
		button.setData(new Integer(index));
		return button;
	}
	
	
	protected Button createButton(Composite parent, String label, int index, FormToolkit factory)
	{
		Button button;

		if (factory != null)
			button = factory.createButton(parent, label, SWT.PUSH);
		else
		{
			button = new Button(parent, SWT.PUSH);
			button.setText(label);
		}
		GridData gd = new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING);
		button.setLayoutData(gd);
		button.setData(new Integer(index));
		return button;
	}

	protected void updateEnabledState()
	{
		for (int i = 0; i < controls.length; i++)
		{
			Control c = controls[i];
			if (c instanceof Button)
				c.setEnabled(isEnabled());
		}
	}

	protected void createMainLabel(Composite parent, int span, FormWidgetFactory factory)
	{
	}
	
	
	/**
	 * Needs to be over-written by clients 
	 */
	protected void createMainLabel(Composite parent, int span, FormToolkit factory)
	{
	}

	public int getButtonLength()
	{
		return buttonLength;
	}

	public int getStandardButtonLength()
	{
		return standardButtonLength;
	}
}
