/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DetailPropertiesSection.java,v 1.4 2007/04/26 18:50:09 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form.util;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.editor.IEditorExtension;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.part.PageBook;


/**
 * This class was created in response to the deprecated APIs about the way that
 * forms are currently handled.
 * 
 * <p>
 * Represents the detailed properties section
 * </p>
 * 
 * @author Ali Mehregani & marcelop
 */
public class DetailPropertiesSection implements IDetailPageContext
{
		/**
		 * An instance of this class is associated to a DetailSection instance,
		 * and is responsible of providing a DetailPage for each element that is
		 * set as an input of the DetailSection.
		 * @author jcanches
		 * @since 4.0
		 */
		public static interface IDetailFormProvider extends IDisposable {
			/**
			 * Returns a DetailPage for a specific element.
			 * @param element An element.
			 * @return A DetailPage for element, or <code>null</code> if this provider
			 * does not know the element type (in that case, the owning DetailSection
			 * displays a default DetailPage).
			 */
			IDetailForm getDetailPage(Object element);
			/**
			 * Invoked when the owning DetailSection is disposed. This method should
			 * dispose any DetailPage that was created by the provider.
			 */
			void dispose();
		}

		private PageBook pageBook;
		private IDetailFormProvider detailPageProvider;
		private Map controlByPage;
		private Composite defaultControl;
		private int maxHeight = 0;
		
		/* The editor that the form of this section is apart of */
		private IEditorExtension editor;
		
		/* The form that this section is apart of */
		private EditorEclipseForm form;
		
		private Object input;
		private FormToolkit toolkit;
		
		
		/**
		 * Constructor for DetailSection
		 * @param editorForm
		 */
		public DetailPropertiesSection(IEditorExtension editor, EditorEclipseForm form, FormToolkit toolkit)
		{
			controlByPage = new HashMap();
			this.editor = editor;
			this.form = form;
			this.toolkit = toolkit;
		}

		/**
		 * 
		 */
		public void dispose()
		{
			input = null;
			controlByPage.clear();
			detailPageProvider.dispose();
		}
		
		public void setDetailPageProvider(IDetailFormProvider detailPageProvider) 
		{
			this.detailPageProvider = detailPageProvider;
		}
		
		/**
		 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.FormSection#createClient(org.eclipse.swt.widgets.Composite, org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory)
		 */
		public Composite createDefaultControl(Composite parent)
		{		
			Composite composite = toolkit.createComposite(parent);
			GridLayout gridLayout = new GridLayout();
			gridLayout.marginHeight = 0;
			gridLayout.marginWidth = 0;
			composite.setLayout(gridLayout);		
			composite.setLayoutData(GridDataUtil.createFill());

			pageBook = new PageBook(composite, SWT.NONE);
			pageBook.setLayoutData(GridDataUtil.createFill());
			pageBook.setBackground(parent.getBackground());
			pageBook.setMenu(parent.getMenu());

			defaultControl = toolkit.createComposite(pageBook);
			defaultControl.setLayout(new GridLayout());
			defaultControl.setLayoutData(GridDataUtil.createFill());
			
			Label label = toolkit.createLabel(defaultControl, UiPluginResourceBundle.NO_DETAILS); 
			GridData gridData = GridDataUtil.createHorizontalFill();
			gridData.verticalAlignment = GridData.VERTICAL_ALIGN_BEGINNING;
			label.setLayoutData(gridData);
					
			return composite;
		}
		
		
		/**
		 * 
		 */
		public void setInput(Object object)
		{
			input = object;
			IDetailForm detailPage = getDetailPage(object);
			if(detailPage == null)
			{
				pageBook.showPage(defaultControl);
			}
			else
			{	
				detailPage.setInput(input);
				pageBook.showPage((Control)controlByPage.get(detailPage));
			}
		}

		/**
		 *
		 */
		public Object getInput()
		{
			return input;
		}


		/**
		 * 
		 */
		public IStructuredSelection getStructuredSelection()
		{
			IDetailForm detailPage = getDetailPage(input);
			if(detailPage == null)
				return StructuredSelection.EMPTY;
			return detailPage.getStructuredSelection();
		}

		/**
		 * @see org.eclipse.ui.part.ISetSelectionTarget#selectReveal(org.eclipse.jface.viewers.ISelection)
		 */
		public void selectReveal(ISelection selection)
		{
			IDetailForm detailPage = getDetailPage(input);
			if(detailPage != null)
				detailPage.selectReveal(selection);
		}
		
		protected IDetailForm getDetailPage(Object input)
		{
			if(input != null)
			{
				if(input instanceof IStructuredSelection)
					input = ((IStructuredSelection)input).getFirstElement();
				
	            if (input != null) {
	    			IDetailForm detailPage = detailPageProvider.getDetailPage(input);
	    			if (detailPage != null) {
	    				if (!controlByPage.containsKey(detailPage)) {
	    					// First time we get this detail page: create the control
	    					Control control = detailPage.createControl(pageBook, toolkit, this);
	    					control.setLayoutData(GridDataUtil.createFill());
	    					controlByPage.put(detailPage, control);
	    					
	    					int height = control.computeSize(SWT.DEFAULT, SWT.DEFAULT, true).y;
	    					if(height>maxHeight) {
	    						maxHeight = height;
	    						((GridData)pageBook.getLayoutData()).heightHint = maxHeight;
	    						// If the form has its control already set, re-layout the form's composite
	    						Control formControl = getEditorForm().getPartControl();
	    						if (formControl != null && formControl instanceof Composite) {
	    							((Composite)formControl).layout(new Control[] {control});
	    						}
	    					}
	    				}
	    			}
	                return detailPage;
	            }
			}		
			return null;
		}

		public EditorEclipseForm getEditorForm() 
		{
			return form;
		}

		public boolean isReadOnly() 
		{
			return editor.getHyadesEditorPart().isReadOnly();
		}

		public void verifyKey(VerifyEvent event) 
		{
			event.doit = !editor.getHyadesEditorPart().isReadOnly();
		}
}


