/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DetailSection.java,v 1.10 2007/05/03 17:27:42 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form.util;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.EditorSection;
import org.eclipse.hyades.test.ui.editor.form.util.IDetailPage;
import org.eclipse.hyades.test.ui.editor.form.util.IDetailPageContext;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.part.PageBook;

/**
 * @author marcelop
 * @since 0.3.0
 */
public class DetailSection extends EditorSection implements IDetailPageContext
{
	/**
	 * An instance of this class is associated to a DetailSection instance,
	 * and is responsible of providing a DetailPage for each element that is
	 * set as an input of the DetailSection.
	 * @author jcanches
	 * @since 4.0
	 */
	public static interface IDetailPageProvider extends IDisposable {
		/**
		 * Returns a DetailPage for a specific element.
		 * @param element An element.
		 * @return A DetailPage for element, or <code>null</code> if this provider
		 * does not know the element type (in that case, the owning DetailSection
		 * displays a default DetailPage).
		 */
		IDetailPage getDetailPage(Object element);
		/**
		 * Invoked when the owning DetailSection is disposed. This method should
		 * dispose any DetailPage that was created by the provider.
		 */
		void dispose();
	}

	private PageBook pageBook;
	private IDetailPageProvider detailPageProvider;
	private Map controlByPage;
	private Composite defaultControl;
	private FormWidgetFactory widgetFactory;
	private int maxHeight = 0;
	
	private Object input;
	
	/**
	 * Constructor for DetailSection
	 * @param editorForm
	 */
	public DetailSection(EditorForm editorForm)
	{
		super(editorForm);
		setCollapsable(true);
		controlByPage = new HashMap();
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.FormSection#dispose()
	 */
	public void dispose()
	{
		input = null;
		controlByPage.clear();
		detailPageProvider.dispose();
		super.dispose();
	}
	
	public void setDetailPageProvider(IDetailPageProvider detailPageProvider) {
		this.detailPageProvider = detailPageProvider;
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.FormSection#createClient(org.eclipse.swt.widgets.Composite, org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory)
	 */
	public Composite createClient(Composite parent, FormWidgetFactory factory)
	{
		widgetFactory = factory;
		Composite composite = widgetFactory.createComposite(parent);
		GridLayout gridLayout = new GridLayout();
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		composite.setLayout(gridLayout);		
		composite.setLayoutData(GridDataUtil.createFill());

		pageBook = new PageBook(composite, SWT.NONE);
		pageBook.setLayoutData(GridDataUtil.createFill());
		pageBook.setBackground(widgetFactory.getBackgroundColor());
		pageBook.addMouseListener(new MouseAdapter()
		{
			public void mouseDown(MouseEvent e)
			{
				((Control) e.widget).setFocus();
			}
		});
		pageBook.setMenu(parent.getMenu());

		defaultControl = widgetFactory.createComposite(pageBook);
		defaultControl.setLayout(new GridLayout());
		defaultControl.setLayoutData(GridDataUtil.createFill());
		
		Label label = widgetFactory.createLabel(defaultControl, UiPluginResourceBundle.NO_DETAILS); 
		GridData gridData = GridDataUtil.createHorizontalFill();
		gridData.verticalAlignment = GridData.VERTICAL_ALIGN_BEGINNING;
		label.setLayoutData(gridData);
				
		return composite;
	}

	public Composite getClient()
	{
		return pageBook;
	}
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorSection#setInput(java.lang.Object)
	 */
	public void setInput(Object object)
	{
		input = object;
		IDetailPage detailPage = getDetailPage(object);
		if(detailPage == null)
		{
			pageBook.showPage(defaultControl);
		}
		else
		{	
			detailPage.setInput(input);
			pageBook.showPage((Control)controlByPage.get(detailPage));
		}
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorSection#getInput()
	 */
	public Object getInput()
	{
		return input;
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorSection#getStructuredSelection()
	 */
	public IStructuredSelection getStructuredSelection()
	{
		IDetailPage detailPage = getDetailPage(input);
		if(detailPage == null)
			return StructuredSelection.EMPTY;
		return detailPage.getStructuredSelection();
	}

	/**
	 * @see org.eclipse.ui.part.ISetSelectionTarget#selectReveal(org.eclipse.jface.viewers.ISelection)
	 */
	public void selectReveal(ISelection selection)
	{
		IDetailPage detailPage = getDetailPage(input);
		if(detailPage != null)
			detailPage.selectReveal(selection);
	}
	
	protected IDetailPage getDetailPage(Object input)
	{
		if(input != null)
		{
			if(input instanceof IStructuredSelection)
				input = ((IStructuredSelection)input).getFirstElement();
			
            if (input != null) {
    			IDetailPage detailPage = detailPageProvider.getDetailPage(input);
    			if (detailPage != null) {
    				if (!controlByPage.containsKey(detailPage)) {
    					// First time we get this detail page: create the control
    					Control control = detailPage.createControl(pageBook, widgetFactory, this);
    					control.setLayoutData(GridDataUtil.createFill());
    					controlByPage.put(detailPage, control);
    					
    					int height = control.computeSize(SWT.DEFAULT, SWT.DEFAULT, true).y;
    					if(height>maxHeight) {
    						maxHeight = height;
    						((GridData)pageBook.getLayoutData()).heightHint = maxHeight;
    						// If the form has its control already set, re-layout the form's composite
    						Control formControl = getEditorForm().getControl();
    						if (formControl != null && formControl instanceof Composite) {
    							((Composite)formControl).layout(new Control[] {control});
    						}
    					}
    				}
    			}
                return detailPage;
            }
		}		
		return null;
	}
	
	public void addControlWithDetailPage(IDetailPage detailPage, Control control)
	{
		controlByPage.put(detailPage, control);
	}
}
