/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionHistoryExtensionsManager.java,v 1.13 2009/05/17 16:16:08 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.editor.form.util.IDetailPageFactory;
import org.eclipse.hyades.test.ui.editor.form.util.IEventLabelProvider;
import org.eclipse.hyades.test.ui.forms.extensions.IEventAction;
import org.eclipse.hyades.test.ui.forms.extensions.IPropertyLabelProvider;
import org.eclipse.hyades.test.ui.forms.extensions.provisional.IVerdictCategoryProvider;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.forms.IDetailsPage;

import com.ibm.icu.util.StringTokenizer;

/**
 * <p>This singleton class handles reading and loading of extensions to
 * the extension point <code>org.eclipse.hyades.test.ui.executionHistoryExtensions</code>.</p>
 * 
 * 
 * @author  Julien Canches
 * @author  Bianca Xue Jiang
 * @author  Paul Slauenwhite
 * @version May 16, 2009
 * @since   March 16, 2005
 */
public class ExecutionHistoryExtensionsManager {

	public final static String ALL_TEST_TYPES = "allTestTypes";
	
	protected final static String EXTENSION_ELEMENT_EVENT_EXTENSION = "eventExtension";
	protected final static String EXTENSION_ELEMENT_ACTION_EXTENSION = "actionExtension";
	protected final static String EXTENSION_ELEMENT_ACTION = "action";
	protected final static String EXTENSION_ELEMENT_PROPERTY = "property";
	protected final static String EXTENSION_ELEMENT_VERDICT_PROVIDER= "verdictProvider";

	protected final static String EXTENSION_ATTRIBUTE_TEST_TYPE = "testType";
	protected final static String EXTENSION_ATTRIBUTE_EVENT_TYPE = "eventType";
	protected final static String EXTENSION_ATTRIBUTE_EVENT_LABEL_PROVIDER = "labelProvider";
	protected final static String EXTENSION_ATTRIBUTE_EVENT_DETAILS_PAGE = "detailsPage";
	protected final static String EXTENSION_ATTRIBUTE_ACTION = "action";
	protected final static String EXTENSION_ATTRIBUTE_ACTION_TEXT = "text";
	protected final static String EXTENSION_ATTRIBUTE_ACTION_ICON = "icon";
	protected final static String EXTENSION_ATTRIBUTE_ACTION_ASBUTTON = "asButton";
	protected final static String EXTENSION_ATTRIBUTE_ACTION_ASMENU = "asContextMenu";
	protected final static String EXTENSION_ATTRIBUTE_PROP_NAME = "name";
	protected final static String EXTENSION_ATTRIBUTE_PROP_VISIBLE = "visible";
	protected final static String EXTENSION_ATTRIBUTE_PROP_UNLESS = "unless";
	protected final static String EXTENSION_ATTRIBUTE_PROP_UNIT = "unit";
	protected final static String EXTENSION_ATTRIBUTE_PROVIDER = "provider";
	protected final static String EXTENSION_ATTRIBUTE_PROP_PROVIDER = "propertyProvider";
	protected final static String TEST_TYPE_DELIMINATOR = ";";

	private static ExecutionHistoryExtensionsManager instance;
			
	//private Extension[] actionExtensions;
	/**
	 * Map of test type (String) to array of Extension objects.
	 */
	private Hashtable testTypeToEventActionsMap;
	
	private Hashtable eventTypeToEventExtensionMap;
	
	private Hashtable testTypeToVerdictProviderMap;
	
	public static ExecutionHistoryExtensionsManager getInstance() {
		if (instance == null) {
			instance = new ExecutionHistoryExtensionsManager();
		}
		return instance;
	}
	
	private ExecutionHistoryExtensionsManager() {
		loadFromRegistry();
	}
	
	private class EventExtension {
		private List configElements;
		private String eventType;
		private boolean isLableProviderCalculated = false;
		private IEventLabelProvider labelProvider;
		private Hashtable eobjectToPageMap;
		private Hashtable properties;
		private IPropertyLabelProvider propProvider;
		private boolean isPropProviderCalculated = false;
		
		public EventExtension(String eventType, IConfigurationElement configElement) {
			if(!EXTENSION_ELEMENT_EVENT_EXTENSION.equals(configElement.getName()) ||
				!eventType.equals(configElement.getAttribute(EXTENSION_ATTRIBUTE_EVENT_TYPE)))
				throw new IllegalArgumentException();
			
			this.eventType = eventType;
			this.configElements = new ArrayList();

			configElements.add(configElement);
			eobjectToPageMap = new Hashtable();
		}
		
		public void addExtension(IConfigurationElement configElement)
		{
			if(!EXTENSION_ELEMENT_EVENT_EXTENSION.equals(configElement.getName()) ||
				!eventType.equals(configElement.getAttribute(EXTENSION_ATTRIBUTE_EVENT_TYPE)))
				throw new IllegalArgumentException();
			
			this.configElements.add(configElement);
		}
		
		/**
		 * When there are more than one label provider registered for the same event type, 
		 * only one will be returned at random since the order of extensions are not garanteed.
		 */
		public IEventLabelProvider getLabelProvider() {
			if(!isLableProviderCalculated)
			{
				try {
					for(int i = 0; i < configElements.size(); i++)
					{
						IConfigurationElement configElement = (IConfigurationElement)configElements.get(i);
						String provider = configElement.getAttribute(EXTENSION_ATTRIBUTE_EVENT_LABEL_PROVIDER);
						if(provider != null)
						{
							labelProvider = (IEventLabelProvider)configElement.createExecutableExtension(EXTENSION_ATTRIBUTE_EVENT_LABEL_PROVIDER);
							break;
						}
					}
				} catch (Throwable t) {
					UiPlugin.logError(t);
				}
				isLableProviderCalculated = true;
			}
			return labelProvider;
		}
		
		/**
		 * When there are more than one details page registered for the same event type, 
		 * only one will be returned at random since the order of registered extensions are not garanteed.
		 */
		public IDetailsPage getDetailsPage(EObject eObject) {
			IDetailsPage page = (IDetailsPage)eobjectToPageMap.get(eObject);
			if(page == null)
			{
				try {
					for(int i = 0; i < configElements.size(); i++)
					{
						IConfigurationElement configElement = (IConfigurationElement)configElements.get(i);						
						String pageClass = configElement.getAttribute(EXTENSION_ATTRIBUTE_EVENT_DETAILS_PAGE);
						if(pageClass != null)
						{
							page = (IDetailsPage)configElement.createExecutableExtension(EXTENSION_ATTRIBUTE_EVENT_DETAILS_PAGE);
							eobjectToPageMap.put(eObject, page);
							break;
						}
					}
				} catch (Throwable t) {
					UiPlugin.logError(t);
				}
			}
			return page;
		}
		
		/**
		 * When there are more than one extensions registered for properties with the same event type, 
		 * properties from all extensions will be searched for the specified propertyName.
		 * But if there are more than one properties with the same name, only a random one will be returned
		 * because the order of registered extensions are not garanteed.
		 */
		public Property getProperty(String propertyName)
		{
			if(propertyName == null)
				return null;
			
			// load all properties from all registered extensions.
			if(properties == null)
			{				
				properties = new Hashtable();
				for(int j = 0; j < configElements.size(); j++)
				{
					IConfigurationElement configElement = (IConfigurationElement)configElements.get(j);					
					IConfigurationElement[] children = configElement.getChildren(EXTENSION_ELEMENT_PROPERTY);
					for(int i = 0; i < children.length; i++)
					{
						String name = children[i].getAttribute(EXTENSION_ATTRIBUTE_PROP_NAME);
						if(name != null)
							properties.put(name, new Property(children[i], getPropertyProvider()));
					}
				}
			}
			
			return (Property)properties.get(propertyName);
		}
		
		/**
		 * When there are more than one property provider registered for the same event type, 
		 * only one will be returned at random since the order of extensions are not garanteed.
		 */
		public IPropertyLabelProvider getPropertyProvider()
		{
			if(!isPropProviderCalculated)
			{
				try {
					for(int i = 0; i < configElements.size(); i++)
					{
						IConfigurationElement configElement = (IConfigurationElement)configElements.get(i);						
						String provider = configElement.getAttribute(EXTENSION_ATTRIBUTE_PROP_PROVIDER);
						if(provider != null)
						{
							propProvider = (IPropertyLabelProvider)configElement.createExecutableExtension(EXTENSION_ATTRIBUTE_PROP_PROVIDER);
							break;
						}
					}
				} catch (Throwable t) {
					UiPlugin.logError(t);
				}
				isPropProviderCalculated = true;
			}
			return propProvider;
		}
	}
	
	public class Property
	{
		private IConfigurationElement configElement;
		private String name;
		private String isVisibleUnless;
		private String unit;
		private IPropertyLabelProvider provider;
		
		public Property(IConfigurationElement element, IPropertyLabelProvider propProvider)
		{
			if(!EXTENSION_ELEMENT_PROPERTY.equals(element.getName()))
				throw new IllegalArgumentException();
			
			this.configElement = element;
			this.provider = propProvider;
		}
		
		public String getName()
		{
			if(name == null)
				name = configElement.getAttribute(EXTENSION_ATTRIBUTE_PROP_NAME);
			return name;
		}
		
		public boolean isVisible()
		{
			return Boolean.valueOf(configElement.getAttribute(EXTENSION_ATTRIBUTE_PROP_VISIBLE)).booleanValue();
		}
		
		public String getVisibleUnless()
		{
			if(isVisibleUnless == null)
				isVisibleUnless = configElement.getAttribute(EXTENSION_ATTRIBUTE_PROP_UNLESS);
			return isVisibleUnless;
		}
		
		public String getUnit()
		{
			if(unit == null)
				unit = configElement.getAttribute(EXTENSION_ATTRIBUTE_PROP_UNIT);
			return unit;
		}
		
		public IPropertyLabelProvider getProvider()
		{
			return provider;
		}
	}
	
	public class EventAction
	{
		private IConfigurationElement configElement;
		private String actionText = null;
		private ImageDescriptor actionImage = null;
		private IEventAction action = null;
		
		public EventAction(IConfigurationElement configElement)
		{
			if(!EXTENSION_ELEMENT_ACTION.equals(configElement.getName()))
				throw new IllegalArgumentException();
		
			this.configElement = configElement;
		}
		
		public String getActionText()
		{
			if(actionText == null)
				actionText = configElement.getAttribute(EXTENSION_ATTRIBUTE_ACTION_TEXT);
			return actionText;
		}
		
		public ImageDescriptor getActionImage()
		{
			if(actionImage == null)
			{
				String imageKey = configElement.getAttribute(EXTENSION_ATTRIBUTE_ACTION_ICON);
				if(imageKey != null) {
					actionImage = UIUtil.getImageDescriptorFromPlugin(Platform.getBundle(configElement.getDeclaringExtension().getNamespace()), imageKey);
                    if(actionImage != null) {
                        UiPlugin.getInstance().getImageRegistry().put(imageKey, actionImage);
                    }
                }
			}
			return actionImage;
		}
		
		public boolean isActionButton()
		{
			return Boolean.valueOf(configElement.getAttribute(EXTENSION_ATTRIBUTE_ACTION_ASBUTTON)).booleanValue();
		}
		
		public boolean isActionContextMenu()
		{
			return Boolean.valueOf(configElement.getAttribute(EXTENSION_ATTRIBUTE_ACTION_ASMENU)).booleanValue();
		}
		
		public IEventAction getAction()
		{
			if(action == null)
			{
				String attr = configElement.getAttribute(EXTENSION_ATTRIBUTE_ACTION);
				if (attr != null && !"".equals(attr)) {
					try {
						action = (IEventAction) configElement.createExecutableExtension(EXTENSION_ATTRIBUTE_ACTION);
						action.setText(getActionText());
						//if(getActionImage() != null)
							//action.setImageDescriptor(getActionImage());
					} catch (Throwable t) {
						UiPlugin.logError(t);
					}
				}
			}
			return action;
		}
	}
	
	private void loadFromRegistry() {
		testTypeToEventActionsMap = new Hashtable();
		eventTypeToEventExtensionMap = new Hashtable();
		testTypeToVerdictProviderMap = new Hashtable();
		
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + ".executionHistoryExtension"); //$NON-NLS-1$
		if (extPoint != null) 
		{
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) 
			{
				IConfigurationElement element = members[i];
				if(EXTENSION_ELEMENT_EVENT_EXTENSION.equals(element.getName()))
				{
					String eventType = element.getAttribute(EXTENSION_ATTRIBUTE_EVENT_TYPE);
					EventExtension eventExt = (EventExtension)eventTypeToEventExtensionMap.get(eventType);
					if(eventExt == null)
						eventTypeToEventExtensionMap.put(eventType, new EventExtension(eventType, element));
					else
						eventExt.addExtension(element);
				}
				else if(EXTENSION_ELEMENT_ACTION_EXTENSION.equals(element.getName()))
				{
					String testTypes = element.getAttribute(EXTENSION_ATTRIBUTE_TEST_TYPE);
					EventAction[] extensions = null;
					if(testTypes != null)
					{
						StringTokenizer testType = new StringTokenizer(testTypes, TEST_TYPE_DELIMINATOR);
						while(testType.hasMoreTokens())
						{
							String token = testType.nextToken().trim();
							extensions = (EventAction[])testTypeToEventActionsMap.get(token);
							if(extensions == null)
								extensions = new EventAction[0];
							
							IConfigurationElement[] childrenElements = element.getChildren(EXTENSION_ELEMENT_ACTION);
							if(childrenElements != null)
							{
								EventAction[] temp = new EventAction[extensions.length + childrenElements.length];
								
								for(int j = 0; j < childrenElements.length; j++)
									temp[j] = new EventAction(childrenElements[j]);
								
								if(extensions.length > 0)
									System.arraycopy(extensions, 0, temp, childrenElements.length, extensions.length);
								
								extensions = temp;
							}
							
							testTypeToEventActionsMap.put(token, extensions);
						}
					}
				}
				else if(EXTENSION_ELEMENT_VERDICT_PROVIDER.equals(element.getName()))
				{
					String testType = element.getAttribute(EXTENSION_ATTRIBUTE_TEST_TYPE);
					if(testType != null && testType.trim().length() > 0)
					{
						testTypeToVerdictProviderMap.put(testType, element);
					}
				}
			}
		}	
	}
	
	/**
	 * Returns the <code>IEventLabelProvider</code> for the sepecified <i>eventType</i> 
	 * from all registered extensions regardless of the test type. 
	 * @param eventType
	 * @return
	 * @deprecated use {@link #getEventLabelProvider(String) } instead.
	 */
	public IEventLabelProvider getEventLabelFactory(String eventType) 
	{
		return getEventLabelProvider(eventType);
	}
	
	/**
	 * Returns the <code>IEventLabelProvider</code> for the sepecified <i>eventType</i>.
	 * 
	 * @param eventType
	 * @return
	 */
	public IEventLabelProvider getEventLabelProvider(String eventType) 
	{
		if(eventType == null)
			return null;
		EventExtension extension = (EventExtension)eventTypeToEventExtensionMap.get(eventType);
		return (extension == null) ? null : extension.getLabelProvider();
	}
	
	public IDetailsPage getEventDetailsPage(EObject eObject) 
	{
		String eventType = null;		
		if(eObject instanceof TPFExecutionEvent)
			eventType = ((TPFExecutionEvent)eObject).getEventType();
		else if(eObject instanceof TPFExecutionResult)
			eventType = ((TPFExecutionResult)eObject).getType();		
		if(eventType == null)
			return null;
		
		EventExtension extension = (EventExtension)eventTypeToEventExtensionMap.get(eventType);
		return (extension == null) ? null : extension.getDetailsPage(eObject);
	}
	
	public Property getEventProperty(String eventType, String propName)
	{
		if(eventType == null || propName == null)
			return null;
		EventExtension extension = (EventExtension)eventTypeToEventExtensionMap.get(eventType);
		if(extension != null)
			return extension.getProperty(propName);
		return null;
	}
	
	public EventAction[] getEventActions(String testType)
	{		
		if(testType == null || ALL_TEST_TYPES.equals(testType))
		{
			List actions = new ArrayList();
			Collection values = testTypeToEventActionsMap.values();
			for(Iterator it = values.iterator(); it.hasNext();)
			{
				actions.addAll(Arrays.asList((EventAction[])it.next()));
			}
			return (EventAction[])actions.toArray(new EventAction[actions.size()]);
		}
		
		return (EventAction[])testTypeToEventActionsMap.get(testType);
	}
	
	public IVerdictCategoryProvider getVerdictProvider(String testType)
	{
		if(testType != null)
		{
			Object provider = testTypeToVerdictProviderMap.get(testType);
			if(provider instanceof IConfigurationElement)
			{
				try {
					Object obj = ((IConfigurationElement)provider).createExecutableExtension(EXTENSION_ATTRIBUTE_PROVIDER);
					if(obj instanceof IVerdictCategoryProvider)
					{
						testTypeToVerdictProviderMap.put(testType, obj);
						return (IVerdictCategoryProvider)obj;
					}
				}
				catch (Throwable t) {
					UiPlugin.logError(t);
				}
			}
			else if(provider instanceof IVerdictCategoryProvider)
				return (IVerdictCategoryProvider)provider;
		}
		
		return null;
	}

	/**
	 * This method always returns <code>null</code> because 
	 * <code>IDetailPageFactory</code> does not work with the new eclipse forms based Test Log Viewer since 4.1.
	 * Extensions should implement {@link org.eclipse.ui.forms.IDetailsPage IDetailsPage} instead.
	 * @return null 
	 * @deprecated use {@link #getEventDetailsPage(String) } instead.
	 */
	public IDetailPageFactory getEventDetailPageFactory(String eventType) 
	{
		return null;
	}
	
	/**
	 * This method always returns <code>null</code> because 
	 * <code>IDetailPageFactory</code> does not work with the new eclipse forms based Test Log Viewer since 4.1.
	 * Extensions should implement {@link org.eclipse.ui.forms.IDetailsPage IDetailsPage} instead.
	 * @return null
	 * @deprecated use {@link #getEventDetailsPage(String) } instead.
	 */
	public IDetailPageFactory getResultDetailPageFactory(String resultType) {
		return null;
	}

}
