/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: StructuredViewerSection.java,v 1.5 2005/05/18 08:18:11 jcanches Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form.util;

import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.EditorSection;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.base.StructuredViewerPart;
import org.eclipse.hyades.test.ui.internal.model.ui.LocalTransfer;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IEditorPart;

/**
 * @author marcelop
 * @since 0.3.0
 */
public abstract class StructuredViewerSection 
extends EditorSection
{
	private StructuredViewerPart viewerPart;
	private boolean handleDefaultButton;
	
	/**
	 * Constructor for EMFStructuredViewerSection
	 * @param editorForm
	 * @param buttonLabels
	 */
	protected StructuredViewerSection(EditorForm editorForm, String[] buttonLabels)
	{
		super(editorForm);
		setButtonLabels(buttonLabels);
	}
	
	/**
	 * Constructor for EMFStructuredViewerSection.  Callers must execute 
	 * {@link #setButtonLabels(String[])} in the constructor.
	 * @param editorForm
	 */
	protected StructuredViewerSection(EditorForm editorForm)
	{
		super(editorForm);
	}	

	/**
	 * Sets the button labels.  This method should not be invoked more than once.
	 * @param buttonLabels
	 */
	protected void setButtonLabels(String[] buttonLabels)
	{
		if(viewerPart != null)
			return;
			
		viewerPart = createViewerPart(buttonLabels);
		viewerPart.setMinimumSize(50, 50);
	}
    
    /**
     * Creates this section's viewer part.
     * @param buttonLabels
     * @param toolTips
     * @return StructuredViewerPart
     */
    protected StructuredViewerPart createViewerPart(String[] buttonLabels,String[] toolTips)
    {
        return null;
    }

    /**
     * Sets the button labels.  This method should not be invoked more than once.
     * @param buttonLabels
     * @param tooltips
     */
    protected void setButtonLabels(String[] buttonLabels, String[] tooltips)
    {
        if(viewerPart != null)
            return;
            
        viewerPart = createViewerPart(buttonLabels,tooltips);
        viewerPart.setMinimumSize(50, 50);
    }

	
	/**
	 * @see org.eclipse.update.ui.forms.internal.FormSection#dispose()
	 */
	public void dispose()
	{
		viewerPart = null;
		super.dispose();
	}

	protected void setHandleDefaultButton(boolean handle)
	{
		handleDefaultButton = handle;	
	}
	
	protected boolean isHandlingDefaultButton()
	{
		return handleDefaultButton;
	}
	
	/**
	 * Creates this section's viewer part.
	 * @param buttonLabels
	 * @return StructuredViewerPart
	 */
	abstract protected StructuredViewerPart createViewerPart(String[] buttonLabels);
	
	protected StructuredViewerPart getViewerPart()
	{
		return viewerPart;
	}
	
	/**
	 * @see org.eclipse.update.ui.forms.internal.FormSection#setFocus()
	 */
	public void setFocus()
	{
		getViewerPart().getViewer().getControl().setFocus();
	}
	
	protected Composite createClientContainer(Composite parent, int span, FormWidgetFactory factory)
	{
		Composite container = factory.createComposite(parent);
		GridLayout layout = new GridLayout();
		layout.marginWidth = layout.marginHeight = 2;
		layout.numColumns = span;
		container.setLayout(layout);
		return container;
	}

	protected void createViewerPartControl(Composite parent, int style, int span, FormWidgetFactory factory)
	{
		viewerPart.createControl(parent, style, span, factory);
		createActions();		
		
		MenuManager popupMenuManager = new MenuManager();
		IMenuListener listener = new IMenuListener()
		{
			public void menuAboutToShow(IMenuManager menuManager)
			{
				fillContextMenu(menuManager);
			}
		};
		popupMenuManager.addMenuListener(listener);
		popupMenuManager.setRemoveAllWhenShown(true);
		Control control = viewerPart.getControl();
		Menu menu = popupMenuManager.createContextMenu(control);
		control.setMenu(menu);
		String menuId = getPopupMenuId();
		if (menuId != null) {
			IEditorPart part = getEditorForm().getBaseEditorExtension().getHyadesEditorPart().getEditorPart();
			part.getSite().registerContextMenu(menuId, popupMenuManager, viewerPart.getViewer());
		}
		control.getAccessible().addAccessibleListener(this);
	}
	
	/**
	 * Returns the menu id used for the popup menu associated to the structured viewer.
	 * This id must be used by object contributions from the eclipse UI popup menu
	 * extension point, in order to contribute to the menu.
	 * @return A menu name, or null if popup menu contributions should not be supported by the
	 * viewer popup menu.
	 * @author jcanches
	 */
	protected String getPopupMenuId() {
		return null;
	}
	
	protected void createActions()
	{
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseEditorSection#setInput(java.lang.Object)
	 */
	public void setInput(Object object)
	{
		getViewerPart().getViewer().setInput(object);
		if(object != null)
			updateActionsAndButtons(new StructuredSelection(object));
		else
			updateActionsAndButtons(StructuredSelection.EMPTY);
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseEditorSection#getInput()
	 */
	public Object getInput()
	{
		return getViewerPart().getViewer().getInput();
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseEditorSection#getStructuredSelection()
	 */
	public IStructuredSelection getStructuredSelection()
	{
		return (IStructuredSelection)getViewerPart().getViewer().getSelection();
	}

	/**
	 * @see org.eclipse.ui.part.ISetSelectionTarget#selectReveal(org.eclipse.jface.viewers.ISelection)
	 */
	public void selectReveal(ISelection selection)
	{
		if(UIUtil.areEquals(getStructuredSelection(), selection))
			return;
		
		getViewerPart().getViewer().setSelection(selection, true);
	}	

	/**
	 * Subclasses may overwrite to add menu.
	 * @param menuManager
	 */
	protected void fillContextMenu(IMenuManager menuManager)
	{
	}

	protected void entryModified(Object entry, String value)
	{
	}
	
	protected void selectionChanged(IStructuredSelection structuredSelection)
	{
		fireSelectionNotification(structuredSelection);
		getHyadesEditorPart().setSelection(structuredSelection);
		updateActionsAndButtons(structuredSelection);		
	}
	
	protected void handleDoubleClick(IStructuredSelection structuredSelection)
	{
	}

	/**
	 * Invoked when one of the buttons is selected.
	 * @param index
	 */
	protected void buttonSelected(int index)
	{
	}
	
	/**
	 * Method invoked when the actions and buttons need to be updated.
	 * @param structuredSelection
	 */
	protected void updateActionsAndButtons(IStructuredSelection structuredSelection)
	{
	}
	
	/**
	 * Prepares the section to do the paste
	 */
	protected void doPaste()
	{
		IStructuredSelection structuredSelection = getStructuredSelection();
		if (structuredSelection.size() >1 )
			return;
		
		Object target = structuredSelection.getFirstElement();
		
		Clipboard clipboard = getClipboard();
		LocalTransfer localTransfer = LocalTransfer.getInstance();
		Object [] objects = (Object[])clipboard.getContents(localTransfer);
		if((objects!=null) && (objects.length > 0))
			doPaste(target, objects);
	}	
	
	/**
	 * Subclasses should overwrite to execute the paste.  This method is invoked
	 * only if the objects array has at least one element. 
	 * @param target
	 * @param objects
	 */
	protected void doPaste(Object target, Object[] objects)
	{
	}
	
	/**
	 * @see org.eclipse.update.ui.forms.internal.FormSection#canPaste(org.eclipse.swt.dnd.Clipboard)
	 */
	public boolean canPaste(Clipboard clipboard)
	{
		IStructuredSelection structuredSelection = getStructuredSelection();
		if(structuredSelection.size()>1)
			return false;
			
		Object target = structuredSelection.getFirstElement();
		LocalTransfer modelTransfer = LocalTransfer.getInstance();
		Object [] objects = (Object[])clipboard.getContents(modelTransfer);
		if (objects!=null && objects.length>0)
			return canPaste(target, objects);
			
		return false;
	}
	
	/**
	 * Returns whether the paste action is allowed. 
	 * @param target
	 * @param objects
	 * @return boolean
	 */
	protected boolean canPaste(Object target, Object []objects)
	{
		return false;
	}	
}
