/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LaunchPropertyTester.java,v 1.5 2009/05/11 12:50:24 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.shortcuts;

import org.eclipse.core.expressions.PropertyTester;
import org.eclipse.core.resources.IFile;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.navigator.ITestCaseProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITestSuiteProxyNode;

/**
 * <p>This class is used by the <code>PropertyTester</code> extension point for evaluating the
 * <code>isLaunchable</code> property. This property is used by the <code>LaunchShortcut</code> extension
 * for determining whether a Run, Debug or Profile menu should be provided for the current
 * selection.</p>
 * 
 * <p>Starting with TPTP 4.0, this property accepts an optional argument, the mode
 * (i.e. run, debug or profile), since tests may be available in a subset of 
 * all available modes only.</p>
 * 
 * 
 * @author  Julien Canches
 * @author  Paul Slauenwhite
 * @author  Jeffrey Nevicosi
 * @version May 12, 2009
 * @since   February 1, 2005
 */
public class LaunchPropertyTester extends PropertyTester {
	
	public static final String PROPERTY_IS_LAUNCHABLE = "isLaunchable"; //$NON-NLS-1$
	
	/* (non-Javadoc)
	 * @see org.eclipse.core.expressions.IPropertyTester#test(java.lang.Object, java.lang.String, java.lang.Object[], java.lang.Object)
	 */
	public boolean test(Object receiver, String property, Object[] args, Object expectedValue) {
		if (PROPERTY_IS_LAUNCHABLE.equals(property)) {
			String mode;
			if (args.length == 0) {
				mode = ILaunchManager.RUN_MODE; // Default to run (backward compatibility)
			} else if (args[0] instanceof String) {
				mode = (String)args[0];
			} else {
				UiPlugin.logError("Invalid arg passed to isLaunchable property: " + args[0]); //$NON-NLS-1$
				return false;
			}
			return new Boolean(getRunnableItemFromElement(receiver, mode, false) != null)
				.equals(expectedValue);
		}
		return false;
	}
	
	/**
	 * Consider an object (listed in a selection) and try to extract its underlying Test Suite object
	 * if it is possible.
     *
	 * @param element Any object.
	 * @return A TPFTestSuite if the object is or is associated to a test suite, null otherwise.
	 */
	public static Object getRunnableItemFromElement(Object element, String mode) {
		return getRunnableItemFromElement(element, mode, true);
	}

	/**
	 * Consider an object (listed in a selection) and try to extract its underlying Test Suite object
	 * if it is possible.
	 * 
	 * @param element Any object.
	 * @param forceLoad  If false, will not load element if element is a proxy node is 
	 * @return A TPFTestSuite if the object is or is associated to a test suite, null otherwise.
	 */
	public static Object getRunnableItemFromElement(Object element, String mode, boolean forceLoad) {
		Object candidateElement = null;
		if (element instanceof TPFTestSuite ||
			element instanceof TPFTestCase ||
			element instanceof TPFTestComponent) {
			candidateElement = element;
		} else if (element instanceof IFile) {
			IFile file = (IFile) element;
			if (ICommonConstants.TEST_SUITE_FILE_EXTENSION.equals(file.getFileExtension())) {
				EObject[] eobjs = EMFUtil.load(null, file);
				for (int i = 0; i < eobjs.length; i++) {
					if (eobjs[i] instanceof TPFTestSuite) {
						candidateElement = eobjs[i];
						break;
					}
				}
			} else if (ICommonConstants.TEST_COMPONENT_EXTENSION.equals(file.getFileExtension())) {
				EObject[] eobjs = EMFUtil.load(null, file);
				for (int i = 0; i < eobjs.length; i++) {
					if (eobjs[i] instanceof TPFTestComponent) {
						candidateElement = eobjs[i];
						break;
					}
				}
			}
		}
		if (candidateElement != null) {
			
			if (LaunchConfigurationExtensionsManager.getInstance().isShortcutAvailable(candidateElement, mode)) {
				return candidateElement;
			} 
		}
		else{

			String testID = null;

			if (element instanceof ITestSuiteProxyNode) {

				ITestSuiteProxyNode tsProxy = ((ITestSuiteProxyNode)(element));

				//Note: Loading the test suite results in a memory leak.
				if (forceLoad){
					candidateElement = tsProxy.getTestSuite();
				}

				testID = tsProxy.getType();
			} 
			else if (element instanceof ITestCaseProxyNode) {

				ITestCaseProxyNode tcProxy = ((ITestCaseProxyNode)(element));

				//Note: Loading the test suite results in a memory leak.
				if (forceLoad){
					candidateElement = tcProxy.getTestCase();
				}

				testID = tcProxy.getType();
			}

			if ((testID != null) && (LaunchConfigurationExtensionsManager.getInstance().isShortcutAvailable(testID, mode))) {

				if (forceLoad){
					return candidateElement;
				}
				else{
					return element;
				}
			}
		}
		
		return null;
	}
}
