/*******************************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LaunchShortcut.java,v 1.16 2006/05/18 16:26:44 jgout Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.shortcuts;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugModelPresentation;
import org.eclipse.debug.ui.ILaunchShortcut;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.core.launch.configurations.DeploymentLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.configurations.ExecutionHistoryLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.configurations.TestComponentLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.extensions.IDeploymentsProvider;
import org.eclipse.hyades.test.core.launch.extensions.IExecutionHistoryDefaultsProvider;
import org.eclipse.hyades.test.core.launch.extensions.ITestLaunchConfigurationValidator;
import org.eclipse.hyades.test.core.launch.extensions.ITestLaunchConfigurationValidator.Diagnostic;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.launch.LaunchMessages;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

/**
 * 
 * @author pnedelec
 * @author jcanches
 */
public class LaunchShortcut implements ILaunchShortcut {

	/*
	 * @see org.eclipse.debug.ui.ILaunchShortcut#launch(org.eclipse.ui.IEditorPart,
	 *      java.lang.String)
	 */
	public void launch(IEditorPart editor, String mode) {
		IEditorInput input = editor.getEditorInput();
		if (input instanceof IFileEditorInput) {
			IFileEditorInput finput = (IFileEditorInput) input;
			Object obj = LaunchPropertyTester.getRunnableItemFromElement(finput
					.getFile(), mode);
			if (obj != null) {
				launchElement(obj, mode);
			}
		}
	}

	/*
	 * @see org.eclipse.debug.ui.ILaunchShortcut#launch(org.eclipse.jface.viewers.ISelection,
	 *      java.lang.String)
	 */
	public void launch(ISelection selection, String mode) {
		if (selection instanceof IStructuredSelection) {
			// Use the first element from the collection
			IStructuredSelection ss = (IStructuredSelection) selection;
			Iterator it = ss.iterator();
			while (it.hasNext()) {
				Object element = it.next();
				Object launchable = LaunchPropertyTester
						.getRunnableItemFromElement(element, mode);
				if (launchable != null) {
					launchElement(launchable, mode);
				}
			}
		}
	}

	/**
	 * Launch a launchable element. If a Launch Configuration already exists for
	 * the specified element, this launch configuration is used to launch the
	 * element. Otherwise, a new Launch Configuration is created. If several
	 * launch configurations match, a dialog is opened to allow the user to
	 * choose from the candidate launch configurations.
	 * 
	 * @param testSuites
	 *            The Test Suites to launch
	 * @param mode
	 *            The launch mode (debug, run or profile).
	 */
	private void launchElement(Object launchable, String mode) {
		// Find an existing launch configuration for the set of Test Suites
		ILaunchConfiguration lconfig = findLaunchConfiguration(launchable, mode);
		try {
			if (lconfig == null) {
				// If no match, create a new Launch Configuration
				lconfig = createLaunchConfiguration(launchable);
			}
            //- Validate the launch
            boolean ignoreWarnings = true;
            ITestLaunchConfigurationValidator validator = LaunchConfigurationExtensionsManager.getInstance().getLaunchConfigurationValidator(launchable);
            if (validator != null) {
                ITestLaunchConfigurationValidator.Diagnostic diag = validator.validate(lconfig, new ResourceSetImpl());
                if (diag != null) {
                    if (diag.getSeverity() == ITestLaunchConfigurationValidator.Diagnostic.ERROR) {
                        reportFailure(diag);
                        return;
                    }
                    else if (diag.getSeverity() == ITestLaunchConfigurationValidator.Diagnostic.WARNING) {
                        ignoreWarnings = reportWarning(diag);
                    }
                }
            }
			if (lconfig != null && ignoreWarnings) {
				DebugUITools.launch(lconfig, mode);
			}
		} catch (Throwable t) {
			reportFailure(t);
		}
	}

	/**
	 * Find a Launch Configuration that matches the launchable element.
	 * 
	 * @param launchable
	 *            The launchable element that the returned Launch Configuration
	 *            must run.
	 * @return An existing Launch Configuration if found, null otherwise.
	 */
	private ILaunchConfiguration findLaunchConfiguration(Object launchable,
			String mode) {
		List candidateConfigs = Collections.EMPTY_LIST;
		ILaunchConfigurationType configsType = getConfigType(launchable);
		if (configsType == null) {
			throw new NullPointerException("Configs type must be non-null"); //$NON-NLS-1$
		}

		//- Bugzilla #112750
		ResourceSet shortcutResourceSet = new ResourceSetImpl();
		try {
			ILaunchConfiguration[] configs = getLaunchManager()
					.getLaunchConfigurations(configsType);
			candidateConfigs = new ArrayList(configs.length);
			for (int i = 0; i < configs.length; i++) {
				Object configLaunchableElement = null;
				if (launchable instanceof TPFTest) {
					configLaunchableElement = TestLaunchConfigurationFacade
							.getTest(configs[i], shortcutResourceSet);
				} else if (launchable instanceof TPFTestComponent) {
					configLaunchableElement = TestComponentLaunchConfigurationFacade
							.getTestComponent(configs[i], shortcutResourceSet);
                }
                if (launchable instanceof EObject && configLaunchableElement instanceof EObject) {
                    if (EcoreUtil.getURI((EObject) launchable).equals(EcoreUtil.getURI((EObject) configLaunchableElement))) {
                        candidateConfigs.add(configs[i]);
                    }
                }
			}
			if (candidateConfigs.size() == 0) {
				return null;
			} else if (candidateConfigs.size() == 1) {
				return (ILaunchConfiguration) candidateConfigs.get(0);
			} else {
				return chooseFromCandidateConfigs(candidateConfigs, mode);
			}
		} catch (CoreException e) {
			UiPlugin.logError(e);
		}
		return null;
	}

	/**
	 * Show a selection dialog that allows the user to choose one of the
	 * specified launch configurations. Return the chosen config, or
	 * <code>null</code> if the user cancelled the dialog.
	 */
	private ILaunchConfiguration chooseFromCandidateConfigs(List configList,
			String mode) {
		IDebugModelPresentation labelProvider = DebugUITools
				.newDebugModelPresentation();
		try {
			ElementListSelectionDialog dialog = new ElementListSelectionDialog(this
					.getActiveShell(), labelProvider);
			dialog.setElements(configList.toArray());
			dialog.setTitle(LaunchMessages.LaunchShortcut_ambiguousLaunchConfigurationTitle);
			if (mode.equals(ILaunchManager.DEBUG_MODE)) {
				dialog.setMessage(LaunchMessages.LaunchShortcut_selectDebugLaunchConfigPrompt);
			} else if (mode.equals(ILaunchManager.PROFILE_MODE)) {
				dialog.setMessage(LaunchMessages.LaunchShortcut_selectProfileLaunchConfigPrompt);
			} else {
				dialog.setMessage(LaunchMessages.LaunchShortcut_selectRunLaunchConfigPrompt);
			}
			dialog.setMultipleSelection(false);
			if (dialog.open() == Window.OK) {
				return (ILaunchConfiguration) dialog.getFirstResult();
			}
			return null;
		} finally {
			labelProvider.dispose();
		}
	}

	/**
	 * Create a new Launch Configuration that runs the specified launchable
	 * element.
	 * 
	 * @param launchable
	 * @return A new Launch Configuration.
	 */
	private ILaunchConfiguration createLaunchConfiguration(Object launchable)
			throws CoreException {
		ILaunchConfiguration config = null;
		ILaunchConfigurationType configType = getConfigType(launchable);
        String launchName = ""; //$NON-NLS-1$
        if (launchable instanceof CMNNamedElement) {
            //- Use the name of the launchable element rather than the default
            launchName = getLaunchManager().generateUniqueLaunchConfigurationNameFrom(((CMNNamedElement) launchable).getName());
        } else {
            launchName = getLaunchManager().generateUniqueLaunchConfigurationNameFrom(LaunchMessages.LaunchShortcut_uniqueNamePrefix);
        }
		ILaunchConfigurationWorkingCopy wc = configType.newInstance(null, launchName);
		if (launchable instanceof TPFTest) {
			TestLaunchConfigurationFacade.setTest(wc, (TPFTest) launchable);
			TestLaunchConfigurationFacade.setDefaultSourcePathProvider(wc);
		} else if (launchable instanceof TPFTestComponent) {
			TestComponentLaunchConfigurationFacade.setTestComponent(wc,
					(TPFTestComponent) launchable);
		}
		IDeploymentsProvider dprovider = LaunchConfigurationExtensionsManager
				.getInstance().getDeploymentsProvider(launchable);
		TPFDeployment depl = null;
		depl = dprovider.getDefaultDeployment(launchable);
		if (depl == null) {
			throwException(LaunchMessages._EXC_LaunchShortcut_noDefaultDeployment, null);
		}
		DeploymentLaunchConfigurationFacade.setDeployment(wc, depl);
		String ehName = null;
		IContainer ehLocation = null;
		IExecutionHistoryDefaultsProvider eprovider = LaunchConfigurationExtensionsManager
				.getInstance().getExecutionHistoryDefaultsProvider(launchable);
		ehName = eprovider.getDefaultName(launchable);
		if (ehName == null) {
			ehName = LaunchConfigurationExtensionsManager.getInstance()
					.getDefaultExecutionHistoryDefaultsProvider()
					.getDefaultName(launchable);
		}
		ehLocation = eprovider.getDefaultLocation(launchable);
		if (ehLocation == null) {
			ehLocation = LaunchConfigurationExtensionsManager.getInstance()
					.getDefaultExecutionHistoryDefaultsProvider()
					.getDefaultLocation(launchable);
		}
		if (ehName == null || ehLocation == null) {
			throwException(LaunchMessages._EXC_LaunchShortcut_noDefaultExecHistoryInfo, null);
		}
		ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryName(wc,
				ehName);
		ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryLocation(
				wc, ehLocation);
		config = wc.doSave();
		return config;
	}

	private void reportFailure(Throwable exception) {
		UiPlugin.logError(exception);
		MessageBox msgBox = new MessageBox(PlatformUI.getWorkbench()
				.getActiveWorkbenchWindow().getShell(), SWT.ICON_ERROR | SWT.OK);
		msgBox.setText(LaunchMessages.LaunchShortcut_errorDialogTitle);
		String message = null;
		if (exception instanceof CoreException) {
			CoreException ce = (CoreException) exception;
			if (ce.getStatus() != null && ce.getStatus().getMessage() != null) {
				message = ce.getStatus().getMessage();
			}
		}
		if (message == null) {
			message = LaunchMessages.LaunchShortcut_errorMessage;
		}
		msgBox.setMessage(message);
		msgBox.open();
	}
    
    private void reportFailure(Diagnostic diagnostic) {
        MessageBox msgBox = new MessageBox(PlatformUI.getWorkbench()
                .getActiveWorkbenchWindow().getShell(), SWT.ICON_ERROR | SWT.OK);
        msgBox.setText(LaunchMessages.LaunchShortcut_errorDialogTitle);
        msgBox.setMessage(diagnostic.getMessage());
        msgBox.open();
    }
    
    private boolean reportWarning(Diagnostic diagnostic) {
        boolean hideWarningDialog = UiPlugin.getDefault().getPreferenceStore().getBoolean(TestUI.HIDE_LAUNCH_VALIDATION_WARNING);
        if (!hideWarningDialog) {
            String message = NLS.bind(LaunchMessages.LaunchShortcut_Message, diagnostic.getMessage());
            MessageDialogWithToggle messageDialog = openWarning(LaunchMessages.LaunchShortcut_Dialog_Title, message, false, UiPlugin.getDefault().getPreferenceStore(),
                    TestUI.HIDE_LAUNCH_VALIDATION_WARNING);
            return (messageDialog.getReturnCode() == IDialogConstants.YES_ID);
        }
        return true;
    }
    
    protected MessageDialogWithToggle openWarning(String title, String message, boolean toggleState, IPreferenceStore store, String key) {
        MessageDialogWithToggle dialog = new WarningDialogWithToggle(title, message, toggleState);
        dialog.setPrefStore(store);
        dialog.setPrefKey(key);
        dialog.open();
        return dialog;
    }

    //- Bugzilla #107652
    private static class WarningDialogWithToggle extends MessageDialogWithToggle {

        public WarningDialogWithToggle(String dialogTitle, String message, boolean toggleState) {
            super(Display.getCurrent().getActiveShell(), dialogTitle, null, message, MessageDialog.WARNING, new String[] { IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL},
                    0, LaunchMessages.LaunchShortcut_Remember_Label, toggleState);
        }

        protected void buttonPressed(int buttonId) {
            setReturnCode(buttonId);
            close();
            IPreferenceStore prefStore = getPrefStore();
            String prefKey = getPrefKey();
            if (prefStore != null && prefKey != null) {
                switch (buttonId) {
                    case IDialogConstants.YES_ID:
                        prefStore.setValue(prefKey, getToggleState());
                        break;
                    case IDialogConstants.NO_ID:
                    default:
                        prefStore.setValue(prefKey, false);
                        break;
                }
            }
        }

    }

	private void throwException(String localizedMessage, Throwable exception)
			throws CoreException {
		IStatus status = new Status(IStatus.ERROR, UiPlugin.getID(), 0,
				localizedMessage, exception);
		throw new CoreException(status);
	}

	/**
	 * Return the appropriate Launch Configuration Type for launching the
	 * supplied launchable element.
	 * 
	 * @param launchable
	 * @return
	 */
	private ILaunchConfigurationType getConfigType(Object launchable) {
		ILaunchConfigurationType configType = null;
		if (launchable instanceof TPFTest) {
			configType = TestLaunchConfigurationFacade
					.getLaunchConfigurationType();
		} else if (launchable instanceof TPFTestComponent) {
			configType = TestComponentLaunchConfigurationFacade
					.getLaunchConfigurationType();
		}
		return configType;
	}

	/**
	 * Return the shell of the active window. Conveniency method.
	 * 
	 * @return
	 */
	private Shell getActiveShell() {
		IWorkbenchWindow window = UiPlugin.getInstance().getWorkbench()
				.getActiveWorkbenchWindow();
		if (window == null) {
			return null;
		}
		return window.getShell();
	}

	/**
	 * Return the Debug Plug-in's Launch Manager. Conveniency method.
	 * 
	 * @return
	 */
	private ILaunchManager getLaunchManager() {
		return DebugPlugin.getDefault().getLaunchManager();
	}

}
