/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionHistoryTab.java,v 1.20 2007/04/27 21:45:12 jptoomey Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.tabs;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.core.launch.configurations.ExecutionHistoryLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.extensions.IExecutionResultsLocationFilter;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.launch.LaunchMessages;
import org.eclipse.hyades.test.ui.internal.launch.providers.ExecutionHistoryLocationProvider;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.launch.tabGroups.AbstractTabGroup;
import org.eclipse.hyades.ui.util.LocationValidator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;

/**
 * Provides the Test Execution History Tab. This tab allows the user to choose the name
 * and the location of the Test Execution History file.
 * @author jcanches
 */
public class ExecutionHistoryTab extends AbstractLaunchConfigurationTab {
	
	private Button useDefaults;
	private Composite valuesComposite;
	private TreeViewer tree;
	private Text nameText;
	private Button overwrite;
	private Button database;
	private Button testLogFile;
	private List valuesCompositeControls = new ArrayList();
	protected boolean disableNotification = false;
	private String launchConfigurationType;
	private AbstractTabGroup tabGroup;
	
	/**
	 * Creates a Test Log tab.
	 * @param launchConfigurationType The type of the launch configuration
	 * @param tabGroup The tab group that contains the tab. <code>null</code> is allowed
	 * but not recommended as some nice-to-have features are disabled in this case.
	 */
	public ExecutionHistoryTab(String launchConfigurationType, AbstractTabGroup tabGroup) {
		this.launchConfigurationType = launchConfigurationType;
		this.tabGroup = tabGroup;
	}
	
	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NULL);
        GridLayout gl = new GridLayout();        
        composite.setLayout(gl);
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        composite.setFont(parent.getFont());

        useDefaults = new Button(composite, SWT.CHECK);
        useDefaults.setText(LaunchMessages.ExecutionHistoryTab_useDefaults);
        useDefaults.setToolTipText(LaunchMessages.ExecutionHistoryTab_useDefaultsTooltip);
        useDefaults.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				setUserActionsPerformed(true);
				updateValuesCompositeEnabledState();
				updateLaunchConfigurationDialog();
			}
		});
        
        createValuesComposite(composite);

        setControl(composite);
	}
	
	private void createValuesComposite(Composite parent) {
		valuesCompositeControls.clear();
        valuesComposite = new Composite(parent, SWT.NULL);
        GridLayout gl = new GridLayout();        
        valuesComposite.setLayout(gl);
        valuesComposite.setLayoutData(new GridData(GridData.FILL_BOTH));
        valuesComposite.setFont(parent.getFont());

        Label lblName = new Label(valuesComposite, SWT.NONE);
        valuesCompositeControls.add(lblName);
        GridData gdl = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
        gdl.horizontalSpan = 2;
        lblName.setLayoutData(gdl);
        lblName.setText(LaunchMessages.ExecutionHistoryTab_name);

        nameText = new Text(valuesComposite, SWT.BORDER);
        GridData gdt = new GridData(GridData.FILL_HORIZONTAL);
        gdt.horizontalSpan = 2;
        nameText.setLayoutData(gdt);
        nameText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				if (!disableNotification) {
					setUserActionsPerformed(true);
					updateLaunchConfigurationDialog();
				}
			}
		});
        valuesCompositeControls.add(nameText);
        
		overwrite = new Button(valuesComposite, SWT.CHECK);
		overwrite.setText(LaunchMessages.ExecutionHistoryTab_override);
		overwrite.setToolTipText(LaunchMessages.ExecutionHistoryTab_overrideTooltip);
		overwrite.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if (!disableNotification) {
					setUserActionsPerformed(true);
					updateLaunchConfigurationDialog();
				}
			}
		});
        GridData gdl3 = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
        gdl3.horizontalSpan = 2;
        overwrite.setLayoutData(gdl3);
        valuesCompositeControls.add(overwrite);

        // This feature is not shipping in 4.4.
        // Only create this option is the debug option is set.
		if(ModelDebugger.INSTANCE.debugDatabaseResourcePostfix!=null && 
				   ModelDebugger.INSTANCE.debugDatabaseResourcePostfix.length()!=0) 
		{        
			database = new Button(valuesComposite, SWT.CHECK);
		    database.setText(LaunchMessages.ExecutionHistoryTab_publishToDatabase);
		    database.setToolTipText(LaunchMessages.ExecutionHistoryTab_publishToDatabaseTooltip);
		    database.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					if (!disableNotification) {
						setUserActionsPerformed(true);
						updateLaunchConfigurationDialog();
					}
				}
			});
		    GridData gdl4 = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
		    gdl4.horizontalSpan = 2;
		    database.setLayoutData(gdl4);
		    valuesCompositeControls.add(database);
			database.setEnabled(false);
		}
        

        testLogFile = new Button(valuesComposite, SWT.CHECK);
        testLogFile.setText(LaunchMessages.ExecutionHistoryTab_testLogFile);
        testLogFile.setToolTipText(LaunchMessages.ExecutionHistoryTab_testLogFileTooltip);
        testLogFile.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if (!disableNotification) {
					setUserActionsPerformed(true);
					updateLaunchConfigurationDialog();
				}
			}
		});
        GridData gdl5 = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
        gdl5.horizontalSpan = 2;
        testLogFile.setLayoutData(gdl5);
        valuesCompositeControls.add(testLogFile);

        Label lblLocation = new Label(valuesComposite, SWT.NONE);
        GridData gdl2 = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
        gdl2.horizontalSpan = 2;
        lblLocation.setLayoutData(gdl2);
        lblLocation.setText(LaunchMessages.ExecutionHistoryTab_location);
        valuesCompositeControls.add(lblLocation);

        createLocationSelector(valuesComposite);
	}
	
    private void createLocationSelector(Composite composite) {
    	GridData grid = new GridData(GridData.FILL_BOTH);
		grid.widthHint	= composite.getClientArea().width/2 ;
		grid.heightHint	= composite.getClientArea().height/2;

        tree = new TreeViewer(composite);
        tree.getControl().setLayoutData(grid);
        IExecutionResultsLocationFilter filter = LaunchConfigurationExtensionsManager.getInstance().getExecutionResultsLocationFilter(this.launchConfigurationType);
        ExecutionHistoryLocationProvider provider = new ExecutionHistoryLocationProvider(filter);
        tree.setContentProvider(provider);
        tree.setLabelProvider(provider);
        tree.setInput(ResourcesPlugin.getWorkspace().getRoot());
        
        tree.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				if (!disableNotification) {
					setUserActionsPerformed(true);
					updateLaunchConfigurationDialog();
				}
			}
		});
        valuesCompositeControls.add(tree.getControl());
    }
	
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
		ExecutionHistoryLaunchConfigurationFacade.setUseDefaults(configuration, true);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	public void initializeFrom(ILaunchConfiguration configuration) {
		try {
			boolean useDefs = ExecutionHistoryLaunchConfigurationFacade.isUseDefaults(configuration);
			useDefaults.setSelection(useDefs);
			updateValuesCompositeEnabledState();
			
    		applyValuesComposite(configuration);
        } catch (CoreException e) {
            UiPlugin.logError(e);
		}
	}
	
	private void applyValuesComposite(ILaunchConfiguration configuration) throws CoreException {
		// Because we are programmatically modifying the controls' values, we should not
		// react to change events. Otherwise we may end up with an infinite loop
		this.disableNotification = true;
		try {
			String name = ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryName(configuration);
			if (name == null) {
				nameText.setText(""); //$NON-NLS-1$
			} else {
				nameText.setText(name);
			}
			
			boolean overrideState = ExecutionHistoryLaunchConfigurationFacade.isExecutionHistoryOverriden(configuration);
			overwrite.setSelection(overrideState);
			
	        // This feature is not shipping in 4.4.
	        // Only create this option is the debug option is set.
			if(ModelDebugger.INSTANCE.debugDatabaseResourcePostfix!=null && 
					   ModelDebugger.INSTANCE.debugDatabaseResourcePostfix.length()!=0) 
			{        
				boolean databaseState = ExecutionHistoryLaunchConfigurationFacade.isExecutionHistoryDatabase(configuration);
				database.setSelection(databaseState);
			}
			
			boolean testLogFileState = ExecutionHistoryLaunchConfigurationFacade.isExecutionHistoryTestLogFile(configuration);
			testLogFile.setSelection(testLogFileState);
			
			IContainer location = ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryLocation(configuration);
			if (location == null) {
				tree.setSelection(new StructuredSelection());
			} else {
				StructuredSelection sel = new StructuredSelection(location);
				tree.setSelection(sel, /*reveal*/true);
			}
		} finally {
			this.disableNotification = false;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
		boolean useDefs = getUseDefaultsState();
		ExecutionHistoryLaunchConfigurationFacade.setUseDefaults(configuration, useDefs);
		if (!useDefs) {
	        String name = getExecutionHistoryName();
	        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryName(configuration, name);
	        boolean overrideState = getOverwriteState();
	        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryOverriden(configuration, overrideState);

	        // This feature is not shipping in 4.4.
	        // Only create this option is the debug option is set.
			if(ModelDebugger.INSTANCE.debugDatabaseResourcePostfix!=null && 
					   ModelDebugger.INSTANCE.debugDatabaseResourcePostfix.length()!=0) 
			{        
		        boolean databaseState = getDatabaseState();
		        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryDatabase(configuration, databaseState);
			}
			
	        boolean testLogFileState = getTestLogFileState();
	        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryTestLogFile(configuration, testLogFileState);
	        IContainer location = getSelectedLocation();
	        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryLocation(configuration, location);
		} else {
			try {
				applyValuesComposite(configuration);
			} catch (CoreException e) {
				UiPlugin.logError(e);
			}
		}
	}

	public String getName() {
		return LaunchMessages.ExecutionHistoryTab_tabName;
	}
	
	public Image getImage() {
		return TestUIImages.INSTANCE.getImage(TestUIImages.IMG_LCONFIG_EXECRESULTS_TAB);
	}

	protected boolean getUseDefaultsState() {
		return useDefaults.getSelection();
	}
	
	protected String getExecutionHistoryName() {
		return nameText.getText();
	}
	
	protected boolean getOverwriteState() {
		return overwrite.getSelection();
	}

	protected boolean getDatabaseState() {
		return database.getSelection();
	}
	
	protected boolean getTestLogFileState() {
		return testLogFile.getSelection();
	}
	
	protected IContainer getSelectedLocation() {
		ISelection selection = tree.getSelection();
		if (!selection.isEmpty() && selection instanceof IStructuredSelection) {
			Object obj = ((IStructuredSelection)selection).getFirstElement();
			if (obj instanceof IContainer) {
				return (IContainer)obj;
			}
		}
		return null;
	}
	
    /*
     * @see org.eclipse.debug.ui.ILaunchConfigurationTab#isValid(org.eclipse.debug.core.ILaunchConfiguration)
     */
    public boolean isValid(ILaunchConfiguration configuration) {
        //- Sets the state of the Run button
        setErrorMessage(null);
		
		String name;
		try {
			name = ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryName(configuration);
		} catch (CoreException e) {
			UiPlugin.logError(e);
			name = null;
		}
		if (name == null || name.equals("")) { //$NON-NLS-1$
			if (getUserActionsPerformed()) {
				setErrorMessage(LaunchMessages._ERROR_ExecutionHistoryTab_invalidName);
			}
        	return false;
        } else {
            // bugzilla 86998 mdd
            String invalidChar = LocationValidator.validateOSFileName(name);
            if (invalidChar != null) {
                if (LocationValidator.INVALID_PATH.equals(invalidChar)) {
                    setErrorMessage(CommonUIMessages.OS_Invalid_Name_ERROR_);
                    return false;
                } else {
                    String[] errors = { UiPluginResourceBundle.LOGICAL_NAME, invalidChar}; 
                    setErrorMessage(NLS.bind(UiPluginResourceBundle._ERROR_NAME_SPECIAL_CHAR, errors)); 
                    return false;
                }
            }
        }
		
		IContainer location;
		try {
			location = ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryLocation(configuration);
		} catch (CoreException e) {
			UiPlugin.logError(e);
			location = null;
		}
		if (location == null) {
			if (getUserActionsPerformed()) {
				setErrorMessage(LaunchMessages._ERROR_ExecutionHistoryTab_invalidLocation);
			}
        	return false;
        }
        //- Validate the path to avoid forbidden characters
        String invalidChar = LocationValidator.validatePath(location.getFullPath());
        if (invalidChar != null) {
            if (LocationValidator.INVALID_PATH.equals(invalidChar)) {
                setErrorMessage(CommonUIMessages.OS_Invalid_Location_ERROR_);
            } else {
                String[] errors = { CommonUIMessages.LocationPage_PATH, invalidChar}; 
                setErrorMessage(NLS.bind(CommonUIMessages._25, errors)); 
            }
            return false;
        }
        return true;      		
    }
    
    protected void updateValuesCompositeEnabledState() {
    	boolean enabled = !getUseDefaultsState();
    	Iterator it = valuesCompositeControls.iterator();
    	while (it.hasNext()) {
			Control control = (Control) it.next();
			control.setEnabled(enabled);
		}
    }
    
    private void setUserActionsPerformed(boolean state) {
    	if (tabGroup != null) {
    		tabGroup.setUserActionsPerformed(state);
    	}
    }

    private boolean getUserActionsPerformed() {
    	if (tabGroup != null) {
    		return tabGroup.getUserActionsPerformed();
    	}
    	return true;
    }
}
