/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestAndDeploymentTab.java,v 1.22 2010/05/21 23:58:54 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.tabs;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.core.launch.configurations.DeploymentLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.configurations.ExecutionHistoryLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.extensions.IDeploymentsProvider;
import org.eclipse.hyades.test.core.launch.extensions.ITestLaunchConfigurationValidator;
import org.eclipse.hyades.test.core.launch.extensions.ITestLaunchConfigurationValidator.Diagnostic;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.launch.LaunchMessages;
import org.eclipse.hyades.test.ui.internal.launch.providers.DeploymentProvider;
import org.eclipse.hyades.test.ui.internal.launch.providers.LaunchFilter;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.TestProvider;
import org.eclipse.hyades.test.ui.launch.tabGroups.AbstractTabGroup;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxyNodeListener;
import org.eclipse.hyades.test.ui.navigator.ITestCaseProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITestSuiteProxyNode;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.IElementComparer;
import org.eclipse.jface.viewers.ILabelDecorator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
	
/**
 * <p>Test and Deployment tab of the Test launch configuration.</p>
 * 
 * 
 * @author  Julien Canches
 * @author  Paul Slauenwhite
 * @version May 21, 2010
 * @since   February 1, 2005
 */
public class TestAndDeploymentTab extends AbstractLaunchConfigurationTab {
	
	protected TreeViewer testTreeViewer;
	protected ListViewer deploymentListViewer;
	private String mode;
	private String launchConfigurationType;
	private Resource selectedTestResource;
    Diagnostic lastValidationDiagnostic;
    private ResourceSet resourceSet = new ResourceSetImpl();
    /* The following fields are used to record the test and deployment that were
     * referenced by the launch configuration when it is first opened. In launch modes
     * that are not supported for the selected test/deployment, these values are lost in
     * the widgets (ie. values returned by getSelectedTest/Deployment()), but
     * we record them in these fields, so that performApply does not alter the launch
     * configuration until the user actually clicks on a different value in the test or
     * deployment list. (see bugzilla #129633).
     */
    private TPFTest preservedTest;
    private TPFDeployment preservedDeployment;
    private AbstractTabGroup tabGroup;
	private boolean hasTestChanged;
	
	/**
	 *  
	 * @param mode The mode in which the test shall be launched.
	 * @param launchConfigurationType The type of the launch configuration.
	 * @param tabGroup The tab group that contains the tab. <code>null</code> is allowed
	 * but not recommended as some nice-to-have features are disabled in this case.
	 */
	public TestAndDeploymentTab(String mode, String launchConfigurationType, AbstractTabGroup tabGroup) {
		this.mode = mode;
		this.launchConfigurationType = launchConfigurationType;
		this.tabGroup = tabGroup;
	}

	public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NULL);
        GridLayout gl = new GridLayout();
        gl.numColumns = 2;
        composite.setLayout(gl);
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        composite.setFont(parent.getFont());
        
        // First column
        Composite col1 = new Composite(composite, SWT.NULL);
        GridLayout col1l = new GridLayout();
        col1l.numColumns = 1;
        col1.setLayoutData(new GridData(GridData.FILL_BOTH));
        col1.setLayout(col1l);
        
        Label lblTest = new Label(col1, SWT.NONE);
        lblTest.setText(LaunchMessages.TestAndDeploymentTab_selectTestPrompt);
        
        createTestSelector(col1);

        // Second column
        Composite col2 = new Composite(composite, SWT.NULL);
        GridLayout col2l = new GridLayout();
        col2l.numColumns = 1;
        col2.setLayoutData(new GridData(GridData.FILL_BOTH));
        col2.setLayout(col2l);
        
        Label lblDeployment = new Label(col2, SWT.NONE);
        lblDeployment.setText(LaunchMessages.TestAndDeploymentTab_selectDeploymentPrompt);

        createDeploymentSelector(col2);
        
        setControl(composite);
        //- Listener on close event of the dialog
        Shell shell = getShell();
        if (shell != null && !shell.isDisposed()) {
            getShell().addDisposeListener(new DisposeListener() {
                public void widgetDisposed(DisposeEvent e) {
                    //- Notify the LaunchConfigurationExtensionsManager and its listeners that the launch dialog closes
                    LaunchConfigurationExtensionsManager.getInstance().handleLaunchDialogClosed();
                }
            });
        }
	}
	
    private Control createTestSelector(Composite composite) {
    	GridData grid = new GridData(GridData.FILL_BOTH);
		grid.widthHint	= 250 ;
		grid.heightHint	= 150;		

    	testTreeViewer = new TreeViewer(composite, SWT.SINGLE | SWT.BORDER);
    	testTreeViewer.getControl().setLayoutData(grid);
    	
    	TestProvider provider = new TestProvider(new LaunchFilter(mode, launchConfigurationType), false);
        testTreeViewer.setContentProvider(provider);
        ILabelDecorator decorator = PlatformUI.getWorkbench().getDecoratorManager().getLabelDecorator();
        testTreeViewer.setLabelProvider(new DecoratingLabelProvider(provider, decorator));
    	testTreeViewer.setInput(ResourcesPlugin.getWorkspace().getRoot());
    	provider.addListener(new IProxyNodeListener() {
			public void nodeChanged(Object node) {
				testTreeViewer.refresh(node);
			}
		});
        
    	testTreeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				hasTestChanged = true;
				setUserActionsPerformed(true);
				TPFTest newTest = getSelectedTest();
				if (newTest != null) preservedTest = null;
				TPFTestSuite deploymentInput = fillDeploymentsList(newTest);
				TPFDeployment defaultDeployment;
				if (deploymentInput == null) {
					defaultDeployment = null;
				} else {
					IDeploymentsProvider dprovider = LaunchConfigurationExtensionsManager.getInstance().getDeploymentsProvider(deploymentInput);
					defaultDeployment = dprovider.getDefaultDeployment(deploymentInput);
				}
				setSelectedDeployment(defaultDeployment);
				updateLaunchConfigurationDialog();
			}
		});
    	
    	return testTreeViewer.getControl();
    }
    
    private Control createDeploymentSelector(Composite composite){
    	GridData grid = new GridData(GridData.FILL_BOTH);
		grid.widthHint	= 250 ;
		grid.heightHint	= 150;
		
    	deploymentListViewer = new ListViewer(composite, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
    	deploymentListViewer.getControl().setLayoutData(grid);
    	
    	DeploymentProvider provider = new DeploymentProvider();
    	deploymentListViewer.setContentProvider(provider);
    	deploymentListViewer.setLabelProvider(provider);
    	deploymentListViewer.setComparer(provider);
    	deploymentListViewer.setInput(null);
    	
    	deploymentListViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				setUserActionsPerformed(true);
				TPFDeployment newDeployment = getSelectedDeployment();
				if (newDeployment != null) preservedDeployment = null;
				updateLaunchConfigurationDialog();
			}
		});
    	
    	return deploymentListViewer.getControl();
    }
    
    /**
     * @param test
     * @return The TestSuite that is the input of the deployment list
     */
    protected TPFTestSuite fillDeploymentsList(TPFTest test) {
    	TPFTestSuite deploymentInput = null;
		if (test != null) {
			if (test instanceof TPFTestSuite) {
				deploymentInput = (TPFTestSuite)test;
			} else if (test instanceof TPFTestCase) {
				deploymentInput = ((TPFTestCase)test).getTestSuite();
			}
		}
		TestAndDeploymentTab.this.deploymentListViewer.setInput(deploymentInput);
		return deploymentInput;
    }
	
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
        TestLaunchConfigurationFacade.setTest(configuration, getContext(resourceSet));
		TestLaunchConfigurationFacade.setDefaultSourcePathProvider(configuration);
	}
	
	public TPFTest getSelectedTest() {
		TPFTest newSelectedTest = null;
		ISelection selection = testTreeViewer.getSelection();
		if (!selection.isEmpty() && selection instanceof IStructuredSelection) {
			Object obj = ((IStructuredSelection)selection).getFirstElement();
			if (obj instanceof ITestSuiteProxyNode) {
				newSelectedTest = ((ITestSuiteProxyNode)obj).getTestSuite();
			} else if (obj instanceof ITestCaseProxyNode) {
				newSelectedTest = ((ITestCaseProxyNode)obj).getTestCase();
			}
		}
		if (newSelectedTest == null || newSelectedTest.eResource() != this.selectedTestResource) {
			disposeSelectedTestResource();
		}
		this.selectedTestResource = newSelectedTest == null ? null : newSelectedTest.eResource();
		return newSelectedTest;
	}
	
	private void disposeSelectedTestResource() {
		if (this.selectedTestResource != null) {
			this.selectedTestResource.unload();
		}
	}
	
	private void setSelectedTest(TPFTest test) {
		IProxyNode proxyNode = ((TestProvider)testTreeViewer.getContentProvider())
			.getCorrespondingProxy(test);
		if (proxyNode != null) {
			testTreeViewer.setSelection(new StructuredSelection(proxyNode), /*reveal*/true);
		} else {
			testTreeViewer.setSelection(StructuredSelection.EMPTY);
		}
	}
	
	protected TPFDeployment getSelectedDeployment() {
		ISelection selection = deploymentListViewer.getSelection();
		if (!selection.isEmpty() && selection instanceof IStructuredSelection) {
			Object obj = ((IStructuredSelection)selection).getFirstElement();
			if (obj instanceof TPFDeployment) {
				return (TPFDeployment)obj;
			}
		}
		return null;
	}
	
	private void setSelectedDeployment(TPFDeployment deployment) {
		if (deployment == null) {
			deploymentListViewer.setSelection(StructuredSelection.EMPTY);
			return;
		}
		// Because the list viewer's setSelection does not use the list viewer's comparer
		// for finding the element to select, we have to do this ourselves.
		List list = deploymentListViewer.getList();
		list.deselectAll();
		if (deployment == null) return;
		int n = list.getItemCount();
		IElementComparer comparer = deploymentListViewer.getComparer();
		for (int i = 0; i < n; i++) {
			if (comparer.equals(deployment, deploymentListViewer.getElementAt(i))) {
				list.select(i);
				return;
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	public void initializeFrom(ILaunchConfiguration configuration) {
        try {
        	TPFTest test = TestLaunchConfigurationFacade.getTest(configuration, resourceSet);
        	if (test != null) {
        		preservedTest = test;
        		setSelectedTest(test);
        		fillDeploymentsList(getSelectedTest()); // getSelectedTest may return null even if test is not null, when the test is not selectable in the current launch mode
        		TPFDeployment depl = DeploymentLaunchConfigurationFacade.getDeployment(configuration, resourceSet);
        		preservedDeployment = depl;
        		setSelectedDeployment(depl);
        	} else {
        		testTreeViewer.setSelection(new StructuredSelection());
        		deploymentListViewer.setSelection(new StructuredSelection());
        	}
        	hasTestChanged = false;
        	setUserActionsPerformed(false);
        } catch (CoreException e) {
            UiPlugin.logError(e);
        }
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
        TPFTest test = this.getSelectedTest();
        if (test == null) test = preservedTest;
        TestLaunchConfigurationFacade.setTest(configuration, test);
        
        TPFDeployment depl = this.getSelectedDeployment();
        if (depl == null) depl = preservedDeployment;
        DeploymentLaunchConfigurationFacade.setDeployment(configuration, depl);
        
        if (hasTestChanged) {
        	// Reset the Test Log configuration to default values (bugzilla_103492)
        	ExecutionHistoryLaunchConfigurationFacade.setUseDefaults(configuration, true);
        	hasTestChanged = false;
        }
	}

	public String getName() {
		return LaunchMessages.TestAndDeploymentTab_testTabName;
	}
	
	public Image getImage() {
		return TestUIImages.INSTANCE.getImage(TestUIImages.IMG_LCONFIG_TEST_TAB);
	}

    /*
     * @see org.eclipse.debug.ui.ILaunchConfigurationTab#isValid(org.eclipse.debug.core.ILaunchConfiguration)
     */
    public boolean isValid(final ILaunchConfiguration launchConfig) {
        //- Sets the state of the Run button
    	setMessage(null);
        setErrorMessage(null);
        TPFTest test;
        boolean exceptionCaught = false;
        try {
            test = TestLaunchConfigurationFacade.getTest(launchConfig, resourceSet);
        } catch (CoreException e) {
            UiPlugin.logError(e);
            test = null;
            exceptionCaught = true;
        }
        if (test == null) {
        	if (!getUserActionsPerformed()) {
        		setMessage(LaunchMessages.TestAndDeploymentTab_selectTestMessage);
        	} else if (!exceptionCaught) {
                setErrorMessage(LaunchMessages._ERROR_TestAndDeploymentTab_noTestError);
            } else {
                setErrorMessage(LaunchMessages._ERROR_TestAndDeploymentTab_noTestException);
            }
            return false;
        } else if (!LaunchConfigurationExtensionsManager.getInstance().isLaunchableElement(test, mode, launchConfigurationType)) {
        	setErrorMessage(LaunchMessages._ERROR_TestAndDeploymentTab_unsupportedMode);
        	return false;
        }
        
        if (deploymentListViewer.getList().getItemCount() == 0) {
            setErrorMessage(LaunchMessages._ERROR_TestAndDeploymentTab_noDeploymentAvailable);
            return false;
        }
        TPFDeployment depl;
        exceptionCaught = false;
        try {
            depl = DeploymentLaunchConfigurationFacade.getDeployment(launchConfig, resourceSet);
        } catch (CoreException e1) {
            UiPlugin.logError(e1);
            depl = null;
            exceptionCaught = true;
        }
        if (depl == null) {
        	if (!exceptionCaught) {
                setErrorMessage(LaunchMessages._ERROR_TestAndDeploymentTab_noDeploymentError);
            } else {
                setErrorMessage(LaunchMessages._ERROR_TestAndDeploymentTab_noDeploymentException);
            }
            return false;
        }
        // Invoke an extension point's validator
        lastValidationDiagnostic = null;
        final ITestLaunchConfigurationValidator validator = LaunchConfigurationExtensionsManager.getInstance().getLaunchConfigurationValidator(test);
        BusyIndicator.showWhile(Display.getCurrent(), new Runnable() {
            public void run() {
                if (validator != null) {
                    lastValidationDiagnostic = validator.validate(launchConfig, resourceSet);
                }
            }
        });
        if (lastValidationDiagnostic != null && lastValidationDiagnostic.getSeverity() < Diagnostic.OK) {
            //- The diagnostic is a warning or error
            setErrorMessage(lastValidationDiagnostic.getMessage());
            return lastValidationDiagnostic.getSeverity() != Diagnostic.ERROR;
        }
        return true;
    }

    /**
     * @param resourceSet The resource set used to load the EMF objects.
     * @return the first test element from the workbench selection
     */
    public static TPFTest getContext(ResourceSet resourceSet) {
        try {
            IWorkbenchPage page = UiPlugin.getInstance().getWorkbench().getActiveWorkbenchWindow().getActivePage();
            if (page != null) {
                ISelection selection = page.getSelection();
                if (selection instanceof IStructuredSelection) {
                    IStructuredSelection ss = (IStructuredSelection) selection;
                    Object obj = ss.getFirstElement();
                    if (obj != null) {
                    	
                    	if(obj instanceof ITestSuiteProxyNode){
                    		return ((ITestSuiteProxyNode)(obj)).getTestSuite();
                    	}
                    	
                        if (obj instanceof TPFTest) {
                            return (TPFTest)obj;
                        }
                        
                        if (obj instanceof IFile) {
                            IFile file = (IFile) obj;
                            if (ICommonConstants.TEST_SUITE_FILE_EXTENSION.equals(file.getFileExtension())) {
                            	return loadTestSuite(file, resourceSet);
                            }
                        }
                    }
                }
            }
        } catch (Throwable t) {
            UiPlugin.logError(t);
        }
        return null;
    }

	private static TPFTestSuite loadTestSuite(IFile resource, ResourceSet resourceSet) {
		EObject[] eobjs = EMFUtil.load(resourceSet, resource);
		for (int i = 0; i < eobjs.length; i++) {
			if (eobjs[i] instanceof TPFTestSuite) {
				return (TPFTestSuite)eobjs[i];
			}
		}
		return null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#activated(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void activated(ILaunchConfigurationWorkingCopy workingCopy) {
		// This tab's content is not dependent on other tabs' content. Nothing
		// to do
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#deactivated(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void deactivated(ILaunchConfigurationWorkingCopy workingCopy) {
		// Everything is applied as soon as selected. Nothing to do
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#dispose()
	 */
	public void dispose() {
		disposeSelectedTestResource();
		super.dispose();
	}
	
    private void setUserActionsPerformed(boolean state) {
    	if (tabGroup != null) {
    		tabGroup.setUserActionsPerformed(state);
    	}
    }

    private boolean getUserActionsPerformed() {
    	if (tabGroup != null) {
    		return tabGroup.getUserActionsPerformed();
    	}
    	return true;
    }

}
