/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventUtil.java,v 1.8 2010/05/28 14:17:03 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.model;

import java.util.Date;
import java.util.List;

import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.test.ui.UiPlugin;

import com.ibm.icu.text.DateFormat;
import com.ibm.icu.util.TimeZone;

/**
 * <p>Event utilities.</p>
 * 
 *  
 * @author  Marcelo Paternostro
 * @author  Paul Slauenwhite
 * @version May 28, 2010
 * @since   February 1, 2005
 */
public class EventUtil{
	
	/**
	 * <p>Resolves the formatted time string for the positive millisecond 
	 * time stamp based on the default time zone.</p>
	 * 
	 * <p>The format of the formatted time string is based on the {@link DateFormat#MEDIUM} date
	 * style and the {@link DateFormat#LONG} time style.<p>
	 * 
	 * <p>If the millisecond time stamp zero or negative, an empty string is returned.</p>
	 * 
	 * @param timestamp The positive millisecond time stamp.
	 * @return The formatted time string of the millisecond time stamp, otherwise an empty string.
	 * @see TimeZone#getDefault()
	 * @see DateFormat
	 * @see DateFormat#MEDIUM
	 * @see DateFormat#LONG
	 * @see #getTime(long, TimeZone)
	 */
	public static String getTime(long timestamp){
		return (getTime(timestamp, TimeZone.getDefault()));
	}
	
	/**
	 * <p>Resolves the formatted time string for the positive millisecond 
	 * time stamp based on the time zone.</p>
	 * 
	 * <p>The format of the formatted time string is based on the {@link DateFormat#MEDIUM} date
	 * style and the {@link DateFormat#LONG} time style.<p>
	 * 
	 * <p>If the millisecond time stamp zero or negative, an empty string is returned.</p>
	 * 
	 * @param timestamp The positive millisecond time stamp.
	 * @param timeZone The time zone of the positive millisecond time stamp. 
	 * @return The formatted time string of the millisecond time stamp, otherwise an empty string.
	 * @see TimeZone#getDefault()
	 * @see DateFormat
	 * @see DateFormat#MEDIUM
	 * @see DateFormat#LONG
	 */
	public static String getTime(long timestamp, TimeZone timeZone){
		
		if(timestamp > 0){
			
			try{
				
				DateFormat format = DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.LONG);
				format.setTimeZone(timeZone);
				
				return (format.format(new Date(timestamp)));
			}
			catch (RuntimeException e){
				UiPlugin.logError(e);
			}
		}
		
		return ""; //$NON-NLS-1$	
	}

	/**
	 * <p>Resolves the formatted time string for the positive delta between the start and end millisecond 
	 * time stamps based on the Greenwich Mean Time (GMT) time zone.</p>
	 * 
	 * <p>The format of the formatted time string is based on the {@link DateFormat#MEDIUM} date
	 * style and the {@link DateFormat#LONG} time style.<p>
	 * 
	 * <p>If the start and end millisecond time stamps are zero or negative or the start millisecond time stamp is 
	 * greater than the end millisecond time stamp, an empty string is returned.</p>
	 * 
	 * @param startTimeStamp The start positive millisecond time stamp.
	 * @param endTimeStamp The end positive millisecond time stamp.
	 * @return The formatted time string of the delta between the start and end millisecond time stamps, otherwise an empty string.
	 * @see DateFormat
	 * @see DateFormat#MEDIUM
	 * @see DateFormat#LONG
	 * @see #getTime(long, TimeZone)
	 */
	public static String getDuration(long startTimeStamp, long endTimeStamp) {

		if((startTimeStamp > 0) && (endTimeStamp > 0) && (endTimeStamp >= startTimeStamp)){

			//Note: The time zone is set to GMT since the time stamps are relative to GMT.
			return (getTime((endTimeStamp - startTimeStamp), TimeZone.getTimeZone("GMT")));
		}

		return ""; //$NON-NLS-1$	
	}
	
	public static long getStartTimeStamp(TPFExecutionResult executionResult){

		if(executionResult != null){

			TPFExecutionHistory executionHistory = executionResult.getExecutionHistory();

			if(executionHistory != null){

				List<TPFExecutionEvent> executionEvents = executionHistory.getExecutionEvents();

				//Return the time stamp of the first execution event with a positive time stamp:
				for (int counter = 0; counter < executionEvents.size(); counter++) {

					TPFExecutionEvent executionEvent = executionEvents.get(counter);

					if(executionEvent != null){

						long timeStamp = executionEvent.getTimestamp();

						if(timeStamp > 0){
							return timeStamp;
						}
					}
				}
			}
		}
		
		return 0;
	}

	public static long getStopTimeStamp(TPFExecutionResult executionResult){

		if(executionResult != null){

			TPFExecutionHistory executionHistory = executionResult.getExecutionHistory();

			if(executionHistory != null){

				List<TPFExecutionEvent> executionEvents = executionHistory.getExecutionEvents();

				//Return the time stamp of the last execution event with a positive time stamp:
				for (int counter = (executionEvents.size() - 1); counter > -1; counter--) {

					TPFExecutionEvent executionEvent = executionEvents.get(counter);

					if(executionEvent != null){

						long timeStamp = executionEvent.getTimestamp();

						if(timeStamp > 0){
							return timeStamp;
						}
					}
				}
			}
		}

		return 0;
	}
}
