/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogicalTestNavigatorContentProvider.java,v 1.2 2010/06/28 12:40:54 bjerome Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyManager;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.async.IProxiesRequestListener;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.async.LogicalProjectProxiesRequest;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.async.ProxiesRequest;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.ui.extension.IAssociationConstants;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.internal.extension.AssociationMappingRegistry;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.swt.graphics.Image;

/** 
 * <p>Content provider for the Logical view of the Test Navigator.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Paul Slauenwhite
 * @author  Jerome Bozier
 * @version June 28, 2010
 * @since   February 1, 2005
 */
public class LogicalTestNavigatorContentProvider extends TestNavigatorContentProvider {

    /** The file proxy manager to acces to proxies from files */
    private static FileProxyManager fileProxyManager = new FileProxyManager();
    
    private ProxiesRequests requests = new ProxiesRequests();
    
    /**
     * @return Returns the fileProxyManager.
     */
    public static IFileProxyManager getSharedFileProxyManager() {
        return fileProxyManager;
    }

    public LogicalTestNavigatorContentProvider(TestNavigator testNavigator) {
    	super(testNavigator);
    }
    
    protected IFileProxyManager getFileProxyManager() {
    	return fileProxyManager;
    }

	public boolean hasChildren(Object parentElement) {
		if (parentElement == null)
			return false;
		if(parentElement instanceof IProject) {
			return true;
		}
		return super.hasChildren(parentElement);
	}
	
	public Object[] getChildren(Object parentElement) {
		if(parentElement instanceof IProject) {
			IProject project = (IProject)parentElement;
			if (project.isAccessible()) {
				Collection types = getTypes();
				Collection extensions = getExtensions();
				ArrayList ret = new ArrayList(types.size() + extensions.size());
				LogicalProjectProxiesRequest request = requests.getRequest(project, types, extensions);
				for (Iterator it = types.iterator(); it.hasNext();) {
					String type = (String) it.next();
					IProxyNode proxy = request.getProxy(LogicalProjectProxiesRequest.TYPE_PROVIDER_PREFIX + type);
					if (proxy != null) {
						ret.add(proxy);
					}
				}
				for (Iterator it = extensions.iterator(); it.hasNext();) {
					String ext = (String) it.next();
					IProxyNode proxy = request.getProxy(LogicalProjectProxiesRequest.TEST_ASSET_EXTENSION_PREFIX + ext);
					if (proxy != null) {
						ret.add(proxy);
					}
				}
				if (request.isPending()) {
					ret.add(0, new PendingProxy(project));
				}
				return ret.toArray();
			}
			return new Object[0];
		}
		return super.getChildren(parentElement);
	}

	private Collection getTypes() {
        AssociationMappingRegistry registry = (AssociationMappingRegistry)TestUIExtension.getTestSuiteMappingRegistry();
        IAssociationMapping associationMapping = registry.getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS);
        String[] types = associationMapping.getTypes();
        List typeProviders = new ArrayList(types.length);
        for (int i = 0; i < types.length; i++) {
        	IAssociationDescriptor descriptor = associationMapping.getDefaultAssociationDescriptor(types[i]);
        	if (descriptor != null) {
                typeProviders.add(types[i]);
        	}
        }
        return typeProviders;
    }
	
	private Collection getExtensions() {
		return TestNavigator.getTestAssetGroupProxyManager().getExtensions();
	}
	
    public IGlobalProxyNodeListener getProxyNodeListener() {
    	return requests;
    }

	public void dispose() {
		requests.dispose();
	}

    class ProxiesRequests implements IProxiesRequestListener, IGlobalProxyNodeListener, IDisposable {
    	
    	private Map projectToRequestMap = Collections.synchronizedMap(new HashMap());
    	
    	public LogicalProjectProxiesRequest getRequest(IProject project, Collection types, Collection extensions) {
    		LogicalProjectProxiesRequest req = (LogicalProjectProxiesRequest)projectToRequestMap.get(project);
    		if (req == null) {
    			ArrayList requests = new ArrayList(types.size() + extensions.size());
    			for(Iterator it = types.iterator(); it.hasNext(); ) {
    				requests.add(LogicalProjectProxiesRequest.TYPE_PROVIDER_PREFIX + (String)it.next());
    			}
    			for(Iterator it = extensions.iterator(); it.hasNext(); ) {
    				requests.add(LogicalProjectProxiesRequest.TEST_ASSET_EXTENSION_PREFIX + (String)it.next());
    			}
    			req = new LogicalProjectProxiesRequest(requests, TestNavigator.getTypeProviderManager(), TestNavigator.getTestAssetGroupProxyManager(), project, testNavigator);
    			req.setPriority(getJobPriority());
    			testNavigator.getJobPool().scheduleJob(req);
                if (!req.wait(getResponseTime(), this, 2000)) {
                	projectToRequestMap.put(project, req);
                }
    		}
    		return req;
    	}
    	
		public void computationCompleted(ProxiesRequest request) {
			IProject project = ((LogicalProjectProxiesRequest)request).getProject();
			if (projectToRequestMap.containsKey(project)) {
				// A last refresh is necessary to remove the "Pending..." node.
				testNavigator.refresh(project);
			}
		}

		public void proxiesComputed(ProxiesRequest request) {
			IProject project = ((LogicalProjectProxiesRequest)request).getProject();
			testNavigator.refresh(project);
		}

		public void proxyComputed(ProxiesRequest request, Object key) {
		}

		public void nodeChanged(Object node) {
			if (node instanceof IProject) {
				LogicalProjectProxiesRequest req = (LogicalProjectProxiesRequest) projectToRequestMap.get(node);
				if (req != null) {
					// Then we need to cancel the current request. A new one will be spawned
					// when getChildren() is invoked on this project.
					if (req.cancel(1000)) {
						projectToRequestMap.remove(node);
					}
				}
			}
		}

		public void nodesChanged() {
			Object[] copyKeySet = null;
			synchronized(projectToRequestMap) {
				copyKeySet = projectToRequestMap.keySet().toArray();
			}
			for (int i=0; i<copyKeySet.length; i++) {
				nodeChanged(copyKeySet[i]);
			}
		}
		
		public void computationCancelled(ProxiesRequest request) {
			IProject project = ((LogicalProjectProxiesRequest)request).getProject();
			projectToRequestMap.remove(project);
			testNavigator.refresh(project);
		}
    	
		public void dispose() {
			// Cancel all running requests. Note that a LogicalProjectProxiesRequest may take
			// considerable time before it can honor the cancel request. So in general the
			// request may be still running for a long time after this instance has been
			// disposed.
			// If the user switches from the logical view to the resource view, and then back
			// to the logical view, this provider is disposed, and then another instance gets
			// created. As a result, two requests for the same project may be running: the old
			// one, which has been cancelled but has not stopped yet, and the new one. This
			// will look as if two jobs are doing the same thing at the same time. However what
			// really happens is that the new job is blocked on a wait until the old request 
			// completes, and then the new job automatically leverages the computation performed
			// by the old one, so there is no CPU waste. The only drawback is that the new job
			// occupies a slot in the Test navigator job pool.
			Object [] values = null;
			synchronized(projectToRequestMap) {
				values = projectToRequestMap.values().toArray();
			}
			for (int i=0; i<values.length; i++) {
				ProxiesRequest request = (ProxiesRequest)values[i];
				request.dispose();
				projectToRequestMap.values().remove(request);
			}
		}

    }
    
    private static class PendingProxy implements IProxyNode {
    	public PendingProxy(Object parent) {
    		this.parent = parent;
    	}
    	private Object parent;
    	private static IProxyNode[] NO_CHILDREN = new IProxyNode[0];
		public IProxyNode[] getChildren() {
			return NO_CHILDREN;
		}
		public Image getImage() {
			return null;
		}
		public Object getParent() {
			return parent;
		}
		public String getText() {
			return TestNavigatorMessages.NODE_PENDING;
		}
		public Object getAdapter(Class adapter) {
			return null;
		}
		public String getIdentifier() {
			return "~"; //$NON-NLS-1$
		}
		public IResource getUnderlyingResource() {
			return null;
		}
    }

}
