/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ResourceTestNavigatorContentProvider.java,v 1.1 2009/05/19 00:03:43 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyManager;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.async.FileProxiesRequest;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.async.IProxiesRequestListener;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.async.ProxiesRequest;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.ui.util.IDisposable;

/** 
 * <p>Content provider for the Resource view of the Test Navigator.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Paul Slauenwhite
 * @version May 18, 2009
 * @since   February 1, 2005
 */
public class ResourceTestNavigatorContentProvider extends TestNavigatorContentProvider {

    /** The file proxy manager to acces to proxies from files */
    private static FileProxyManager fileProxyManager = new FileProxyManager();
    
    /**
     * @return Returns the fileProxyManager.
     */
    public static FileProxyManager getSharedFileProxyManager() {
        return fileProxyManager;
    }

    public ResourceTestNavigatorContentProvider(TestNavigator testNavigator) {
    	super(testNavigator);
    }
    
    private FileProxyRequests requests = new FileProxyRequests();
    
    protected IFileProxyManager getFileProxyManager() {
    	return fileProxyManager;
    }

	public Object getParent(Object element) {
		if (element instanceof IResource) {
			return ((IResource) element).getParent();
		}
		return super.getParent(element);
	}
	
	public boolean hasChildren(Object parentElement) {
		if (parentElement == null)
			return false;
		if (parentElement instanceof IContainer) {
			IContainer container = (IContainer) parentElement;
			if (!container.isAccessible())
				return false;

			IResource[] resources = null;
			try {
				resources = container.members();
			} catch (CoreException e) {
				UiPlugin.logError(e);
				return false;
			}
			if (resources.length == 0) return false;
			if (parentElement instanceof IProject) {
				return true;
			}
			for (int i = 0; i < resources.length; i++) {
				if (resources[i].getType() != IResource.FILE) {
					if (!TestNavigator.getFiltersManager().filter(resources[i]))
						return true;
				} else {
					String fileExtension = resources[i].getFileExtension();
					if (fileExtension != null) {
						if (TestNavigator.getFiltersManager().isVisibleResource(resources[i])) return true;
					}
				}
			}
			return false;
		}
		return super.hasChildren(parentElement);
	}
	
	public Object[] getChildren(Object parentElement) {
		if (parentElement instanceof IContainer) {
			return getContainerChildren(parentElement);
		} else {
			return super.getChildren(parentElement);
		}
	}

	/** returns the children of a container node (project or folder)
	 * @param parentElement
	 * @return an array of Object which cn be empty but not null
	 */
	private Object[] getContainerChildren(Object parentElement) {
		IResource[] resources = null;
		IContainer container = ((IContainer) parentElement);
        if (container.isAccessible()) {
            try {
                resources = container.members();
            } catch (CoreException e) {
                UiPlugin.logError(e);
                return new Object[0];
            }
            List children = new ArrayList(resources.length);
            FileProxiesRequest request = requests.getRequest(container, resources);
            for (int i = 0, maxi = resources.length; i < maxi; i++) {
                if (!TestNavigator.getFiltersManager().filter(resources[i])) {
                	if (resources[i].getType() != IResource.FILE) { 
                		children.add(resources[i]);
                	} else {
                        //- convert the file in something else (proxy)
                    	IFile file = (IFile)resources[i];
                        IProxyNode node = request.getProxy(file);
                        if (!TestNavigator.getFiltersManager().filter(node)) {
                            children.add(node);
                            fileProxyManager.cacheProxy(file, node);
                        }
                    }
                }
            }
            return children.toArray();
        }
        return new Object[0];
	}

    public IGlobalProxyNodeListener getProxyNodeListener() {
    	return requests;
    }

	private class FileProxyRequests implements IProxiesRequestListener, IDisposable, IGlobalProxyNodeListener {
		
	    private Map containerToRequestMap = Collections.synchronizedMap(new HashMap());
	    
	    public FileProxiesRequest getRequest(IContainer container, IResource[] members) {
	    	FileProxiesRequest request = (FileProxiesRequest)containerToRequestMap.get(container);
	    	if (request == null) {
	    		request = newRequest(container, members);
	    	}
	    	return request;
	    }
	    
	    public FileProxiesRequest newRequest(IContainer container, IResource[] members) {
	    	ArrayList filesToRequest = new ArrayList(members.length);
            for (int i = 0, maxi = members.length; i < maxi; i++) {
            	if (members[i].getType() == IResource.FILE) {
            		if (!TestNavigator.getFiltersManager().filter(members[i])) {
                    	filesToRequest.add(members[i]);
                    }
                }
            }
            if (filesToRequest.isEmpty()) return null;
            FileProxiesRequest req = new FileProxiesRequest(filesToRequest, fileProxyManager, container, testNavigator);
            req.setPriority(getJobPriority());
            testNavigator.getJobPool().scheduleJob(req);
            if (!req.wait(getResponseTime(), this, 2000)) {
            	containerToRequestMap.put(container, req);
            }
            return req;
	    }
	    
		public void proxyComputed(final ProxiesRequest request, Object key) {
		}
	
		public void computationCompleted(ProxiesRequest request) {
			IContainer parent = ((FileProxiesRequest)request).getParent();
			if (containerToRequestMap.containsKey(parent)) {
				testNavigator.refresh(parent);
			}
		}

		public void dispose() {
			// Cancel all running requests. As the atomic action of FileProxiesRequest is
			// the computation of a single file proxy, all requests should honor the cancel
			// request quite quickly. However we don't wait them to complete (i.e. they may
			// be still running while this instance has been disposed).
			for (Iterator it = containerToRequestMap.values().iterator(); it.hasNext();) {
				FileProxiesRequest request = (FileProxiesRequest) it.next();
				request.dispose();
				it.remove();
			}
		}

		public void proxiesComputed(ProxiesRequest request) {
			testNavigator.refresh(((FileProxiesRequest)request).getParent());
		}

		public void nodeChanged(Object node) {
			if (node instanceof IContainer) {
				FileProxiesRequest req = (FileProxiesRequest) containerToRequestMap.remove(node);
				if (req != null) {
					// Then we need to cancel the current request. A new one will be spawned
					// when getChildren() is invoked on this project.
					if (req.cancel(1000)) {
						containerToRequestMap.remove(node);
					}
				}
			}
		}
		
		public void nodesChanged() {
			for (Iterator it = containerToRequestMap.keySet().iterator(); it.hasNext();) {
				nodeChanged(it.next());
			}
		}
		
		public void computationCancelled(ProxiesRequest request) {
			IContainer parent = ((FileProxiesRequest)request).getParent();
			containerToRequestMap.remove(parent);
			testNavigator.refresh(parent);
		}
    	
	}
	
	public void dispose() {
		requests.dispose();
	}
}
