/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestNavigator.java,v 1.45 2010/11/30 13:12:59 bjerome Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.editor.BaseEditorPart;
import org.eclipse.hyades.test.ui.internal.model.ResourceUtil.ResourceAdaptable;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigatorContentProvider.IGlobalProxyNodeListener;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigatorContentProvider.IProxyGroup;
import org.eclipse.hyades.test.ui.internal.navigator.action.TestNavigatorActionGroup;
import org.eclipse.hyades.test.ui.internal.navigator.action.ToggleViewAction;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileFactoryManager;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyNodeCache;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.TestAssetGroupProxyManager;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.TypeProviderManager;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.hyades.test.ui.internal.util.FixedSizeJobPool;
import org.eclipse.hyades.test.ui.navigator.CMNNamedElementProxyNode;
import org.eclipse.hyades.test.ui.navigator.EObjectProxyNode;
import org.eclipse.hyades.test.ui.navigator.FileProxyNode;
import org.eclipse.hyades.test.ui.navigator.IExecutionResultProxyNode;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxyNodeListener;
import org.eclipse.hyades.test.ui.navigator.ITestNavigator;
import org.eclipse.hyades.test.ui.navigator.ITypeProviderContext;
import org.eclipse.hyades.test.ui.navigator.ITypedElementProxyNode;
import org.eclipse.hyades.ui.extension.IAssociationConstants;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.internal.extension.AssociationMappingRegistry;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.navigator.NavigatorViewerSorter;
import org.eclipse.hyades.ui.internal.navigator.TreeNavigator;
import org.eclipse.hyades.ui.internal.navigator.TreeNavigatorFrameSource;
import org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater;
import org.eclipse.hyades.ui.internal.provider.ResourceChangeUpdaterProvider;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.ActionGroup;
import org.eclipse.ui.contexts.IContextService;
import org.eclipse.ui.plugin.AbstractUIPlugin;

/**
 * <p>Concrete implementation of the
 * {@link org.eclipse.hyades.ui.internal.navigator.TreeNavigator} 
 * for test assets.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Jerome Gout
 * @author  Paul E. Slauenwhite
 * @author  Paul Klicnik
 * @author  Kurtis Martin
 * @author  Jerome Bozier
 * @version November 30, 2010
 * @since   February 1, 2005
 */
public class TestNavigator extends TreeNavigator implements ISelectionListener, IGlobalProxyNodeListener, ITestNavigator {
    /**
     * @deprecated use {@link ITestNavigator#ID} instead.
     */
    public static final String ID = ITestNavigator.ID;
    public static final int VIEW_ID_RESOURCE = 0;

    public static final int VIEW_ID_LOGICAL = 1;

    /**
     * The size of the background job pool. This is the maximal number of background
     * jobs that can run at the same time.
     */
    public static final int BACKGROUND_JOB_POOL_SIZE = 3;
   
    private int currentViewIndex;

    /**
     * Map with the instances of this class by the workbench window. It is
     * necessary to keep this cache because adpaters and content providers may
     * need to check properties of a test navigator - such as if the folders are
     * visible.
     */
    private static Map testNavigatorByWorkbenchWindow;

    /**
     * Extensions manager. The extensions managed by this static field is
     * org.eclipse.hyades.test.ui.testNavigatorFilter
     */
    private static TestNavigatorFilterExtensionsManager filtersManager;

    /**
     * File proxy Factory manager. this is an extension point manager for
     * org.eclipse.hyades.test.ui.testNavigatorFileProxyFactory
     */
    private static FileFactoryManager fileFactoryManager;

    /**
     * Type Provider mananger. This instance is responsible to retrieve
     * instances of ITypeProviderProxyNode from a project
     */
    private static TypeProviderManager typeProviderManager;

    /**
     * Test asset group proxy manager. This instance is responsible to retrieve
     * instance of ITestAssetGroupProxyNode from a project
     */
    private static TestAssetGroupProxyManager testAssetGroupProxyManager;
    
	private static TypeProviderContext typeProviderContext;

    /** actions to switch between both logical and resource views. */
    private ToggleViewAction[] viewActions;

    /**
     * Current content provider for the Test Navigator.
     */
    private TestNavigatorContentProvider currentContentProvider;

    // - constants are used in the IDialogSettings
    protected final static String SET_SHOW_FOLDERS = "SHOW_FOLDERS"; //$NON-NLS-1$

    protected final static String SET_SHOW_EOBJECT_CHILDREN = "SHOW_EOBJECT_CHILDREN"; //$NON-NLS-1$

    protected final static String SET_SORTER_CRITERIA = "SORTER_CRITERIA"; //$NON-NLS-1$

    protected final static String DIALOGSTORE_HTN_VIEW = "TestNavigator.lastview"; //$NON-NLS-1$

    protected final static int CP_FOLDER = 0;

    protected final static int CP_LOGICAL = 1;
    
    private List rootLogicalFolders;

    private boolean showingFolders = true;

    private boolean showingEObjectChildren = true;

    private ResourceChangeUpdaterProvider resourceChangeUpdaterProvider;
    
    private FixedSizeJobPool jobPool;
    
    private ResourceTestNavigatorContentProvider resourceProv = null;
    private LogicalTestNavigatorContentProvider logicalProv = null;
    private TestDecoratingLabelProvider labelProv = null;
    
    /**
     * Constructor for TestNavigator
     */
    public TestNavigator() {
        super(ITestNavigator.ID);

        viewActions = new ToggleViewAction[] { new ToggleViewAction(this, VIEW_ID_RESOURCE), new ToggleViewAction(this, VIEW_ID_LOGICAL) };
    }

    /**
     * Adds a test navigator to the <code>testNavigatorByWorkbenchWindow</code>
     * map.
     * 
     * @param testNavigator
     */
    private static void addTestNavigator(TestNavigator testNavigator) {
        if (testNavigatorByWorkbenchWindow == null)
            testNavigatorByWorkbenchWindow = new HashMap();
        testNavigatorByWorkbenchWindow.put(testNavigator.getSite().getWorkbenchWindow(), testNavigator);
    }

    /**
     * Remvoes a test navigator to the
     * <code>testNavigatorByWorkbenchWindow</code> map.
     * 
     * @param testNavigator
     */
    private static void removeTestNavigator(TestNavigator testNavigator) {
        if (testNavigatorByWorkbenchWindow != null) {
            testNavigatorByWorkbenchWindow.remove(testNavigator.getSite().getWorkbenchWindow());
            if (testNavigatorByWorkbenchWindow.isEmpty())
                testNavigatorByWorkbenchWindow = null;
        }
    }

    /**
     * Returns the test navigator opened in a given workbench window or
     * <code>null</code> if there isn't any.
     * 
     * @param workbenchWindow
     * @return TestNavigator
     */
    public static TestNavigator getTestNavigator(IWorkbenchWindow workbenchWindow) {
        if ((workbenchWindow != null) && (testNavigatorByWorkbenchWindow != null))
            return (TestNavigator) testNavigatorByWorkbenchWindow.get(workbenchWindow);
        return null;
    }

    /**
     * Give access to the filters manager of the test navigator
     * 
     * @return filters manager
     */
    public static TestNavigatorFilterExtensionsManager getFiltersManager() {
        if (filtersManager == null) {
            filtersManager = new TestNavigatorFilterExtensionsManager();
        }
        return filtersManager;
    }

    /**
     * @return Returns the fileFactoryManager.
     */
    public static FileFactoryManager getFileFactoryManager() {
        if (fileFactoryManager == null) {
            fileFactoryManager = new FileFactoryManager();
        }
        return fileFactoryManager;
    }

    /**
     * Returns the {@link IFileProxyManager} of current 
     * Test Navigator view provider (Resource or Logical).
     * 
     * @return The {@link IFileProxyManager} of current Test Navigator view provider (Resource or Logical).
     */
    public static IFileProxyManager getFileProxyManager() {
    	
    	//Resolve the file proxy manager:
        IFileProxyManager fileProxyManager = null;
        IWorkbench workbench = PlatformUI.getWorkbench();
        
        if(workbench != null){
        	
        	IWorkbenchWindow activeWorkbenchWindow = workbench.getActiveWorkbenchWindow();
        	
        	if(activeWorkbenchWindow != null){
        		
        		TestNavigator testNavigator = TestNavigator.getTestNavigator(activeWorkbenchWindow);
        		
        		if(testNavigator != null){
        			fileProxyManager = testNavigator.getCurrentFileProxyManager();
                }
        	}
        }

        return fileProxyManager;
    }
    
    public IFileProxyManager getCurrentFileProxyManager() {
    	return currentContentProvider.getFileProxyManager();
    }

    /**
     * @return Returns the typeProviderProxyManager.
     */
    public synchronized static TypeProviderManager getTypeProviderManager() {
        if (typeProviderManager == null) {
            typeProviderManager = new TypeProviderManager(getTypeProviderContext());
        }
        return typeProviderManager;
    }
    
    private static class AllInstancesRefresher implements IProxyNodeListener {
        public void nodeChanged(Object node) {
            for(Iterator it = TestNavigator.getAllInstancesIterator(); it.hasNext();) {
                ((TestNavigator)it.next()).nodeChanged(node);
            }
        }
    }
    
    /**
     * This class is responsible to give the correct context to the used ITypeProvider (DefaultTypeProvider).
     * It stores the file proxy manager of the logical view to be used by the ITypeProvider,
     * and the IProxyNodeListener unique instance.
     * @author jgout
     */
    private static class TypeProviderContext implements ITypeProviderContext {
		private IProxyNodeListener refresher;
        public IFileProxyManager getFileProxyManager() {
            return LogicalTestNavigatorContentProvider.getSharedFileProxyManager();
        }
		public IProxyNodeListener getProxyNodeListener() {
			if (refresher == null) {
				refresher = new AllInstancesRefresher();
			}
			return refresher;
		}
		public boolean isStaticView() {
			return false;
		}
    }
    
    public static ITypeProviderContext getTypeProviderContext() {
    	if (typeProviderContext == null) {
    		typeProviderContext = new TypeProviderContext();
    	}
    	return typeProviderContext;
    }

    public synchronized static TestAssetGroupProxyManager getTestAssetGroupProxyManager() {
        if (testAssetGroupProxyManager == null) {
            testAssetGroupProxyManager = new TestAssetGroupProxyManager(getTypeProviderContext());
        }
        return testAssetGroupProxyManager;
    }

    /**
     * @see org.eclipse.ui.IWorkbenchPart#dispose()
     */
    public void dispose() {
    	if (jobPool != null) {
    		// Disposing the job pool cancels all pending jobs that have net been yet
    		// scheduled.
    		jobPool.dispose();
    	}
    	// Cancel all "TestNavigatorJob"s, i.e. jobs that belongs the "family" represented by
    	// this instance. This cancels all jobs that have already been scheduled.
    	Job.getJobManager().cancel(this);
    	
        if (rootLogicalFolders != null)
            rootLogicalFolders.clear();
        removeTestNavigator(this);

        FileProxyNodeCache.getInstance().removeResourceListener(resourceChangeUpdaterProvider);
        resourceChangeUpdaterProvider.getResourceChangeUpdater().dispose();
        resourceChangeUpdaterProvider.dispose();
        resourceChangeUpdaterProvider = null;

        if(testAssetGroupProxyManager != null){
            testAssetGroupProxyManager.dispose();
            testAssetGroupProxyManager=null;
        }
        
        if(typeProviderManager != null){
            typeProviderManager.dispose();
            typeProviderManager=null;
        }
        super.dispose();
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#getStoreSection()
     */
    public String getStoreSection() {
        return "TestNavigator"; //$NON-NLS-1$
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#initSettings()
     */
    protected void initSettings() {
        setShowingFolders(true);
        setShowingEObjectChildren(false);
    }

    /**
     * @see org.eclipse.ui.IViewPart#init(org.eclipse.ui.IViewSite,
     *      org.eclipse.ui.IMemento)
     */
    public void init(IViewSite site, IMemento memento) throws PartInitException {

    	super.init(site, memento);
        
    	addTestNavigator(this);

    	//Add the Test Navigator context to the context service:
        ((IContextService)(this.getSite().getService(IContextService.class))).activateContext(ITestNavigator.CONTEXT_ID);
    }

    private void adjustCurrentProviderContext(boolean isLoading) {
		if (currentContentProvider != null) {
			if (isLoading) {
				currentContentProvider.setContext(TestNavigatorContentProvider.CONTEXT_LOADING);
			} else {
				currentContentProvider.setContext(TestNavigatorContentProvider.CONTEXT_INTERACTIVE);
			}
		}
	}

	/**
     * Returns the file associated to a given object that is located in this
     * navigator.
     * 
     * @param object
     * @return IFile
     */
    public IFile getFile(Object object) {
        if (object instanceof IFile)
            return (IFile) object;

        if (object instanceof FileProxyNode) {
            FileProxyNode proxy = (FileProxyNode) object;
            return proxy.getFile();
        }

        if (object instanceof EObjectProxyNode) {
            EObjectProxyNode proxy = (EObjectProxyNode) object;
            Object parent = getCurrentFileProxyManager().getParent(proxy);
            if (parent instanceof IContainer) {
                // - this proxy node is a root object (place instead of a file
                // node)
                URI uri = proxy.getOriginatorURI();
                String path = uri.trimFragment().toString().substring("platform:/resource".length()); //$NON-NLS-1$
                Path filePath = new Path(path);
                return (IFile) ResourcesPlugin.getWorkspace().getRoot().findMember(filePath);
            }
        }
        if (object instanceof Resource)
            return EMFUtil.getWorkspaceFile((Resource) object);

        if (object instanceof EObject) {
            EObject eObject = (EObject) object;
            if (eObject.eContainer() != null)
                return null;

            Resource resource = eObject.eResource();
            if (resource == null)
                return null;
            return EMFUtil.getWorkspaceFile(resource);
        }

        if (object instanceof IAdaptable) {
            IFile file = (IFile) ((IAdaptable) object).getAdapter(IFile.class);
            if (file != null)
                return file;
        }

        return null;
    }

    /**
     * Sets whether the resource navigator should monitor resource changes.
     * 
     * @param enabled
     */
    public void setMonitoringResourceChange(boolean enabled) {
        if (resourceChangeUpdaterProvider != null)
            resourceChangeUpdaterProvider.getResourceChangeUpdater().setActive(enabled);
    }

    /**
     * Returns whether the test navigator is monitoring resource changes
     */
    public boolean isMonitoringResourceChange() {
        if (resourceChangeUpdaterProvider != null)
            resourceChangeUpdaterProvider.getResourceChangeUpdater().isActive();
        return false;
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#loadSettings()
     */
    protected void loadSettings() {
        super.loadSettings();
        // setShowingFolders(getSettings().getBoolean(SET_SHOW_FOLDERS));
        setShowingEObjectChildren(getSettings().getBoolean(SET_SHOW_EOBJECT_CHILDREN));
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.TreeNavigator#adjustTreeViewer(org.eclipse.jface.viewers.TreeViewer)
     */
    protected void adjustTreeViewer(TreeViewer treeViewer) {
        try {
            currentViewIndex = getSettings().getInt(DIALOGSTORE_HTN_VIEW);
            if (currentViewIndex < 0 || currentViewIndex > 1) {
                currentViewIndex = VIEW_ID_RESOURCE;
            }
        } catch (NumberFormatException e) {
            currentViewIndex = VIEW_ID_RESOURCE;
        }
        int criteria = NavigatorViewerSorter.NAME;
        try {
            criteria = getSettings().getInt(SET_SORTER_CRITERIA);
        } catch (NumberFormatException e) {
        }
        NavigatorViewerSorter viewerSorter = new NavigatorViewerSorter(this, criteria) {
        	
			/**
			 * <p>A {@link Pattern} for matching an execution result file name.</p>
			 * 
			 * <p>The regular expression used to match the execution result file name is:</p>
			 * 
			 * <p><code>&lt;One or more non-whitespace or whitespace characters&gt;_&lt;13 digits&gt;</code></p>
			 * 
			 * <p><b>Note:</b> Time stamps with 13 digits range from 1000000000000 - 9999999999999 (Sep 8, 2001 9:46:40 PM EDT - 
			 * Nov 20, 2286 12:46:39 PM EST), which is within the lifespan of the Hyades/TPTP project (inception: Sep 29, 2003) 
			 * and subsequently all consuming products.</p>
			 */
			private final Pattern EXECUTION_RESULT_FILE_NAME_PATTERN = Pattern.compile("(\\S|\\s)+_\\d{13}"); //$NON-NLS-1$
			
			/**
			 * <p>An artificial time stamp representing the first time stamp to the existing name (<code>&lt;execution result file name&gt;_1000000000000.execution</code>)
			 * for correct lexicographical sorting.</p>
			 * 
			 * <p><b>Note:</b> Time stamps with 13 digits range from 1000000000000 - 9999999999999 (Sep 8, 2001 9:46:40 PM EDT - 
			 * Nov 20, 2286 12:46:39 PM EST), which is within the lifespan of the Hyades/TPTP project (inception: Sep 29, 2003) 
			 * and subsequently all consuming products.</p>
			 */
			private final String ARTIFICIAL_TIME_STAMP = "_1000000000000"; //$NON-NLS-1$

            protected int compareCategories(Object e1, Object e2) {
            	if (e1 instanceof IProxyGroup) {
            		if (e2 instanceof IProxyGroup) {
            			return ((IProxyGroup)e1).getOrder() - ((IProxyGroup)e2).getOrder();
            		}
            		return -1;
            	}
            	if (e2 instanceof IProxyGroup) {
            		return 1;
            	}
				return super.compareCategories(e1, e2);
			}

			public int category(Object element) {
            	if (element instanceof IResource) return super.category(element);
				if (element instanceof IAdaptable) {
					IResource res = (IResource)((IAdaptable)element).getAdapter(IResource.class);
					if (res != null) {
						return super.category(res);
					}
				}
				return super.category(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.hyades.ui.internal.navigator.NavigatorViewerSorter#getValue(org.eclipse.jface.viewers.Viewer, java.lang.Object)
			 */
			protected String getValue(Viewer viewer, Object object) {
				
				String value = super.getValue(viewer, object);

				//Work-around: An execution result without a time stamp appended to the file name (see defect #308365) 
				//is the first execution result with the same name in the container and require an artificial time stamp 
				//representing the first time stamp (1000000000000) with the same number of digits (13) for correct 
				//lexicographical sorting (see HyadesViewerSorter.compareValues(String, String)).
				
				//If the object is a IExecutionResultProxyNode, append an artificial time stamp representing the first 
				//time stamp to the existing name (<execution result file name>_1000000000000.execution) for correct 
				//lexicographical sorting:
				if(object instanceof IExecutionResultProxyNode){

					int resourceValueSeparatorIndex = value.indexOf(RESOURCE_VALUE_DELIMITER);
					
					if(resourceValueSeparatorIndex == -1){
						
						if(!EXECUTION_RESULT_FILE_NAME_PATTERN.matcher(value).matches()){							
							value = (value.concat(ARTIFICIAL_TIME_STAMP)); 
						}						
					}

					//Case 1: <execution result file name>_+_execution (NAME category):
					else if(getCriteria() == NAME){

						String name = value.substring(0, resourceValueSeparatorIndex);

						if(!EXECUTION_RESULT_FILE_NAME_PATTERN.matcher(name).matches()){							
							value = (name.concat(ARTIFICIAL_TIME_STAMP).concat(value.substring(resourceValueSeparatorIndex))); 
						}	
					}

					//Case 2: execution_+_<execution result file name> (TYPE category):
					else{						

						String name = value.substring(resourceValueSeparatorIndex + RESOURCE_VALUE_DELIMITER.length());

						if(!EXECUTION_RESULT_FILE_NAME_PATTERN.matcher(name).matches()){							
							value = (value.concat(ARTIFICIAL_TIME_STAMP));
						}
					}					
				}
				
				//If the object is a ITypedElementProxyNode, prepend (TYPE category)/append (NAME category) the proxy 
				//node type to the existing type (if exists) to increase the sorting precision (for example, JUnit, 
				//JUnit Plug-in, and URL test types all have the *.testsuite file extension):
				if (object instanceof ITypedElementProxyNode) {

					String type = ((ITypedElementProxyNode)(object)).getType();

					if(type != null){

						//Case 1: <resource name>_+_[<file extension (IFile only)>] (NAME category):
						if(getCriteria() == NAME){

							if(value.endsWith(RESOURCE_VALUE_DELIMITER)){
								value = (value.concat(type));								
							}
							else{
								value = (value.concat(RESOURCE_VALUE_DELIMITER).concat(type));		
							}
						}

						//Case 2: [<file extension (IFile only)>]_+_<resource name> (TYPE category):
						else{						

							if(value.startsWith(RESOURCE_VALUE_DELIMITER)){
								value = (type.concat(value));							
							}
							else{
								value = (type.concat(RESOURCE_VALUE_DELIMITER).concat(value));		
							}
						}
					}
				}

				return value;
            }
        };
        treeViewer.setSorter(viewerSorter);

        ViewerFilter viewerFilter = new ViewerFilter() {
            public boolean select(Viewer viewer, Object parentElement, Object element) {
                if (!isShowingEObjectChildren())
                    return (!(parentElement instanceof EObject));

                return true;
            }
        };
        treeViewer.addFilter(viewerFilter);

        resourceChangeUpdaterProvider = new ResourceChangeUpdaterProvider.UIUpdaterProvider(false);
        FileProxyNodeCache.getInstance().addResourceListener(resourceChangeUpdaterProvider);
        IResourceChangeUpdater resourceChangeUpdater = new TestResourceChangeUpdater(this);
        resourceChangeUpdaterProvider.setResourceChangeUpdater(resourceChangeUpdater);

        updateTestNavigatorProvider(treeViewer);
    }

    private void updateTestNavigatorProvider(TreeViewer treeViewer) {
    	
    	if (currentViewIndex == VIEW_ID_RESOURCE) {
    		if (resourceProv == null) {
    			resourceProv = new ResourceTestNavigatorContentProvider(this);
    		}
    		currentContentProvider = resourceProv;
    	} else {
    		if (logicalProv == null) {
    			logicalProv = new LogicalTestNavigatorContentProvider(this);
    		}
    		currentContentProvider = logicalProv;
    	}
    	
    	adjustCurrentProviderContext(true);
        
    	treeViewer.setContentProvider(currentContentProvider);
        if (labelProv == null) {
        	labelProv = new TestDecoratingLabelProvider(new TestNavigatorLabelProvider(), PlatformUI.getWorkbench().getDecoratorManager().getLabelDecorator(), this);
        	treeViewer.setLabelProvider(labelProv);
        }

        adjustCurrentProviderContext(false);
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#editorActivated(org.eclipse.ui.IEditorPart)
     */
    protected boolean editorActivated(IEditorPart editor) {
        if (getViewer() == null)
            return false;

		IFile file = null;
		IEditorInput input = editor.getEditorInput();
        if (input instanceof IFileEditorInput) {
            IFileEditorInput fileInput = (IFileEditorInput) input;
            file = fileInput.getFile();
        } else {
        	 file = (IFile)input.getAdapter(IFile.class);
        }
        if(file != null) {
            selectReveal(new StructuredSelection(file));
            return true;
        }

        return false;
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#linkToEditor(org.eclipse.jface.viewers.IStructuredSelection)
     */
    protected void linkToEditor(IStructuredSelection structuredSelection) {
        IFile file = getFile(structuredSelection.getFirstElement());
        if (file != null) {
        	IWorkbenchPage page = getSite().getPage();
        	
        	// If the active editor is already open on the file which was selected in TestNav, then
        	// do nothing.
        	if(page.getActiveEditor() != null){
        		IEditorInput input = page.getActiveEditor().getEditorInput();
    			IFile editorsFile  = null;
    			if(input instanceof IFileEditorInput){
    				editorsFile = ((IFileEditorInput)input).getFile();
    			} else {
    				editorsFile = (IFile)input.getAdapter(IFile.class);
    			}
    			if(editorsFile != null && editorsFile.equals(file)){
    				return;
    			}
        	}
        	
        	IEditorPart editor = null;
        	IEditorReference[] openEditors = page.getEditorReferences();
        	try {
        		for(int i=0; i < openEditors.length; ++i){
        			IEditorInput input = ((IEditorReference)openEditors[i]).getEditorInput();
        			IFile editorsFile  = null;
        			if(input instanceof IFileEditorInput){
        				editorsFile = ((IFileEditorInput)input).getFile();
        			} else {
        				editorsFile = (IFile)input.getAdapter(IFile.class);
        			}
        			if(editorsFile != null && editorsFile.equals(file)){
        				editor = ((IEditorReference)openEditors[i]).getEditor(false);
        				break;
        			}
        		}
        	} catch(PartInitException e){
        		UiPlugin.logError(e);
        	}
        	if (editor != null) {
        		if (page.getActiveEditor() != editor) {
        			page.bringToTop(editor);
        		}
        	}
        }
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#getPlugin()
     */
    public AbstractUIPlugin getPlugin() {
        return UiPlugin.getInstance();
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#createViewerInput()
     */
    protected Object createViewerInput() {
        return ResourcesPlugin.getWorkspace().getRoot();
        // return getResourceSet();
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#createActions()
     */
    protected void createActions() {
        TestNavigatorActionGroup HTNActionGroup = new TestNavigatorActionGroup(this);
        setActionGroup(HTNActionGroup);
        // - linked HTN actions to standard workbench corresponding actions
        getViewSite().getActionBars().setGlobalActionHandler(ActionFactory.RENAME.getId(), HTNActionGroup.getRenameAction());
        getViewSite().getActionBars().setGlobalActionHandler(ActionFactory.COPY.getId(), HTNActionGroup.getCopyAction());
        getViewSite().getActionBars().setGlobalActionHandler(ActionFactory.PASTE.getId(), HTNActionGroup.getPasteAction());
        getViewSite().getActionBars().setGlobalActionHandler(ActionFactory.REFRESH.getId(), HTNActionGroup.getRefreshAction());
        getViewSite().getActionBars().setGlobalActionHandler(ActionFactory.MOVE.getId(), HTNActionGroup.getMoveAction());
        IActionBars actionBars = getViewSite().getActionBars();
        IToolBarManager viewMenu = actionBars.getToolBarManager();
        
        /* Set the enablement of the copy/paste options when the Test Navigator opens */
        if ( HTNActionGroup != null && currentViewIndex == VIEW_ID_LOGICAL ) {
        	HTNActionGroup.getPasteAction().setEnabled( false );
        	HTNActionGroup.getCopyAction().setEnabled( false );
        } else if ( HTNActionGroup != null && currentViewIndex == VIEW_ID_RESOURCE ) {
        	HTNActionGroup.getPasteAction().setEnabled( true );
        	HTNActionGroup.getCopyAction().setEnabled( true );
        }
        
        for (int i = 0; i < viewActions.length; i++) {
            ToggleViewAction action = viewActions[i];
            viewMenu.add(action);
            action.setEnabled(true);
            action.setChecked(currentViewIndex == action.getViewerIndex());
        }
        viewMenu.add(new Separator());
        // - Add actions to switch between the logical and resource view
        IMenuManager menuManager = actionBars.getMenuManager();
        menuManager.add(new Separator());
        for (int i = 0; i < viewActions.length; i++) {
            ToggleViewAction action = viewActions[i];
            menuManager.add(action);
            action.setEnabled(true);
            action.setChecked(currentViewIndex == action.getViewerIndex());
        }
        menuManager.add(new Separator());
    }

    /**
     * Returns whether the navigator is showing projects and folders.
     * 
     * @return <code>true</code> if is showing project and folders,
     *         <code>false</code> if not.
     */
    public boolean isShowingFolders() {
        return showingFolders;
    }

    /**
     * Sets whether this navigator's is showing projects and folders.
     * 
     * @param enabled
     *            <code>true</code> to enable, <code>false</code> to disable
     */
    public void setShowingFolders(boolean enabled) {
        showingFolders = enabled;

        // remember the last setting in the dialog settings
        getSettings().put(SET_SHOW_FOLDERS, showingFolders);
        if (getViewer() == null)
            return;

        getViewer().refresh();
    }

    /**
     * Returns whether the navigator is showing the children of the emf objects.
     * 
     * @return <code>true</code> if is showing the children,
     *         <code>false</code> if not.
     */
    public boolean isShowingEObjectChildren() {
        return showingEObjectChildren;
    }

    /**
     * Sets whether this navigator's is showing the children of the EMF objects.
     * 
     * @param enabled
     *            <code>true</code> to enable, <code>false</code> to disable
     */
    public void setShowingEObjectChildren(boolean enabled) {
        // showingEObjectChildren = enabled;
        // modelContentProvider.setShowEnabledChildren(enabled);
        //
        // // remember the last setting in the dialog settings
        // getSettings().put(SET_SHOW_EOBJECT_CHILDREN, showingEObjectChildren);
        // if (getViewer() == null)
        // return;
        //
        // getViewer().refresh();
        // selectionChanged(new SelectionChangedEvent(getViewer(),
        // StructuredSelection.EMPTY));
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#getStatusLineMessage(org.eclipse.jface.viewers.IStructuredSelection)
     */
    protected String getStatusLineMessage(IStructuredSelection structuredSelection) {
        if (structuredSelection.size() == 1) {
            Object selection = structuredSelection.getFirstElement();
            if (selection instanceof IResource)
                return ((IResource) selection).getFullPath().toString();

            if (selection instanceof LogicalFolder) {
                LogicalFolder logicalFolder = (LogicalFolder) selection;
                if (logicalFolder.getChildren().size() == 0)
                    return logicalFolder.getName();
                return NLS.bind(UiPluginResourceBundle.STS_LNE_LOG_FLD, (new String[] { logicalFolder.getName(), Integer.toString(logicalFolder.getChildren().size()) }));
            }

            if (selection instanceof CMNNamedElement) {
                CMNNamedElement namedElement = (CMNNamedElement) selection;
                String label = ((ILabelProvider) getViewer().getLabelProvider()).getText(namedElement);
                IFile file = EMFUtil.getWorkspaceFile(namedElement);
                String filePath = ""; //$NON-NLS-1$
                if (file != null)
                    filePath = file.getFullPath().toString();
                return NLS.bind(UiPluginResourceBundle.STS_LNE_FILE_ELEM, (new String[] { label, filePath })); 
            }

            if (selection instanceof CMNNamedElementProxyNode) {
                CMNNamedElementProxyNode proxy = (CMNNamedElementProxyNode) selection;
                if (proxy.getUnderlyingResource() != null) {
                    return NLS.bind(UiPluginResourceBundle.STS_LNE_FILE_ELEM, (new String[] { proxy.getText(), proxy.getUnderlyingResource().getFullPath().toString() })); 
                }
            }
            if (selection instanceof FileProxyNode) {
                FileProxyNode proxy = (FileProxyNode) selection;
                return NLS.bind(UiPluginResourceBundle.STS_LNE_FILE_ELEM, (new String[] { proxy.getText(), proxy.getUnderlyingResource().getFullPath().toString() })); 
            }

            if (selection instanceof Resource) {
                Resource resource = (Resource) selection;
                String label = ((ILabelProvider) getViewer().getLabelProvider()).getText(resource);
                IFile file = EMFUtil.getWorkspaceFile(resource);
                String filePath = ""; //$NON-NLS-1$
                if (file != null)
                    filePath = file.getFullPath().toString();
                return NLS.bind(UiPluginResourceBundle.STS_LNE_FILE_ELEM, (new String[] { label, filePath })); 
            }

        }

        return super.getStatusLineMessage(structuredSelection);
    }

    /**
     * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
     */
    public void doubleClick(DoubleClickEvent event) {
        if (event.getViewer() == getTreeViewer()) {
            IStructuredSelection selection = (IStructuredSelection) event.getSelection();
            Object element = selection.getFirstElement();
            if (element instanceof IContainer) {
                TreeViewer treeViewer = getTreeViewer();
                if (treeViewer.isExpandable(element))
                    treeViewer.setExpandedState(element, !treeViewer.getExpandedState(element));
            }
        }
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#convertToViewer(org.eclipse.jface.viewers.ISelection)
     */
    protected IStructuredSelection convertToViewer(ISelection selection) {
        IStructuredSelection structuredSelection = super.convertToViewer(selection);
        if (structuredSelection.isEmpty())
            return structuredSelection;

        List elements = new ArrayList(structuredSelection.size());
        for (Iterator i = structuredSelection.iterator(); i.hasNext();) {
            Object object = i.next();
            if (object instanceof IFile) {
                EObject[] eObjects = EMFUtil.getEObjects(null, (IFile) object);
                for (int j = 0, maxj = eObjects.length; j < maxj; j++) {
                    if (eObjects[j] instanceof CMNNamedElement)
                        elements.add(eObjects[j]);
                }
            } else if (object instanceof EObject) {
                EObject eObject = EMFUtil.getEObject(null, (EObject) object, false);
                if (eObject != null) {
                    if (eObject.eContainer() == null) {
                        elements.add(eObject);
                    } else if (isShowingEObjectChildren()) {
                        elements.add(eObject);
                    } else {
                        elements.add(EcoreUtil.getRootContainer(eObject));
                    }
                }
            } else
                elements.add(object);
        }

        return new StructuredSelection(elements);
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.INavigator#handleChange(int)
     */
    public void handleChange(int type) {
        if (NavigatorViewerSorter.SORTER_TYPE == type) {
            getSettings().put(SET_SORTER_CRITERIA, ((NavigatorViewerSorter) getViewer().getSorter()).getCriteria());
            ActionGroup actionGroup = getActionGroup();
            if (actionGroup != null)
                actionGroup.updateActionBars();
        }
        super.handleChange(type);
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#getInitialViewerInput()
     */
    public Object getInitialViewerInput() {
        return super.getInitialViewerInput();
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.TreeNavigator#createFrameSource()
     */
    protected TreeNavigatorFrameSource createFrameSource() {
        return new TestNavigatorFrameSource(this);
    }

    /**
     * Registers a root logical folder to this test navigator.
     * 
     * @param logicalFolder
     */
    public void registerRootLogcalFolder(LogicalFolder logicalFolder) {
        if ((logicalFolder != null) && (logicalFolder.getParent() == null)) {
            if (rootLogicalFolders == null)
                rootLogicalFolders = new ArrayList();
            rootLogicalFolders.add(logicalFolder);
        }
    }

    /**
     * Returns the root logical folder that has the given name or
     * <code>null</code> if there is no such folder.
     * 
     * @param name
     */
    public LogicalFolder getRootLogicalFolder(String name) {
        if ((rootLogicalFolders != null) && (name != null)) {
            for (Iterator i = rootLogicalFolders.iterator(); i.hasNext();) {
                LogicalFolder logicalFolder = (LogicalFolder) i.next();
                if (name.equals(logicalFolder.getName()))
                    return logicalFolder;
            }
        }
        return null;
    }

    /**
     * @see org.eclipse.hyades.ui.internal.navigator.Navigator#loadElements(org.eclipse.ui.IMemento)
     */
    protected List loadElements(IMemento memento) {
        Object[] objects = super.loadElements(memento).toArray();
        List elements = new ArrayList(objects.length);
        for (int i = 0, maxi = objects.length; i < maxi; i++) {
            if (objects[i] instanceof LogicalFolder) {
                LogicalFolder logicalFolder = (LogicalFolder) objects[i];
                Object parent = logicalFolder.getParent();
                if (parent == null) {
                    logicalFolder = getRootLogicalFolder(logicalFolder.getName());
                    if (logicalFolder != null)
                        elements.add(logicalFolder);
                } else if (parent instanceof EObject) {
                    EObject eObject = EMFUtil.getEObject(null, (EObject) parent, true);
                    if (eObject != null) {
                        LogicalFolder newLogicalFolder = new LogicalFolder(eObject, logicalFolder.getName());
                        newLogicalFolder.setData(logicalFolder.getData());
                        newLogicalFolder.setDescription(logicalFolder.getDescription());

                        elements.add(newLogicalFolder);
                    }
                } else {
                    elements.add(logicalFolder);
                }
            } else if (objects[i] instanceof EObject) {
                EObject eObject = EMFUtil.getEObject(null, (EObject) objects[i], true);
                if (eObject != null)
                    elements.add(eObject);
            } else if (objects[i] instanceof ResourceAdaptable) {
                Resource resource = ((ResourceAdaptable) objects[i]).getResource();
                if (resource != null) {
                    resource = EMFUtil.getResource(null, resource.getURI(), true);
                    if (resource != null)
                        elements.add(resource);
                }
            } else {
                elements.add(objects[i]);
            }
        }

        return elements;
    }

    /**
     * Returns the help context id for this navigator or <code>null</code> if
     * no help is provided.
     * 
     * @return String
     */
    protected String getHelpContextId() {
        return UiPlugin.getID() + ContextIds.TEST_NAVIGATOR;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.ui.internal.navigator.INavigator#getID()
     */
    public String getID() {
        return ITestNavigator.ID;
    }

    public boolean isVisibleResource(IResource resource) {
        return getFiltersManager().isVisibleResource(resource);
    }

    /**
     * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart,
     *      org.eclipse.jface.viewers.ISelection)
     */
    public void selectionChanged(IWorkbenchPart part, ISelection selection) {
        TreeViewer treeViewer = getTreeViewer();
        if (treeViewer == null || !isLinkingEnabled()) return;
        //- PN: Bugzilla #90657
        //-     The idea is to listen for all tptp editors (when the link with editor is ON), 
        //-     and to select the input of the editor rather than the selection that may be on 
        //-     a different tptp object (and then causes selection conflicts). 
        if (part instanceof BaseEditorPart) {
            Object element = ((BaseEditorPart)part).getEditorObject();
            IProxyNode proxy = getCurrentFileProxyManager().getCorrespondingProxy(element);
            if (proxy != null) {
                treeViewer.setSelection(new StructuredSelection(proxy), true);
            }
        } 
    }

    /*
     * Sets the current view (see view id) called from ToggleViewAction. Must be
     * called after creation of the view part.
     */
    public void setView(int viewerIndex) {
        if (viewerIndex < viewActions.length && currentViewIndex != viewerIndex) {
            currentViewIndex = viewerIndex;
            //- switching providers of the view
            updateTestNavigatorProvider((TreeViewer) getViewer());
            //- refresh the treeviewer in order to rebuild its content
            ((TreeViewer) getViewer()).refresh();
            updateActionStates();
            //- save this new index in the dialog settings
            getSettings().put(DIALOGSTORE_HTN_VIEW, viewerIndex);
        }
    }

    /**
     * 
     */
    private void updateActionStates() {
        // - toggle the view actions
        for (int i = 0; i < viewActions.length; i++) {
            ToggleViewAction action = viewActions[i];
            action.setChecked(currentViewIndex == action.getViewerIndex());
        }
        /* Change the enablement of the copy/past options when the view changes */
        TestNavigatorActionGroup HTNActionGroup = (TestNavigatorActionGroup)getActionGroup();
        if ( HTNActionGroup != null && currentViewIndex == VIEW_ID_LOGICAL ) {
        	HTNActionGroup.getPasteAction().setEnabled( false );
        	HTNActionGroup.getCopyAction().setEnabled( false );
        } else if ( HTNActionGroup != null && currentViewIndex == VIEW_ID_RESOURCE ) {
        	HTNActionGroup.getPasteAction().setEnabled( true );
        	HTNActionGroup.getCopyAction().setEnabled( true );
        }
    }

    /**
     * @return Returns the currentViewIndex.
     */
    public int getCurrentViewIndex() {
        return currentViewIndex;
    }

    class Refresher implements Runnable {
        private Object node;

        public Refresher(Object node) {
            this.node = node;
        }

        public void run() {
            StructuredViewer viewer = getViewer();
            if (viewer != null) {
            	if (node != null) {
            		viewer.refresh(node);
            	} else {
            		viewer.refresh();
            	}
            }
        }
    }

    /**
     * Invoked by test providers to notify that a node and its children should be refreshed.
     */
    public void nodeChanged(Object node) {
    	currentContentProvider.getProxyNodeListener().nodeChanged(node);
    	refresh(node);
    }
    
    public void nodesChanged() {
    	currentContentProvider.getProxyNodeListener().nodesChanged();
    	refresh(null);
    }
    
    /**
     * Internal method used by the provider to refresh a node and its sub-nodes. The difference
     * with nodeChanged(Object) is that this method does not notify the provider with the change.
     * @param node The node to be refreshed, or null for resfreshing the whole tree.
     */
    /*package*/void refresh(Object node) {
    	Display.getDefault().asyncExec(new Refresher(node));
    }
    
    /**
     * @return
     */
    public static Iterator getAllInstancesIterator() {
    	if (testNavigatorByWorkbenchWindow == null) {
    		return Collections.EMPTY_LIST.iterator();
    	} else {
    		return testNavigatorByWorkbenchWindow.values().iterator();
    	}
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.ui.part.ISetSelectionTarget#selectReveal(org.eclipse.jface.viewers.ISelection)
     */
    public void selectReveal(ISelection selection) {
        if (selection instanceof IStructuredSelection) {
            IStructuredSelection sel = (IStructuredSelection) selection;
            if(sel.size() == 1) {
                Object selObj = sel.getFirstElement();
                if(selObj instanceof IFile) {
                    IProxyNode proxy = getProxy((IFile)selObj);
                    if(proxy != null) {
                        getViewer().setSelection(new StructuredSelection(proxy), true);
                        return;
                    }
                }
            }
        } 
        super.selectReveal(selection);
    }
    
    /**
     * Ensures that a (potentially not already displayed) proxy is linked with its parent
     * in the current file proxy manager.
     * @param proxy
     */
    private IProxyNode getProxy(IFile file) {
    	IFileProxyManager proxyManager = getCurrentFileProxyManager();
    	// Force the proxy to be linked with its parent in the current file proxy manager.
    	switch (currentViewIndex) {
	    	case VIEW_ID_LOGICAL: {
	    		IProject project = file.getProject();
	    		if (project != null)currentContentProvider.getChildren(project);
	    		return proxyManager.getProxy(file, null);
	    	}
	    	case VIEW_ID_RESOURCE: {
	    		IProxyNode proxy = proxyManager.getProxy(file, file.getParent());
	    		if (proxyManager.getParent(proxy) == null) {
	    			proxyManager.cacheProxy(file, proxy);
	    		}
	    		return proxy;
	    	}
    	}
    	// Never reached
    	return null;
    }

    /**
     * Returns the name of the given used in UI. This name is declared in the extension of org.eclipse.hyades.ui.typeDescriptions 
     * @param type a test type.
     * @return the exposed name of the given type. 
     */
    static public String getRegisteredTypeName(String type) {
        AssociationMappingRegistry registry = (AssociationMappingRegistry)TestUIExtension.getTestSuiteMappingRegistry();
        IAssociationMapping associationMapping = registry.getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS);
        IAssociationDescriptor descriptor = associationMapping.getDefaultAssociationDescriptor(type);
        if (descriptor != null) {
            return descriptor.getName();
        } else {
            return "[Invalid Descriptor]"; //$NON-NLS-1$ 
        }
    }

	public FixedSizeJobPool getJobPool() {
		if (jobPool == null) {
			 jobPool = new FixedSizeJobPool(BACKGROUND_JOB_POOL_SIZE);
		}
		return jobPool;
	}
}