/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestNavigatorContentProvider.java,v 1.1 2009/05/19 00:03:44 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator;

import java.util.LinkedList;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxyNodeListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;

/**
 * <p>Content provider for the Test Navigator.</p>
 *
 * 
 * @author  Jerome Gout
 * @author  Julien Canches
 * @author  Paul E. Slauenwhite
 * @version May 18, 2009
 * @since   March 18, 2005 
 */
public abstract class TestNavigatorContentProvider implements ITreeContentProvider {

	private static final int MAX_PROXY_NODE_CHILDREN = 100;
	public static final int CONTEXT_LOADING = 0;
	public static final int CONTEXT_INTERACTIVE = 1;
	
	protected TestNavigator testNavigator;
	private int context;
	
	public TestNavigatorContentProvider(TestNavigator testNavigator) {
		this.testNavigator = testNavigator;
		this.context = CONTEXT_LOADING;
	}

	protected abstract IGlobalProxyNodeListener getProxyNodeListener();

	protected abstract IFileProxyManager getFileProxyManager();
		
	public Object getParent(Object element) {
		if (element instanceof IProxyNode) {
            Object parent = ((IProxyNode) element).getParent();
            if(parent instanceof FileProxyManager.IUnboundedParent) {
                return getFileProxyManager().getParent((IProxyNode) element);
            } else {
                return parent;
            }
		}
		return null;
	}

	public boolean hasChildren(Object parentElement) {
		if (parentElement instanceof IProxyNode) {
			IProxyNode proxy = (IProxyNode) parentElement;
			return proxy.getChildren().length > 0;
		}
		if (parentElement instanceof IProxyGroup) {
			return true;
		}
		return false;
	}

	public Object[] getChildren(Object parentElement) {
		if (parentElement instanceof IProxyNode) {
			IProxyNode proxy = (IProxyNode) parentElement;
			return foldProxyNodes(proxy);
		}
		if (parentElement instanceof IProxyGroup) {
			return ((IProxyGroup)parentElement).getChildren();
		}
		return new Object[0];
	}
	
	public Object[] getElements(Object inputElement) {
		if (inputElement instanceof IWorkspaceRoot) {
			IProject [] projects = ((IWorkspaceRoot) inputElement).getProjects();
			LinkedList visibleProjects = new LinkedList();
			for(int i = 0; i < projects.length; i++) {
				if(!TestNavigator.getFiltersManager().filter(projects[i])) {
					visibleProjects.add(projects[i]);
				}
			}
			return visibleProjects.toArray();
		}
		return getChildren(inputElement);
	}
    
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
	}
	
	public static interface IGlobalProxyNodeListener extends IProxyNodeListener {
		
		void nodesChanged();
		
	}
	
	public static interface IProxyGroup extends IProxyNode{
		int getOrder();
	}
	
	private static class ProxyGroup implements IProxyGroup {
		private IProxyNode parent;
		private int sequence;
		public ProxyGroup(IProxyNode node, int sequence) {
			this.parent = node;
			this.sequence = sequence;
		}
		public IProxyNode[] getChildren() {
			int start = sequence * MAX_PROXY_NODE_CHILDREN;
			IProxyNode[] completeChildren = parent.getChildren();
			IProxyNode[] children = new IProxyNode[MAX_PROXY_NODE_CHILDREN];
			System.arraycopy(completeChildren, start, children, 0, MAX_PROXY_NODE_CHILDREN);
			return children;
		}
		public String getText() {
			int start = sequence * MAX_PROXY_NODE_CHILDREN + 1;
			int end = (sequence + 1) * MAX_PROXY_NODE_CHILDREN;
			return ("[".concat(Integer.toString(start)).concat(" - ").concat(Integer.toString(end)).concat("]"));
		}
		public Image getImage() {
			return parent.getImage();
		}
		public int getOrder() {
			return sequence;
		}
		public Object getParent() {	
			return parent;
		}
		public Object getAdapter(Class adapter) {
			return null;
		}
		public String getIdentifier() {
			return null;
		}
		public IResource getUnderlyingResource() {
			return null;
		}
	}
	
	private static class RemainingProxyGroup implements IProxyGroup {
		private IProxyNode parent;
		public RemainingProxyGroup(IProxyNode parent) {
			this.parent = parent;
		}
		public IProxyNode[] getChildren() {
			IProxyNode[] completeChildren = parent.getChildren();
			int completeLength = completeChildren.length;
			int length = completeLength % MAX_PROXY_NODE_CHILDREN;
			IProxyNode[] children = new IProxyNode[length];
			System.arraycopy(completeChildren, completeLength - length, children, 0, length);
			return children;
		}
		public String getText() {
			IProxyNode[] completeChildren = parent.getChildren();
			int end = completeChildren.length;
			int length = end % MAX_PROXY_NODE_CHILDREN;
			int start = end - length + 1;
			return ("[".concat(Integer.toString(start)).concat(" - ").concat(Integer.toString(end)).concat("]"));
		}
		public Image getImage() {
			return parent.getImage();
		}
		public int getOrder() {
			return Integer.MAX_VALUE;
		}
		public Object getParent() {
			return parent;
		}
		public Object getAdapter(Class adapter) {
			return null;
		}
		public String getIdentifier() {
			return null;
		}
		public IResource getUnderlyingResource() {
			return null;
		}
	}
	
	protected Object[] foldProxyNodes(IProxyNode parent) {
		IProxyNode[] children = parent.getChildren();
		int length = children.length;
		if (length <= MAX_PROXY_NODE_CHILDREN) return children;
		int size = length / MAX_PROXY_NODE_CHILDREN;
		int remaining = length % MAX_PROXY_NODE_CHILDREN;
		int totalSize = size + (remaining > 0 ? 1 : 0);
		Object[] groups = new IProxyGroup[totalSize];
		for (int i = 0; i < size; i++) {
			groups[i] = new ProxyGroup(parent, i);
		}
		if (remaining > 0) {
			groups[size] = new RemainingProxyGroup(parent);
		}
		return groups;
	}

	/**
	 * Sets the context in which the provider is used. There are two available values:
	 * INTERACTIVE and LOADING. The context has an effect on two parameters:
	 * 1) responsiveness: time that getChildren() is allowed to spent before returning
	 * control. High responsiveness means that the provider will return incomplete nodes,
	 * compute the remaining in a background task, and update the navigator when computation
	 * is complete. Low responsiveness means that the UI will be blocked for a longer period of
	 * time but the returned node are more likely to be complete.
	 * 2) computation job priority: the priority allocated to the computation job, if the provider
	 * does not manage to compute nodes in the allocated time (responsiveness parameter). This does
	 * not affect the thread priority, but helps eclipse decide which job should be launched first.
	 * In LOADING mode, responsiveness is high (100ms) and priority is low (NORMAL).
	 * In INTERACTIVE mode, responsiveness is lower (800ms) and priority is high (INTERACTIVE).
	 */
	public void setContext(int context) {
		this.context = context;
	}
	
	protected int getContext() {
		return context;
	}
	
	protected int getResponseTime() {
		switch (context) {
		case CONTEXT_INTERACTIVE:
			return 800;
		case CONTEXT_LOADING:
			return 100;
		}
		return 100;
	}
	
	protected int getJobPriority() {
		switch(context) {
		case CONTEXT_INTERACTIVE:
			return Job.INTERACTIVE;
		case CONTEXT_LOADING:
			return Job.LONG;
		}
		return Job.LONG;
	}	
}
