/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestResourceChangeUpdater.java,v 1.21 2010/06/28 12:19:40 bjerome Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;

/**
 * <p>TestResourceChangeUpdater.java</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version June 1, 2010
 * @since   February 1, 2005
 */
public class TestResourceChangeUpdater  implements IResourceChangeUpdater {
	
	private TestNavigator testNavigator;

	private List addedObjects;
	private List removedObjects;
	private List changedObjects;
	private List resourcesToSave;
	private Object listToken = new Object();
	
	private List expandedElements;
	private List selectedElements;
	
	private boolean active;
	
	/**
	 * Flag to determine if this updater has been disposed.
	 */
	private boolean isDisposed = false;
	
	/**
	 * Constructor for TestResourceChangeUpdater
	 */
	public TestResourceChangeUpdater(TestNavigator testNavigator)
	{
		this.testNavigator = testNavigator;
		active = true;
		isDisposed = false;
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		active = false;
		isDisposed = true;
		testNavigator = null;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#started()
	 */
	public void started()
	{
		synchronized(listToken) {
			addedObjects = new UniqueEList();
			removedObjects = new UniqueEList();
			changedObjects = new UniqueEList();
			resourcesToSave = new UniqueEList();
		}
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#ended()
	 */
	public void ended() {
		Object[] copyList = null;
		// added objects
		synchronized(listToken) {
			if(!addedObjects.isEmpty()) {
				copyList = addedObjects.toArray();
			}
		}		
		if(copyList != null) {
			for (int i=0; i<copyList.length; i++) {
				ITreeContentProvider treeContentProvider = (ITreeContentProvider)testNavigator.getViewer().getContentProvider();
				if(treeContentProvider != null) {
					Object parent = treeContentProvider.getParent(copyList[i]);
					if(parent == null)
					{
						testNavigator.nodesChanged();
						return;
					}
					testNavigator.nodeChanged(parent);
				} else {
					return;
				}
			}
		}
		// removed objects
		copyList = null;
		synchronized(listToken) {
			if(!removedObjects.isEmpty()) {
				copyList = removedObjects.toArray();
			}
		}		
		if(copyList != null) {
			final Object[] finalCopyList = copyList;
			org.eclipse.swt.widgets.Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					if(!testNavigator.getTreeViewer().getControl().isDisposed()) {
						ITreeContentProvider treeContentProvider = (ITreeContentProvider)testNavigator.getViewer().getContentProvider();
						for (int i=0; i<finalCopyList.length; i++) {
							Object parent = treeContentProvider.getParent(finalCopyList[i]);
							if(parent == null) {
								testNavigator.nodesChanged();
								return;
							}
							testNavigator.nodeChanged(parent);
						}
					}
				}
			});
		}
		// changed objects
		copyList = null;
		synchronized(listToken) {
			if(!changedObjects.isEmpty()) {
				copyList = changedObjects.toArray();
			}
		}		
		if(copyList != null) {
			for (int i=0; i<copyList.length; i++) {
				testNavigator.nodeChanged(copyList[i]);
			}
		}
		// resource to save
		copyList = null;
		synchronized(listToken) {
			if(!resourcesToSave.isEmpty()) {
				copyList = resourcesToSave.toArray();
			}
		}		
		if(copyList != null) {
			for (int i=0; i<copyList.length; i++) {
				try {
					EMFUtil.save((Resource)(copyList[i]));
				} catch (Exception e) {
				}
			}			
		}
		// clean up
		synchronized(listToken) {		
			addedObjects.clear();
			addedObjects = null;
			removedObjects.clear();
			removedObjects = null;
			changedObjects.clear();
			changedObjects = null;
			resourcesToSave.clear();
			resourcesToSave = null;
		}
		org.eclipse.swt.widgets.Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				if(expandedElements != null)
				{
					convertIdentifierToProxy(expandedElements);
					testNavigator.getTreeViewer().setExpandedElements(expandedElements.toArray());
					expandedElements = null;
				}
				if(selectedElements != null) {
					convertIdentifierToProxy(selectedElements);
					testNavigator.getTreeViewer().setSelection(new StructuredSelection(selectedElements), true);
					selectedElements = null;
				}				
			}
		});			
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#add(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource[])
	 */
	public boolean add(IResource parent, IResource[] affectedResources) {
		int htnMode = testNavigator.getCurrentViewIndex();
		if(htnMode == TestNavigator.VIEW_ID_RESOURCE) {
			for (int i = 0, maxi = affectedResources.length; i < maxi; i++) {
				if (affectedResources[i].getType() != IResource.FILE) {
					//- not a file thus a project or folder, add it if it is no filtered
					if(testNavigator.isVisibleResource(affectedResources[i])) {
						synchronized(listToken) {
							addedObjects.add(affectedResources[i]);
						}
					}
				} else {
					// Don't bother computing the proxy here, since this may be long.
					// Refresh the container so its children are re-computed and the added
					// file gets discovered (the discovery will happen in a background job,
					// which is much better).
					IContainer container = affectedResources[i].getParent();
					synchronized(listToken) {
						changedObjects.add(container);
					}
				}
			}
		} else {
			//- update the htn logical view 
            //- just focus upon the project creation
		    for (int i = 0, maxi = affectedResources.length; i < maxi; i++) {
                if (affectedResources[i].getType() == IResource.PROJECT) {
                    if(testNavigator.isVisibleResource(affectedResources[i])) {
						synchronized(listToken) {
							addedObjects.add(affectedResources[i]);
						}
                    }
                }
            }
        }
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#remove(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource[])
	 */
	public boolean remove(IResource parent, IResource[] affectedResources)
	{
		int htnMode = testNavigator.getCurrentViewIndex();
		if(htnMode == TestNavigator.VIEW_ID_RESOURCE) {
			for(int i = 0, maxi = affectedResources.length; i < maxi; i++) {
				if(affectedResources[i].getType() != IResource.FILE) {
					synchronized(listToken) {
						removedObjects.add(affectedResources[i]);
					}
				} else {
					//- in case of file removed, we need to find the right corresponding element in the test navigator
					//- try to retrieve a proxy from this file
					//- the object to remove from the navigator is the proxy
					IProxyNode proxy = testNavigator.getCurrentFileProxyManager().getProxy((IFile)affectedResources[i], null);
					if(proxy != null) {
						synchronized(listToken) {
							removedObjects.add(proxy);
						}
					}
				}
			}
		} else {
			//- update the htn logical view 
            //- just focus upon the project creation
            for (int i = 0, maxi = affectedResources.length; i < maxi; i++) {
                if (affectedResources[i].getType() == IResource.PROJECT) {
            		synchronized(listToken) {
            			removedObjects.add(affectedResources[i]);
            		}
                }
            }
		}
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#replaced(org.eclipse.core.resources.IResource)
	 */
	public boolean replaced(IResource affectedResource)
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateProperties(org.eclipse.core.resources.IResource)
	 */
	public boolean updateProperties(final IResource affectedResource)
	{
		if(affectedResource.getType() == IResource.FILE)
		{
			final EObject[] eObjects = EMFUtil.getEObjects(null, (IFile)affectedResource);
			if(eObjects.length > 0) {
				org.eclipse.swt.widgets.Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						testNavigator.getTreeViewer().update(eObjects, null);
					}
				});
			}
		}
		else
		{
			org.eclipse.swt.widgets.Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					testNavigator.getTreeViewer().update(affectedResource, null);
				}
			});
		}
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateChildrenType(org.eclipse.core.resources.IResource)
	 */
	public boolean updateChildrenType(IResource affectedResource)
	{
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateContent(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource)
	 */
	public boolean updateContent(IResource parent, IResource affectedResource){
		
		//Check if this updater has been disposed:
		if (isDisposed){ 
			return false;
		}

		if((affectedResource instanceof IFile) && (testNavigator.getCurrentViewIndex() == TestNavigator.VIEW_ID_RESOURCE)) {

			//Check again if this updater has been disposed since it may have been disposed since the previous check:
			if (isDisposed){ 
				return false;
			}

			IProxyNode proxy = testNavigator.getCurrentFileProxyManager().updateProxy((IFile)affectedResource, parent);

			if(proxy != null) {

				//Check again if this updater has been disposed since it may have been disposed since the previous check:
				if (isDisposed){ 
					return false;
				}

				try {
					testNavigator.getCurrentFileProxyManager().cacheProxy((IFile)affectedResource, proxy);					
				} 
				catch (Exception e) {
					CommonPlugin.logError(e);
				}
				synchronized (listToken) {
					changedObjects.add(parent);
				}
			}

			if((changedObjects.size() > 0) && (expandedElements == null)){

				Display.getDefault().asyncExec(new Runnable() {

					/* (non-Javadoc)
					 * @see java.lang.Runnable#run()
					 */
					public void run() {

						expandedElements = new ArrayList();

						Object[] expObjects = testNavigator.getTreeViewer().getExpandedElements();

						for (int counter = 0; counter < expObjects.length; counter++) {
							expandedElements.add(expObjects[counter]);
						}

						convertProxyToIdentifier(expandedElements);

						selectedElements = ((IStructuredSelection)(testNavigator.getTreeViewer().getSelection())).toList();

						convertProxyToIdentifier(selectedElements);
					}
				});
			}		
		} 

		return false;
	}


	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#setActive(boolean)
	 */
	public void setActive(boolean enable)
	{
		active = enable;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#isActive()
	 */
	public boolean isActive()
	{
		return active;
	}
	
	/**
	 * Returns whether the eObject is valid in the context of the 
	 * test navigator.
	 * @param eObject
	 * @return boolean
	 */
	protected boolean isValid(EObject eObject)
	{
		return (eObject instanceof CMNNamedElement);
	}
	
	/**
	 * Small inner class to wrap proxy identifier and underlying resource in a single class
	 * @author jgout
	 */
	class ProxyIdentifier {
		private String identifier;
		private IResource underlyingResource;
		
		/**
		 * 
		 */
		public ProxyIdentifier(String identifier, IResource res) {
			this.identifier = identifier;
			this.underlyingResource = res;
		}

		public String getIdentifier() {
			return identifier;
		}
		public IResource getUnderlyingResource() {
			return underlyingResource;
		}
	}
	
	private void convertProxyToIdentifier(List objects) {
		for (int i = 0; i < objects.size(); i++) {
			if(objects.get(i) instanceof IProxyNode) {
				IProxyNode proxy = (IProxyNode)objects.get(i);
				objects.set(i, new ProxyIdentifier(proxy.getIdentifier(), proxy.getUnderlyingResource()));
			}
		}
	}
		
	private void convertIdentifierToProxy(List objects) {
		for (int i = 0; i < objects.size(); i++) {
			if(objects.get(i) instanceof ProxyIdentifier) {
				ProxyIdentifier ident = (ProxyIdentifier)objects.get(i);
				IResource res = ident.getUnderlyingResource();
				String fragment = ident.getIdentifier();
				IProxyNode root = res instanceof IFile ? testNavigator.getCurrentFileProxyManager().getProxy((IFile)res, null) : null;
				if(root != null) {
					IProxyNode proxy = testNavigator.getCurrentFileProxyManager().findProxyByID(root, fragment);
					if(proxy != null) {
						objects.set(i, proxy);
					}
				}
			}
		}
	}
	
	public void refreshContent(final IResource affectedResource) {
		Control control= testNavigator.getTreeViewer().getControl();
		if(!control.isDisposed()) {
			org.eclipse.swt.widgets.Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					testNavigator.getTreeViewer().refresh(affectedResource);
				}
			});
		}
	}
}
