/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CopyAction.java,v 1.11 2009/07/27 14:49:47 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.action;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.ResourceTestNavigatorContentProvider;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.navigator.FileProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.ProxyNodeTransfer;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWTError;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.ui.part.ResourceTransfer;

/** 
 * <p>Copy action for the Resource view of the Test Navigator.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version July 27, 2009
 * @since   March 7, 2005
 */
public class CopyAction extends Action implements IDisposable {
	private Clipboard clipboard;
	private IStructuredSelection selection;
	private boolean isResourceNav;
	
	static class ClipboardManager {
		private Map clipboardContent;
		private static ClipboardManager instance;
		
		private ClipboardManager() {
			this.clipboardContent = new HashMap();
		}
		
		public static ClipboardManager getInstance() {
			if(instance == null) {
				instance = new ClipboardManager();
			}
			return instance;
		}
		
		public void addText(String text) {
			String content = ""; //$NON-NLS-1$
			if(clipboardContent.containsKey(TextTransfer.getInstance())) {
				content = (String)clipboardContent.get(TextTransfer.getInstance());
				text += "\n"; //$NON-NLS-1$
			}
			content += text;
			clipboardContent.put(TextTransfer.getInstance(), content);
		}
		
		public void addFile(String fileName) {
			List fileNames;
			if(clipboardContent.containsKey(FileTransfer.getInstance())) {
				fileNames = Arrays.asList((Object[])clipboardContent.get(FileTransfer.getInstance()));
			} else {
				fileNames = new LinkedList();
			}
			List tmp = new LinkedList(fileNames);
			tmp.add(fileName);
			clipboardContent.put(FileTransfer.getInstance(),(String[]) tmp.toArray(new String[tmp.size()]));
		}
		
		public void addResource(IResource res) {
			List resources;
			if(clipboardContent.containsKey(ResourceTransfer.getInstance())) {
				resources = Arrays.asList((Object[])clipboardContent.get(ResourceTransfer.getInstance()));
			} else {
				resources = new LinkedList();
			}
			List tmp = new LinkedList(resources);
			tmp.add(res);
			clipboardContent.put(ResourceTransfer.getInstance(),(IResource[]) tmp.toArray(new IResource[tmp.size()]));
		}
		
		public void addProxy(IProxyNode proxy) {
			List proxies;
			if(clipboardContent.containsKey(ProxyNodeTransfer.getInstance())) {
				proxies = Arrays.asList((Object[])clipboardContent.get(ProxyNodeTransfer.getInstance()));
			} else {
				proxies = new LinkedList();
			}
			List tmp = new LinkedList(proxies);
			tmp.add(proxy);
			clipboardContent.put(ProxyNodeTransfer.getInstance(),(IProxyNode[]) tmp.toArray(new IProxyNode[tmp.size()]));
		}

		public Object [] getData() {
			Transfer[] transfers = getTransfer();
			Object[] data = new Object[transfers.length];
			
			for (int i = 0; i < transfers.length; i++) {
				data[i] = clipboardContent.get(transfers[i]);
			}
			return data;
		}
		
		public Transfer [] getTransfer() {
			Set transfers = clipboardContent.keySet();
			return (Transfer[]) transfers.toArray(new Transfer[transfers.size()]);
		}
		
		public void clear() {
			clipboardContent.clear();
		}
	}
	
	
	public CopyAction(Clipboard clipboard) {
		super(UiPluginResourceBundle.command_Copy);
		if (clipboard == null) {
			throw new IllegalArgumentException("Unable to perform copy action due to 'clipboard == null'"); //$NON-NLS-1$
		}
		this.clipboard = clipboard;
		this.isResourceNav = false;
	}
	
	public void dispose() {
		if(clipboard != null) {
			clipboard.dispose();
		}
		clipboard = null;
	}
		
	private void setClipboard() {
		try {
			clipboard.setContents(ClipboardManager.getInstance().getData(), ClipboardManager.getInstance().getTransfer());
		} catch (SWTError e) {
			if (e.code != DND.ERROR_CANNOT_SET_CLIPBOARD) {
				throw e;
			}
			UiPlugin.logError(e);
		}
	}

	private void copyOneResource(IResource res) {
		IPath location = res.getLocation();
		if (location != null) {
			//- file path to be used in external copy
			ClipboardManager.getInstance().addFile(location.toOSString());
		}
		//- file name to by used in textual context copy 
		ClipboardManager.getInstance().addText(res.getName());
		//- resource itself to be used in internal copy (inside Eclipse)
		ClipboardManager.getInstance().addResource(res);
	}
	
	/** Tests whether the given proxy is a proxy of file or not
	 * @param proxy
	 * @return true if this proxy replaces a file
	 */
	private boolean isAFileProxy(IProxyNode proxy) {
        Object parent = ResourceTestNavigatorContentProvider.getSharedFileProxyManager().getParent(proxy);
		return proxy instanceof FileProxyNode || parent instanceof IContainer;
	}

	private void copyOneProxy(IProxyNode proxy) {
		if(isAFileProxy(proxy)) {
			IResource res = proxy.getUnderlyingResource();
			IPath location = res.getLocation();
			if (location != null) {
				//- external file path of the underlying resource to be used in external copy
				ClipboardManager.getInstance().addFile(location.toOSString());
			}
			//- underlying resource to be used in internal copy (inside Eclipse)
			ClipboardManager.getInstance().addResource(res);
		}

		//Copy the text of the project (e.g. logical name of the test asset) to the clipboard:
		ClipboardManager.getInstance().addText(proxy.getText());

		//- proxy itself for test navigator context copy
		ClipboardManager.getInstance().addProxy(proxy);
	}
		
	private void copySelection() {
		ClipboardManager.getInstance().clear();
		List allSelected = selection.toList();
		for (Iterator it = allSelected.iterator(); it.hasNext();) {
			Object item = (it.next());
			if (item instanceof IResource) {
				copyOneResource((IResource)item);
			} else {
				if (item instanceof IProxyNode) {
					copyOneProxy((IProxyNode)item);
				}
			}
		}
		setClipboard();
	}

	public void run() {
		copySelection();
	}

	private boolean selectionIsHomogeneous() {
		Object selItem = null;
		int flag = 0;
		int val = 0;
		for(Iterator it = selection.iterator(); it.hasNext(); ) {
			selItem = it.next();
			if (selItem instanceof IProject) {
				val = 1;
			} else if (selItem instanceof IFolder) {
				val = 1 << 2;
			} else if (selItem instanceof IFile) {
				val = 1 << 3;
			} else if (selItem instanceof IProxyNode) {
				val = 1 << 4;
			} else {
				//- unsupported type
				return false;
			}
			//- boostrap
			if(flag == 0) {
				flag = val;
			} else {
				flag ^= val;
				if(flag != 0) {
					//- this means that the current element type is not the same as one previously found 
					return false;
				}
			}
		}
		return true;
	}

	private Object getParent(Object sel) {
		if (sel instanceof IResource) {
			IResource res = (IResource) sel;
			return res.getParent();
		} else if (sel instanceof IProxyNode) {
			IProxyNode proxy = (IProxyNode) sel;
            return ResourceTestNavigatorContentProvider.getSharedFileProxyManager().getParent(proxy);
		} else {
			return null;
		}
	}
	
	private boolean selectionHasSameParent() {
		Object firstParent = getParent(selection.getFirstElement());
		
		if(firstParent == null) {
			return false;
		}
		Iterator it = selection.iterator();
		while(it.hasNext()) {
			if(!firstParent.equals(getParent(it.next()))) {
				return false;
			}
		}
		return true;
	}

	public boolean isApplicableForSelection() {
		//- Copy action only allowed if the navigator is presenting its resource mode
		if(!isResourceNav) return false;
		if (selection == null || selection.isEmpty()) {
			return false;
		}	
		//- selection should be homogeneous
		if(!selectionIsHomogeneous()) {
			return false;
		}
		//- all elements should have the same parent
		if(!selectionHasSameParent()) {
			return false;
		}
		return true;
	}
	
	public void selectionChanged(IStructuredSelection structuredSelection, boolean isResourceNav) {
		selection = structuredSelection;
		this.isResourceNav = isResourceNav;
	}
}
