/********************************************************************** 
 * Copyright (c) 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: EditorUtil.java,v 1.1 2008/03/20 19:49:54 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.action;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceDescription;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.dialog.ListDialog;
import org.eclipse.hyades.test.ui.internal.navigator.action.resources.ActionMessages;
import org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter;
import org.eclipse.hyades.ui.editor.IEditorExtension;
import org.eclipse.hyades.ui.internal.editor.HyadesEditorPart;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

/**
 * @author jgout,Jerome Bozier
 * @version February 26, 2008
 */
public class EditorUtil {
	
	/**
	 * Saves all modified editors.
	 * @param shell
	 * @return <code>true</code> if all saves have been successfully performed and <code>false</code> otherwise. 
	 */
	public static boolean saveEditors(Shell shell) {
		IEditorPart[] dirtyEditors= getDirtyEditors();
		if (dirtyEditors.length == 0)
			return true;
		if (! saveAllDirtyEditors(shell, dirtyEditors))
			return false;
		try {
			// Save isn't cancelable.
			IWorkspace workspace= ResourcesPlugin.getWorkspace();
			IWorkspaceDescription description= workspace.getDescription();
			boolean autoBuild= description.isAutoBuilding();
			description.setAutoBuilding(false);
			workspace.setDescription(description);
			try {
				for (int i = 0; i < dirtyEditors.length; i++) {
					IEditorPart part = dirtyEditors[i];
					if (part instanceof HyadesEditorPart) {
						HyadesEditorPart hyadesEditor = (HyadesEditorPart) part;
						hyadesEditor.doSave(new NullProgressMonitor());
						IEditorExtension extension = hyadesEditor.getEditorExtension();
						if (extension instanceof ISynchronizedEditorAdapter) {
							((ISynchronizedEditorAdapter) extension).reload();
						}
					}
				}
				if (!UiPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getWorkbench().saveAllEditors(false))
					return false;
			} finally {
				description.setAutoBuilding(autoBuild);
				workspace.setDescription(description);
			}
			return true;
		} catch (CoreException e) {
			UiPlugin.logError("Unable to save modified resources", e); //$NON-NLS-1$
			return false;
		}
	}

	private static boolean saveAllDirtyEditors(Shell shell, IEditorPart[] dirtyEditors) {
		ListDialog dialog= new ListDialog(shell) {
			protected Control createDialogArea(Composite parent) {
				Composite result= (Composite) super.createDialogArea(parent);
				applyDialogFont(result);
				return result;
			}
		};
		dialog.setTitle(ActionMessages.SAVE_ALL_DIALOG_NAME); 
		dialog.setAddCancelButton(true);
		dialog.setLabelProvider(createDialogLabelProvider());
		dialog.setMessage(ActionMessages.SAVE_ALL_DIALOG_MESSAGE); 
		dialog.setContentProvider(new ArrayContentProvider());
		dialog.setInput(Arrays.asList(dirtyEditors));
		return dialog.open() == Window.OK;
	}
	
	private static ILabelProvider createDialogLabelProvider() {
		return new LabelProvider() {
			public Image getImage(Object element) {
				return ((IEditorPart) element).getTitleImage();
			}
			public String getText(Object element) {
				return ((IEditorPart) element).getTitle();
			}
		};
	}	
	
	/**
	 * Returns an array of all editors that have an unsaved content. If the identical content is 
	 * presented in more than one editor, only one of those editor parts is part of the result.
	 * (Copied from jdt internal)
	 * 
	 * @return an array of all dirty editor parts.
	 */
	private static IEditorPart[] getDirtyEditors() {
		Set inputs= new HashSet();
		List result= new ArrayList(0);
		IWorkbench workbench= PlatformUI.getWorkbench();
		IWorkbenchWindow[] windows= workbench.getWorkbenchWindows();
		for (int i= 0; i < windows.length; i++) {
			IWorkbenchPage[] pages= windows[i].getPages();
			for (int x= 0; x < pages.length; x++) {
				IEditorPart[] editors= pages[x].getDirtyEditors();
				for (int z= 0; z < editors.length; z++) {
					IEditorPart ep= editors[z];
					if (ep instanceof HyadesEditorPart) {
						IEditorInput input= ep.getEditorInput();
						if (!inputs.contains(input)) {
							inputs.add(input);
							result.add(ep);
						}
					}
				}
			}
		}
		return (IEditorPart[])result.toArray(new IEditorPart[result.size()]);
	}
}
