/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileFolderInContainerPaste.java,v 1.7 2009/05/19 00:03:42 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.action;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.ResourceTestNavigatorContentProvider;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigatorMessages;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.actions.IPaster;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.actions.CopyFilesAndFoldersOperation;
import org.eclipse.ui.part.ResourceTransfer;

/** 
 * <p>Paste action for the Resource view of the Test Navigator.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version May 18, 2009
 * @since   March 7, 2005
 * @deprecated
 */
public class FileFolderInContainerPaste implements IPaster {

	private IResource [] getResourceFromClipboard(final Clipboard clipboard) {
		final IResource[][] clipboardData = new IResource[1][];
		Display.getCurrent().syncExec(new Runnable() {
			public void run() {
				//- clipboard must have resources or files
				ResourceTransfer resTransfer = ResourceTransfer.getInstance();
				clipboardData[0] = (IResource[]) clipboard.getContents(resTransfer);
			}
		});
		return clipboardData[0];
	}
	
	public boolean isPasteAllowedFor(Clipboard clipboard, Object selection) {
		//- check for internal copy (inside eclipse)
		IResource [] res = getResourceFromClipboard(clipboard);
		if(res != null && res.length > 0) {
			//- all resources stored in the clipboard should be folders/files
			for (int i = 0; i < res.length; i++) {
				if(res[i].getType() != IResource.FOLDER && res[i].getType() != IResource.FILE) {
					return false;
				}
			}
			return true;
		}
		//- check for external files or folders copy (files or folders have been copied outside eclipse)
		TransferData[] transfers = clipboard.getAvailableTypes();
		FileTransfer fileTransfer = FileTransfer.getInstance();
		for (int i = 0; i < transfers.length; i++) {
			if (fileTransfer.isSupportedType(transfers[i])) {
				return true;
			}
		}

		return false;
	}

    /**
     * Returns a new name for a copy of the resource at the given path in the
     * given workspace. This name is determined automatically.
     * 
     * @param originalName the full path of the resource
     * @param workspace the workspace
     * @return the new full path for the copy
     */
    private IPath getAutoNewNameFor(IPath originalName, IWorkspace workspace) {
        int counter = 1;
        String resourceName = originalName.lastSegment();
        IPath leadupSegment = originalName.removeLastSegments(1);

        while (true) {
            String nameSegment;
            if (counter > 1) {
                nameSegment = NLS.bind(TestNavigatorMessages.FileFolderInContainerPaste_NameTwoArg, new Integer(counter), resourceName);
            } else {
                nameSegment = NLS.bind(TestNavigatorMessages.FileFolderInContainerPaste_NameOneArg, resourceName);
            }
            IPath pathToTry = leadupSegment.append(nameSegment);
            if (!workspace.getRoot().exists(pathToTry)) {
                return pathToTry;
            }
            counter++;
        }
    }

    private IPath getNewNameFor(final IPath originalName, final IWorkspace workspace) {
        final IResource resource = workspace.getRoot().findMember(originalName);
        final IPath prefix = resource.getFullPath().removeLastSegments(1);
        final String returnValue[] = { "" }; //$NON-NLS-1$
        final Shell shell = Display.getCurrent().getActiveShell();
        
        shell.getDisplay().syncExec(new Runnable() {
            public void run() {
                IInputValidator validator = new IInputValidator() {
                    public String isValid(String string) {
                        if (resource.getName().equals(string)) {
                            return TestNavigatorMessages.FileFolderInContainerPaste_ErrorSameName;
                        }
                        IStatus status = workspace.validateName(string,
                                resource.getType());
                        if (!status.isOK()) {
                            return status.getMessage();
                        }
                        if (workspace.getRoot().exists(prefix.append(string))) {
                            return TestNavigatorMessages.FileFolderInContainerPaste_ErrorAlreadyExists;
                        }
                        return null;
                    }
                };

                InputDialog dialog = new InputDialog(shell, TestNavigatorMessages.FileFolderInContainerPaste_MessageTitle, 
                        NLS.bind(TestNavigatorMessages.FileFolderInContainerPaste_MessageLabel, resource.getName()),
                        getAutoNewNameFor(originalName, workspace).lastSegment().toString(), validator);
                dialog.setBlockOnOpen(true);
                dialog.open();
                if (dialog.getReturnCode() == Window.CANCEL) {
                    returnValue[0] = null;
                } else {
                    returnValue[0] = dialog.getValue();
                }
            }
        });
        if (returnValue[0] == null) {
            throw new OperationCanceledException();
        }
        //- if the user has omitted/misspelled the correct file extension (same as source one) we should add it 
        String fileExt = resource.getFileExtension();
        if (fileExt != null) {
        	if(!returnValue[0].endsWith(fileExt)) {
        		returnValue[0] += '.'+fileExt;
        	}
        }
        return prefix.append(returnValue[0]);
    }
    
    public boolean performPaste(Clipboard clipboard, Object selection) {
        if (selection instanceof IContainer) {
            IContainer dest = (IContainer) selection;
            //- check for internal copy (inside eclipse)
            IResource[] res = getResourceFromClipboard(clipboard);
            if (res != null && res.length > 0) {
                boolean oneCopied = false;
                for (int i = 0; i < res.length; i++) {
                    IResource source = res[i];
                    boolean conflict = false;
                    //- all resources stored in the clipboard should be folders/files
                    if(source.getType() == IResource.FOLDER || source.getType() == IResource.FILE) {
                        IPath destinationPath = dest.getFullPath().append(source.getName());
                        //- check whether the destination resource already exists.
                        if(dest.findMember(source.getName()) != null) {
                            //- name conflict shoudl ask a new name
                            try {
                                destinationPath = getNewNameFor(destinationPath, source.getWorkspace());
                                conflict = true;
                            } catch (OperationCanceledException e) {
                                //- user has cancelled rename. Ends copy.
                                return true;
                            }                            
                        }
                        if (destinationPath != null) {
                            try {
                                source.copy(destinationPath, IResource.SHALLOW, new NullProgressMonitor());
                                oneCopied = true;
                                if (source instanceof IFile && conflict) {
                                    IFile file = source.getWorkspace().getRoot().getFile(destinationPath);
                                    IProxyNode proxy = ResourceTestNavigatorContentProvider.getSharedFileProxyManager().getProxy(file, null);
                                    IProxyNodeRenamer renamer = (IProxyNodeRenamer) proxy.getAdapter(IProxyNodeRenamer.class);
                                    if (renamer != null) {
                                        //- use the renamer of this proxy node to rename properly
                                        String newName = destinationPath.lastSegment().toString();
                                        newName = newName.substring(0, newName.length()-destinationPath.getFileExtension().length()-1);
                                        renamer.performRename(newName);
                                    }
                                }
                            } catch (CoreException e) {
                                UiPlugin.logError(e);
                                return false;
                            }
                        }
                    }
                }
                if(oneCopied) {
                    return true;
                }
            }
            // - try a file transfer
            FileTransfer fileTransfer = FileTransfer.getInstance();
            String[] fileData = (String[]) clipboard.getContents(fileTransfer);

            if (fileData != null) {
                CopyFilesAndFoldersOperation operation = new CopyFilesAndFoldersOperation(Display.getCurrent().getActiveShell());
                operation.copyFiles(fileData, (IContainer) selection);
                return true;
            }
        }
		return false;
	}
}
