/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PasteAction.java,v 1.7 2009/05/17 16:16:08 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.action;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.IPasterExtended;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.navigator.actions.IPaster;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.actions.CopyProjectOperation;
import org.eclipse.ui.part.ResourceTransfer;

/**
 * <p>PasteAction.java</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @version May 16, 2009
 * @since   March 7, 2005
 */
public class PasteAction extends Action implements IDisposable {
	protected Clipboard clipboard;
	private IStructuredSelection selection;
	private Shell shell;
	private boolean isResourceNav;
	
	public PasteAction(Shell shell, Clipboard clipboard) {
		super(UiPluginResourceBundle.command_Paste); 
		if (shell == null)
			throw new IllegalArgumentException("Unable to perform paste action due to 'shell == null'"); //$NON-NLS-1$
		this.shell = shell;
		if (clipboard == null)
			throw new IllegalArgumentException("Unable to perform paste action due to 'clipboard == null'"); //$NON-NLS-1$
		this.clipboard = clipboard;
		this.isResourceNav = false;
	}
	
	public void dispose() {
		if(clipboard != null) {
			clipboard.dispose();
		}
		clipboard = null;
		shell = null;
	}
		
	public void run() {
		Object target = selection != null ? selection.getFirstElement() : null;
		if(target != null) {			
			IPaster [] pasters = PasteExtensionManager.getInstance().getPasters(target);			
			for (int i = 0; i < pasters.length; i++) {
				IPaster paster = pasters[i];
				if (paster instanceof IPasterExtended) {
					((IPasterExtended)paster).setShell(shell);
				}
				if(paster.performPaste(clipboard, target)) {
					return;
				}
			}
		}
		//- special case of pasting a project regardless the selection
		IResource [] res = getResourceFromClipboard();
		boolean isProjectRes = res != null && res.length > 0 && res[0].getType() == IResource.PROJECT;
		if (isProjectRes) {
			for (int i = 0; i < res.length; i++) {
				CopyProjectOperation operation = new CopyProjectOperation(this.shell);
				operation.copyProject((IProject) res[i]);
			}
		}
	}

	private IResource [] getResourceFromClipboard() {
		final IResource[][] clipboardData = new IResource[1][];
		shell.getDisplay().syncExec(new Runnable() {
			public void run() {
				//- clipboard must have resources or files
				ResourceTransfer resTransfer = ResourceTransfer.getInstance();
				clipboardData[0] = (IResource[]) clipboard.getContents(resTransfer);
			}
		});
		return clipboardData[0];
	}	

	private boolean doesClipboardContainProjects() {
		//- if clipboard contains a project, paste is valid if project is open
		IResource [] res = getResourceFromClipboard();
		boolean isProjectRes = res != null && res.length > 0 && res[0].getType() == IResource.PROJECT;
		if (isProjectRes) {
			for (int i = 0; i < res.length; i++) {
				//- make sure that all resource data are open projects
				if (res[i].getType() != IResource.PROJECT || ((IProject) res[i]).isOpen() == false)
					return false;
			}
			return true;
		}
		return false;
	}
	
	private boolean isSelectionAccessibleResource() {
		// empty or multiple selection already tested
		Object select = selection.getFirstElement();
		if (select instanceof IResource) {
			if (((IResource)select).isAccessible()) {
				return true;
			}
		}
		return false;
	}
	
	public boolean isApplicableForSelection() {
		//- paste action is only allowed for resource view of the test navigator
		if(!isResourceNav) return false;
		//- regardless the selection if clipboard contains a project, this is possible to paste it.
		if (doesClipboardContainProjects()) {
			return true;
		}
		//- multiple selection for pasting is not allowed
		if (selection == null || selection.isEmpty() || selection.size() > 1) {
			return false;
		}	
		// can not paste to not accessible resource
		if (!isSelectionAccessibleResource()) {
			return false;
		}
		//- selection contains only one element
		//- is there at least one client registered that is able to perform a paste action in this context ?
		if(PasteExtensionManager.getInstance().isPasteAllowedFor(clipboard, selection.getFirstElement())) {
			return true;
		}
		return false;
	}
	
	public void selectionChanged(IStructuredSelection structuredSelection, boolean isResourceNav) {
		selection = structuredSelection;
		this.isResourceNav = isResourceNav;
	}
}
