/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RenameAction.java,v 1.20 2010/05/28 16:05:29 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.action;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigatorMessages;
import org.eclipse.hyades.test.ui.internal.navigator.action.resources.ActionMessages;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.RenameRefactoring;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IReferencerProxyNode;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer2;
import org.eclipse.hyades.test.ui.navigator.actions.RenamerUIStatus;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.hyades.ui.internal.navigator.TreeNavigator;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ltk.ui.refactoring.RefactoringWizard;
import org.eclipse.ltk.ui.refactoring.RefactoringWizardOpenOperation;
import org.eclipse.ltk.ui.refactoring.UserInputWizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TreeEditor;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.actions.TextActionHandler;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

/**
 * <p>RenameAction.java</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul E. Slauenwhite
 * @version May 28, 2010
 * @since   March 7, 2005
 */
public class RenameAction extends Action implements IDisposable {
	private IStructuredSelection selection;
	protected Composite textEditorParent;
	private TreeEditor treeEditor;
	protected Tree navigatorTree;
	protected Text textEditor;
	private TextActionHandler textActionHandler;
	protected TreeNavigator navigator;
	protected IContainer renamedResource;

	private class RenameRefactoringWizard extends RefactoringWizard {
		private RenameRefactoringNewNameInputPage page;

		private RenameRefactoringWizard(RenameRefactoring refactoring, String name,IContainer parent) {
			super(refactoring, DIALOG_BASED_USER_INTERFACE/* | NONE*/);
			this.page = new RenameRefactoringNewNameInputPage(refactoring, name,parent);
		}

		protected void addUserInputPages() {
			addPage(page);
		}
	}

	class RenameRefactoringNewNameInputPage extends UserInputWizardPage {
		private RenameRefactoring refactoring;
		private Text nameText;
		private String oldName;
		private IContainer parentContainer;
		
		/**
		 * <p>The new name.</p>
		 * 
		 * <p><b>Note:</b> Must be initialized to <code>null</code> (see {@link #isPageComplete()}).</p>
		 */
		private String newName = null;
		
		public RenameRefactoringNewNameInputPage(RenameRefactoring refactoring, String oldName,IContainer parentContainer) {
			super(""); //$NON-NLS-1$
			this.parentContainer = parentContainer;
			this.oldName = oldName;
			this.refactoring = refactoring;
		}

		public void createControl(Composite parent) {
			Composite result= new Composite(parent, SWT.NONE);
			setControl(result);
			result.setLayout(new GridLayout(2, false));
			
			Label label= new Label(result, SWT.NONE);
			label.setText(ActionMessages.RENAME_NEW_NAME_LABEL);
			nameText = new Text(result, SWT.SINGLE | SWT.BORDER);
			nameText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
			nameText.setText(oldName);
			nameText.addModifyListener(new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					newName = nameText.getText();
					refactoring.setNewName(newName);
					setPageComplete(isPageComplete());
				}
			});
			nameText.selectAll();
			Dialog.applyDialogFont(result);
		}

		public boolean isPageComplete() {

			//Case 1: The selection is missing (should never occur since this 
			//action is only enabled when a resource is selected):			
			if ((selection == null) || (selection.isEmpty())) {

				setMessage(null);
				setErrorMessage(RefactoringMessages.MISSING_RESOURCE);
				
				return false;
			}

			//Case 2: New name is missing:
			if((newName == null) || (newName.trim().length() == 0)){

				//Only display the error message after the user has entered a new name:
				if(newName != null){
					
					setMessage(RefactoringMessages.MISSING_NAME, WARNING);
					setErrorMessage(null);
				}
				
				return false;
			}

			//Case 3: New name is the same as the old name:
			else if(newName.equals(oldName)){

				setMessage(null);
				setErrorMessage(RefactoringMessages.DUPLICATE_NAME);
				
				return false;
			}
			setMessage(null);			
			switch(TestUIUtil.validateResourcePath(newName,parentContainer)) {
				case TestUIUtil.VALID_RESOURCE_PATH:
					setErrorMessage(null);					
					return true;
				case TestUIUtil.INVALID_RESOURCE_PATH :
				case TestUIUtil.INVALID_LOGICAL_NAME :
					setErrorMessage(RefactoringMessages.INVALID_NAME);					
					return false;
				case TestUIUtil.INVALID_RESOURCE_PATH_LENGTH :
				case TestUIUtil.INVALID_LOGICAL_NAME_LENGTH :
					setErrorMessage(RefactoringMessages.LONG_NAME);						
					return false;
				case TestUIUtil.UNAVAILABLE_RESOURCE_PATH:
				case TestUIUtil.UNAVAILABLE_LOGICAL_NAME :
					setErrorMessage(RefactoringMessages.DUPLICATE_NAME);					
					return false;
				default :
					// should not happen
					setErrorMessage(RefactoringMessages.MISSING_RESOURCE);					
					return true;
			}
		}
	}
	
	public RenameAction(TreeNavigator treeNav) {
		super(TestNavigatorMessages.RenameAction_ActionName);
		this.navigator = treeNav;
		navigatorTree = treeNav.getTreeViewer().getTree();
		this.treeEditor = new TreeEditor(navigatorTree);
		renamedResource = null;
	}

	public void dispose() {
		selection = null;
		disposeTextWidget();
	}

	public boolean isApplicableForSelection() {
		if (selection == null || selection.isEmpty()) {
			return false;
		}
		Object objSelected = selection.getFirstElement();
		if (objSelected instanceof IContainer) {
			return ((IContainer)objSelected).isAccessible();
		} else if (objSelected instanceof IProxyNode) {
			IProxyNode proxy = (IProxyNode) objSelected;
			IProxyNodeRenamer2 renamer2 = (IProxyNodeRenamer2)proxy.getAdapter(IProxyNodeRenamer2.class);
			if(renamer2 != null) {
				return true;
			}
			oldActionValidation(proxy);
		}
		return false;
	}

	/**
	 * @param proxy
	 * @return
	 * @deprecated
	 */
	private boolean oldActionValidation(IProxyNode proxy) {
		IProxyNodeRenamer renamer = (IProxyNodeRenamer) proxy.getAdapter(IProxyNodeRenamer.class);
		if (renamer != null) {
			return renamer.isApplicableFor();
		}
		return false;
	}

	/**
	 * This code has been imported from RenameResourceAction 
	 * @return composite
	 */
	private Composite createParent() {
		Composite result = new Composite(navigatorTree, SWT.NONE);
		TreeItem[] selectedItems = navigatorTree.getSelection();
		treeEditor.horizontalAlignment = SWT.LEFT;
		treeEditor.grabHorizontal = true;
		treeEditor.setEditor(result, selectedItems[0]);
		return result;
	}

	/**
	 * Create the text editor widget.
	 * This code has been imported from RenameResourceAction
	 * @param renamer
	 * @deprecated
	 */
	private void createTextEditor(final IProxyNodeRenamer renamer) {
		//- Create text editor parent.  This draws a nice bounding rect.
		textEditorParent = createParent();
		textEditorParent.setVisible(false);
		textEditorParent.addListener(SWT.Paint, new Listener() {
			public void handleEvent(Event e) {
				Point textSize = textEditor.getSize();
				Point parentSize = textEditorParent.getSize();
				e.gc.drawRectangle(0, 0, Math.min(textSize.x + 4, parentSize.x - 1), parentSize.y - 1);
			}
		});
		//- Create inner text editor.
		textEditor = new Text(textEditorParent, SWT.NONE);
		textEditor.setFont(navigatorTree.getFont());
		textEditorParent.setBackground(textEditor.getBackground());
		textEditor.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event e) {
				Point textSize = textEditor.computeSize(SWT.DEFAULT, SWT.DEFAULT);
				//- Add extra space for new characters.
				textSize.x += textSize.y;
				Point parentSize = textEditorParent.getSize();
				textEditor.setBounds(2, 1, Math.min(textSize.x, (parentSize.x - 4)), (parentSize.y - 2));
				textEditorParent.redraw();
			}
		});
		textEditor.addListener(SWT.Traverse, new Listener() {
			public void handleEvent(Event event) {
				switch (event.detail) {
					case SWT.TRAVERSE_ESCAPE :
						//- Do nothing in this case
						disposeTextWidget();
						event.doit = true;
						event.detail = SWT.TRAVERSE_NONE;
						break;
					case SWT.TRAVERSE_RETURN :
						performRename(renamer, textEditor.getText());
						disposeTextWidget();
						event.doit = true;
						event.detail = SWT.TRAVERSE_NONE;
						break;
				}
			}
		});
		textEditor.addFocusListener(new FocusAdapter() {
			public void focusLost(FocusEvent fe) {
				disposeTextWidget();
			}
		});
		if (textActionHandler != null)
			textActionHandler.addText(textEditor);
	}

	/**
	 * 
	 * @param renamer
	 * @param newName
	 * @deprecated
	 */
	protected void performRename(final IProxyNodeRenamer renamer, final String newName) {
		Runnable query = new Runnable() {
			public void run() {
				if(renamer == null) {
					//- this neams that a folder or a project node was renamed
					moveResource(renamedResource, newName);
				} else {
					boolean shouldRefresh = renamer.performRename(newName);
					if(shouldRefresh) {
						navigator.getTreeViewer().refresh();
					}
				}
				if (navigatorTree != null && !navigatorTree.isDisposed()) {
				    navigatorTree.setFocus();
				    //- find the new item to select it as it was before renaming
				    TreeItem[] items = navigatorTree.getItems();
				    findItem(newName, items);
				}
			}

			private void moveResource(final IResource res, String name) {
				final IPath newPath = res.getFullPath().removeLastSegments(1).append(name);
				WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
					public void execute(IProgressMonitor monitor) {
						try {
							res.move(newPath, IResource.KEEP_HISTORY | IResource.SHALLOW, new SubProgressMonitor(monitor, 50));
						} catch (CoreException e) {
							e.printStackTrace();
						}
					}
				};
				try {
					new ProgressMonitorDialog(Display.getCurrent().getActiveShell()).run(true, false, op);
				} catch (InvocationTargetException e) {
					e.printStackTrace();
				} catch (InterruptedException e) {
					//- can't be canceled
				}
			}
				
			private boolean findItem(final String name, TreeItem[] items) {
				for(int i = 0; i < items.length; i++) {
					if(items[i].getText().equals(name)) {
					    navigator.selectReveal(new StructuredSelection(items[i].getData()));
					    return true;
					}
					if(items[i].getItemCount() > 0) {
						if(findItem(name, items[i].getItems())) {
							return true;
						}
					}
				}
				return false;
			}
		};
		navigatorTree.getShell().getDisplay().asyncExec(query);
	}

	/**
	 * Close the text widget and reset the editorText field.
	 * This code has been imported from RenameResourceAction
	 */
	protected void disposeTextWidget() {
		if (textActionHandler != null)
			textActionHandler.removeText(textEditor);
		if (textEditorParent != null) {
			textEditorParent.dispose();
			textEditorParent = null;
			textEditor = null;
			treeEditor.setEditor(null, null);
		}
	}

	/**
	 * This code is inspired from RenameResourceAction
	 * @param oldName current value of the selected item
	 * @param renamer
	 * @deprecated
	 */
	private void getNewNameInline(String oldName, IProxyNodeRenamer renamer) {
		//- Make sure text editor is created only once. 
		if (textEditorParent == null) {
			createTextEditor(renamer);
		}
		textEditor.setText(oldName);
		//- Open text editor with initial size.
		textEditorParent.setVisible(true);
		Point textSize = textEditor.computeSize(SWT.DEFAULT, SWT.DEFAULT);
		//- Add extra space for new characters.
		textSize.x += textSize.y;
		Point parentSize = textEditorParent.getSize();
		textEditor.setBounds(2, 1, Math.min(textSize.x, (parentSize.x - 4)), (parentSize.y - 2));
		textEditorParent.redraw();
		textEditor.selectAll();
		textEditor.setFocus();
	}

	public void run() {
		Object objSelected = selection.getFirstElement();
		if (objSelected instanceof IContainer) {
			renamedResource = (IContainer)objSelected;
			preformContainerRefactoringRename(renamedResource);
		} else if (objSelected instanceof IProxyNode) {
			IProxyNode proxy = (IProxyNode) objSelected;
			IProxyNodeRenamer2 renamer2 = (IProxyNodeRenamer2)proxy.getAdapter(IProxyNodeRenamer2.class);
			if(renamer2 != null) {
				performProxyRefactoringRename(proxy);
			} else {
				IReferencerProxyNode ref = (IReferencerProxyNode)proxy.getAdapter(IReferencerProxyNode.class);
				if(ref != null) {
					performProxyRefactoringRename(proxy);
				} else {
					performOldRename(proxy);
				}
			}
		}
	}

	/**
	 * @param proxy
	 * @deprecated
	 */
	private void performOldRename(IProxyNode proxy) {
		//- this code will be removed in 5.0
		IProxyNodeRenamer renamer = (IProxyNodeRenamer) proxy.getAdapter(IProxyNodeRenamer.class);
		if (renamer != null) {
			RenamerUIStatus status = renamer.performUserInteraction(proxy.getText());
			switch (status.getStatus()) {
			case RenamerUIStatus.OK :
				performRename(renamer, status.getNewName());
				break;
			case RenamerUIStatus.CANCEL :
				//- nothing to do 
				break;
			case RenamerUIStatus.INLINE_EDITOR :
				getNewNameInline(proxy.getText(), renamer);
				break;
			}
		}
	}

	private void preformContainerRefactoringRename(IContainer container) {
		RenameRefactoring refactoring = new RenameRefactoring(container);
		performRefactoringRename(refactoring, container.getName(), container.getParent());
	}

	private void performProxyRefactoringRename(IProxyNode proxy) {
		RenameRefactoring refactoring = new RenameRefactoring(proxy);
		performRefactoringRename(refactoring, proxy.getText(),proxy.getUnderlyingResource().getParent());
	}

	private void performRefactoringRename(final RenameRefactoring refactoring, final String oldName,IContainer parent) {
		Shell shell = Display.getCurrent().getActiveShell();
		if (! canActivate(shell))
			return;
		RefactoringWizard wizard = new RenameRefactoringWizard(refactoring, oldName,parent);
		wizard.setDefaultPageTitle(ActionMessages.RENAME_ACTION_NAME);
		wizard.setNeedsProgressMonitor(true);
		RefactoringWizardOpenOperation op = new RefactoringWizardOpenOperation(wizard);
		try {
			op.run(shell , ""); //$NON-NLS-1$
		} catch (InterruptedException e) {
			UiPlugin.logError(e);
		}
	}

	private boolean canActivate(Shell shell) {
		return EditorUtil.saveEditors(shell);
	}

	public void selectionChanged(IStructuredSelection structuredSelection) {
		selection = structuredSelection;
	}
}