/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestNavigatorActionGroup.java,v 1.26 2010/11/30 13:12:59 bjerome Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.action;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigatorMessages;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.extension.INavigatorContribution;
import org.eclipse.hyades.ui.internal.action.CollapseAllAction;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.hyades.ui.internal.navigator.ITreeNavigator;
import org.eclipse.hyades.ui.internal.navigator.TreeNavigator;
import org.eclipse.hyades.ui.internal.navigator.action.FrameListActionGroup;
import org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup;
import org.eclipse.hyades.ui.internal.navigator.action.NavigatorSorterActionGroup;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.window.SameShellProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchCommandConstants;
import org.eclipse.ui.IWorkingSet;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.AddBookmarkAction;
import org.eclipse.ui.actions.AddTaskAction;
import org.eclipse.ui.actions.CloseResourceAction;
import org.eclipse.ui.actions.OpenResourceAction;
import org.eclipse.ui.actions.RefreshAction;
import org.eclipse.ui.actions.WorkingSetFilterActionGroup;
import org.eclipse.ui.dialogs.PropertyDialogAction;
import org.eclipse.ui.ide.IDEActionFactory;
import org.eclipse.ui.views.framelist.TreeFrame;

/**
 * <p>Group action for the Test Navigator.</p>
 * 
 * <p>This implementation is based on the {@link org.eclipse.ui.views.navigator.MainActionGroup} class.</p>
 * 
 * 
 * @author      Paul E. Slauenwhite
 * @author      Marcelo Paternostro
 * @author 		Jerome Bozier
 * @version 	November 30, 2010
 * @since       February 1, 2005
 */
public class TestNavigatorActionGroup 
extends NavigatorActionGroup
{
	private Clipboard clipboard;
	protected OpenEditorAction openAction;
	protected AddBookmarkAction addBookmarkAction;
	protected AddTaskAction addTaskAction;	
	protected PropertyDialogAction propertyDialogAction;
	protected CollapseAllAction collapseAllAction;
	protected ToggleLinkingAction toggleLinkingAction;
	protected ToggleFolderAction toggleFolderAction;
	protected ToggleEObjectChildrenAction toggleEObjectChildren;
	protected RefreshAction refreshAction;
	protected OpenResourceAction openProjectAction;
	protected CloseResourceAction closeProjectAction;
	protected StopAction stopAction;

	protected DeleteAction deleteAction;
	protected SelectAllAction selectAllAction;
	
	protected NavigatorSorterActionGroup sorterActionGroup;
	protected NewActionGroup newActionGroup;
	protected WorkingSetFilterActionGroup workingSetFilterActionGroup;
	protected FrameListActionGroup frameListActionGroup;
	protected RenameAction renameAction;
	protected CopyAction copyAction;
	protected PasteAction pasteAction;
	protected MoveAction moveAction;
    protected OpenWithActionGroup openWithActionGroup;
	 	
	
	/**
	 * Constructor for NavigatorActionGroup
	 * @param testNavigator
	 */
	public TestNavigatorActionGroup(TestNavigator testNavigator)
	{
		super(testNavigator);
	}
	
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#dispose()
	 */
	public void dispose()
	{
		newActionGroup.dispose();
        openWithActionGroup.dispose();
		workingSetFilterActionGroup.dispose();
		sorterActionGroup.dispose();
		frameListActionGroup.dispose();
		stopAction.dispose();
		renameAction.dispose();
		stopAction.dispose();
		pasteAction.dispose();
		moveAction.dispose();
		if(clipboard != null) {
			clipboard.dispose();
			clipboard = null;
		}
		super.dispose();
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#setContext(org.eclipse.ui.actions.ActionContext)
	 */
	public void setContext(ActionContext context)
	{
		super.setContext(context);
		
		sorterActionGroup.setContext(context);
		newActionGroup.setContext(context);
        openWithActionGroup.setContext(context);
		workingSetFilterActionGroup.setContext(context);
		frameListActionGroup.setContext(context); 
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#createActions()
	 */
	protected void createActions()
	{
		super.createActions();
		Shell shell = getNavigator().getSite().getShell();
		clipboard = new Clipboard(shell.getDisplay());
		
		openAction = new OpenEditorAction((TestNavigator)getNavigator(), UiPluginResourceBundle.ACT_OPEN);
		
		//Set the key binding to the F3 key:
		openAction.setActionDefinitionId("org.eclipse.hyades.test.ui.openCommand"); //$NON-NLS-1$
		
		addBookmarkAction = new AddBookmarkAction(shell);
		addTaskAction = new AddTaskAction(shell);
		
		propertyDialogAction = new PropertyDialogAction(new SameShellProvider(shell), getNavigator().getViewer());	

		//Set the key binding to the Alt+Enter keys:
		propertyDialogAction.setActionDefinitionId(IWorkbenchCommandConstants.FILE_PROPERTIES);
		
		collapseAllAction = new CollapseAllAction(((ITreeNavigator)getNavigator()).getTreeViewer());
		collapseAllAction.useDefaultDecorators();

		toggleLinkingAction = new ToggleLinkingAction(getNavigator(), TestNavigatorMessages.TestNavigatorActionGroup_ToggleAction_Text);
		toggleLinkingAction.setImageDescriptor(getImageDescriptor("elcl16/synced.gif")); //$NON-NLS-1$
		toggleLinkingAction.setHoverImageDescriptor(getImageDescriptor("elcl16/synced.gif")); //$NON-NLS-1$

		toggleFolderAction = new ToggleFolderAction((TestNavigator)getNavigator(), 
		        UiPluginResourceBundle.ACT_SHOW_FLD_TXT); 
		toggleFolderAction.setToolTipText(UiPluginResourceBundle.ACT_SHOW_FLD_DSC); 
		TestUIImages.INSTANCE.setImageDescriptors(toggleFolderAction, TestUIImages.ACT_SHOW_FOLDER);

		toggleEObjectChildren = new ToggleEObjectChildrenAction((TestNavigator)getNavigator(), 
		        UiPluginResourceBundle.ACT_SHOW_EOBJ_CH_TXT); 
		toggleEObjectChildren.setToolTipText(UiPluginResourceBundle.ACT_SHOW_EOBJ_CH_DSC); 
		TestUIImages.INSTANCE.setImageDescriptors(toggleEObjectChildren, TestUIImages.ACT_SHOW_EOBJECT_CHILDREN);

		ISharedImages images = PlatformUI.getWorkbench().getSharedImages();
		deleteAction = new DeleteAction(shell);
		
		//Set the key binding to the Delete key:
		deleteAction.setActionDefinitionId(IWorkbenchCommandConstants.EDIT_DELETE);
		deleteAction.setDisabledImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE_DISABLED));
		deleteAction.setImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE));		
		deleteAction.setHoverImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE));
		
		selectAllAction = new SelectAllAction(shell ,((ITreeNavigator)getNavigator()).getTreeViewer());
		
		//Set the key binding to the Ctrl-A keys:
		selectAllAction.setActionDefinitionId(IWorkbenchCommandConstants.EDIT_SELECT_ALL);

		refreshAction = new RefreshAction(shell);
		
		//Set the key binding to the F5 key:
		refreshAction.setActionDefinitionId(IWorkbenchCommandConstants.FILE_REFRESH);
		refreshAction.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor("e", TestUIImages.IMG_REFRESH_NAV));//$NON-NLS-1$	
		openProjectAction = new OpenResourceAction(shell){

			/* (non-Javadoc)
			 * @see org.eclipse.ui.actions.OpenResourceAction#run()
			 */
			public void run() {

				super.run();
			
				//Problem:  The Properties view is not updated after the selected project is opened.
				//Cause: The Properties view is only updated when the selection changes and not when the selected project is updated (e.g. opened).
				//Note: Reselecting the same selection in the Test Navigator by notifying the registered selection listeners 
				//has no affect since the Properties view (see PropertySheet#selectionChanged(IWorkbenchPart, ISelection)) shortcuts 
				//the same selection, for performance reasons.
				//Work-around: De-select and reselect the selection in the Test Navigator, after the project is opened, to 
				//force an update of the Properties view.
				Display.getDefault().asyncExec(new Runnable() {

					/* (non-Javadoc)
					 * @see java.lang.Runnable#run()
					 */
					public void run(){	

						try {

							TreeViewer testNavigatorTreeViewer = ((TestNavigator)(getNavigator())).getTreeViewer();
							IStructuredSelection structuredSelection = ((IStructuredSelection)(testNavigatorTreeViewer.getSelection()));

							//Note: The Properties view is not displayed when multiple resources are selected.
							if(structuredSelection.size() == 1){

								IProject project = null;
								Object firstElement = structuredSelection.getFirstElement();

								if (firstElement instanceof IProject) {
									project = ((IProject)(firstElement));
								} 
								else if (firstElement instanceof IAdaptable) {
									project = ((IProject)((IAdaptable)(firstElement)).getAdapter(IProject.class));
								}

								if(project != null){

									//Wait for at most 2000 milliseconds (or 2 seconds) for the open operation to complete:
									for (int counter = 0; counter < 10; counter++) {

										if(project.isOpen()){

											testNavigatorTreeViewer.setSelection(StructuredSelection.EMPTY);
											testNavigatorTreeViewer.setSelection(structuredSelection);

											break;
										}
										else{
											Thread.sleep(200);
										}
									}
								}							
							}
						} 
						catch (Exception e) {
							UiPlugin.logError(e);
						}
					}	
				});
			}
		};
		
		closeProjectAction = new CloseResourceAction(shell){

			/* (non-Javadoc)
			 * @see org.eclipse.ui.actions.OpenResourceAction#run()
			 */
			public void run() {

				super.run();

				//Problem:  The Properties view is not updated after the selected project is closed.
				//Cause: The Properties view is only updated when the selection changes and not when the selected project is updated (e.g. closed).
				//Note: Reselecting the same selection in the Test Navigator by notifying the registered selection listeners 
				//has no affect since the Properties view (see PropertySheet#selectionChanged(IWorkbenchPart, ISelection)) shortcuts 
				//the same selection, for performance reasons.
				//Work-around: De-select and reselect the selection in the Test Navigator, after the project is closed, to 
				//force an update of the Properties view.
				Display.getDefault().asyncExec(new Runnable() {

					/* (non-Javadoc)
					 * @see java.lang.Runnable#run()
					 */
					public void run(){	

						try {

							TreeViewer testNavigatorTreeViewer = ((TestNavigator)(getNavigator())).getTreeViewer();
							IStructuredSelection structuredSelection = ((IStructuredSelection)(testNavigatorTreeViewer.getSelection()));

							//Note: The Properties view is not displayed when multiple resources are selected.
							if(structuredSelection.size() == 1){

								IProject project = null;
								Object firstElement = structuredSelection.getFirstElement();

								if (firstElement instanceof IProject) {
									project = ((IProject)(firstElement));
								} 
								else if (firstElement instanceof IAdaptable) {
									project = ((IProject)((IAdaptable)(firstElement)).getAdapter(IProject.class));
								}

								if(project != null){

									//Wait for at most 2000 milliseconds (or 2 seconds) for the close operation to complete:
									for (int counter = 0; counter < 10; counter++) {

										if(!project.isOpen()){

											testNavigatorTreeViewer.setSelection(StructuredSelection.EMPTY);
											testNavigatorTreeViewer.setSelection(structuredSelection);

											break;
										}
										else{
											Thread.sleep(200);
										}
									}
								}							
							}
						} 
						catch (Exception e) {
							UiPlugin.logError(e);
						}
					}					
				});		
			}
		};
		
		//Required to enable/disable the 'Project -> Open/Close Project' file menus:
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		workspace.addResourceChangeListener(openProjectAction);
		workspace.addResourceChangeListener(closeProjectAction);
		
		stopAction = new StopAction();	
		
		renameAction = new RenameAction((TreeNavigator)getNavigator());
	
		//Set the key binding to the F2 key:
		renameAction.setActionDefinitionId(IWorkbenchCommandConstants.FILE_RENAME);
		
		pasteAction = new PasteAction(shell, clipboard);

		//Set the key binding to the Ctrl+V keys:
		pasteAction.setActionDefinitionId(IWorkbenchCommandConstants.EDIT_PASTE);
		pasteAction.setImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_PASTE));
		pasteAction.setDisabledImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_TOOL_PASTE_DISABLED));
		
		copyAction = new CopyAction(clipboard);
		
		//Set the key binding to the Ctrl+C keys:
		copyAction.setActionDefinitionId(IWorkbenchCommandConstants.EDIT_COPY);
		copyAction.setImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_COPY));
		copyAction.setDisabledImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_TOOL_COPY_DISABLED));
		moveAction = new MoveAction(shell);
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#createActionGroups()
	 */
	protected void createActionGroups()
	{
		super.createActionGroups();
		sorterActionGroup = new NavigatorSorterActionGroup(getNavigator());
		newActionGroup = new NewActionGroup((TestNavigator)getNavigator());
        openWithActionGroup = new OpenWithActionGroup((TestNavigator)getNavigator());
		
		IPropertyChangeListener workingSetUpdater = new IPropertyChangeListener()
		{
			public void propertyChange(PropertyChangeEvent event)
			{
				String property = event.getProperty();
				
				if (WorkingSetFilterActionGroup.CHANGE_WORKING_SET.equals(property))
				{
					INavigator navigator = getNavigator();
					Object newValue = event.getNewValue();
					
					if (newValue instanceof IWorkingSet)
					{	
						((TestNavigator)navigator).setWorkingSet((IWorkingSet) newValue);
					}
					else if (newValue == null)
					{
						((TestNavigator)navigator).setWorkingSet(null);
					}
				}
			}
		};
		workingSetFilterActionGroup = new WorkingSetFilterActionGroup(getNavigator().getViewer().getControl().getShell(), workingSetUpdater);
		workingSetFilterActionGroup.setWorkingSet(((TestNavigator)getNavigator()).getWorkingSet());
		frameListActionGroup = new FrameListActionGroup((TestNavigator)getNavigator());
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#fillContextMenu(org.eclipse.jface.action.IMenuManager)
	 */
	public void fillContextMenu(IMenuManager menu)
	{
		super.fillContextMenu(menu);
		boolean hasClosedProjects = false;
		boolean hasOpenProjects = false;
		boolean isProjectSelection = true;
		
		IStructuredSelection selection = (IStructuredSelection)getContext().getSelection();
		IStructuredSelection resSelection = getResourceSelection(selection);
		Iterator resources = selection.iterator();
		
		while (resources.hasNext() &&
				(!hasOpenProjects || !hasClosedProjects || isProjectSelection)) {
			Object next = resources.next();
			IProject project = null;
			
			if (next instanceof IProject) {
				project = (IProject) next;
			} else if (next instanceof IAdaptable) {
				project = (IProject) ((IAdaptable) next).getAdapter(IProject.class);
			}
			if (project == null) {
				isProjectSelection = false;
			} else {
				if (project.isOpen()) {
					hasOpenProjects = true;
				} else {
					hasClosedProjects = true;
				}
			}
		}	

		newActionGroup.fillContextMenu(menu);
		frameListActionGroup.fillContextMenu(menu);
		
		openAction.selectionChanged(selection);
		if(openAction.isApplicableForSelection())
		{
			menu.appendToGroup(INavigatorContribution.GROUP_OPEN, openAction);
		}
		openWithActionGroup.fillContextMenu(menu);
		
		if (!hasClosedProjects) {
			refreshAction.selectionChanged(selection);
			menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, refreshAction);
		}
		
		if (isProjectSelection) {
			if (hasClosedProjects) {
				openProjectAction.selectionChanged(selection);
				menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, openProjectAction);				
			}
			if (hasOpenProjects) {
				closeProjectAction.selectionChanged(selection);
				menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, closeProjectAction);
			}
		}					
		
		stopAction.selectionChanged(selection);
		if(stopAction.isApplicableForSelection())
		{
			menu.appendToGroup(INavigatorContribution.GROUP_DELETE, stopAction);
			stopAction.setEnabled(stopAction.isApplicableForEnablement());
		}
		boolean isResourceTestNav = ((TestNavigator)getNavigator()).getCurrentViewIndex()==TestNavigator.VIEW_ID_RESOURCE;
		copyAction.selectionChanged(selection, isResourceTestNav);
		if(copyAction.isApplicableForSelection()){
				menu.appendToGroup(INavigatorContribution.GROUP_DELETE, copyAction);
		}
		
		pasteAction.selectionChanged(selection, isResourceTestNav);
		if(pasteAction.isApplicableForSelection()){
			menu.appendToGroup(INavigatorContribution.GROUP_DELETE, pasteAction);
		}
					
		moveAction.selectionChanged(selection, isResourceTestNav);
		if(moveAction.isApplicableForSelection()){
			menu.appendToGroup(INavigatorContribution.GROUP_DELETE, moveAction);
		}
		
		deleteAction.selectionChanged(selection);
		if(deleteAction.isApplicableForSelection()) {
			menu.appendToGroup(INavigatorContribution.GROUP_DELETE, deleteAction);
		}
		
		if(selectAllAction.isApplicableForSelection()) {
			menu.appendToGroup(INavigatorContribution.GROUP_DELETE, selectAllAction);
		}
		
		renameAction.selectionChanged(selection);
		if(renameAction.isApplicableForSelection())	{
			menu.appendToGroup(INavigatorContribution.GROUP_DELETE, renameAction);
		}

		if(resSelection.size() == selection.size() && selection.size() > 0)
		{
			addBookmarkAction.selectionChanged(resSelection);
			if(addBookmarkAction.isEnabled())
				menu.appendToGroup(INavigatorContribution.GROUP_DELETE, addBookmarkAction);
		}

		if(selection.size() == 1) {
			propertyDialogAction.selectionChanged(selection);
			if(propertyDialogAction.isApplicableForSelection())
				menu.appendToGroup(INavigatorContribution.GROUP_PROPERTIES, propertyDialogAction);
		}
	}
		
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#fillActionBars(org.eclipse.ui.IActionBars)
	 */
	public void fillActionBars(IActionBars actionBars)
	{
		actionBars.setGlobalActionHandler(ActionFactory.PROPERTIES.getId(),	propertyDialogAction);
		actionBars.setGlobalActionHandler(IDEActionFactory.BOOKMARK.getId(), addBookmarkAction);
		actionBars.setGlobalActionHandler(IDEActionFactory.ADD_TASK.getId(), addTaskAction);
		actionBars.setGlobalActionHandler(ActionFactory.DELETE.getId(), deleteAction);
		actionBars.setGlobalActionHandler(IDEActionFactory.OPEN_PROJECT.getId(), openProjectAction);
		actionBars.setGlobalActionHandler(IDEActionFactory.CLOSE_PROJECT.getId(), closeProjectAction);
				
		frameListActionGroup.fillActionBars(actionBars);
		newActionGroup.fillActionBars(actionBars);
        openWithActionGroup.fillActionBars(actionBars);
		workingSetFilterActionGroup.fillActionBars(actionBars);
		sorterActionGroup.fillActionBars(actionBars);
		
		super.fillActionBars(actionBars);

		IMenuManager menu = actionBars.getMenuManager();
		menu.add(new Separator());
		menu.add(toggleLinkingAction);
		//menu.add(toggleFolderAction);
		menu.add(toggleEObjectChildren);
		
		IToolBarManager toolBar = actionBars.getToolBarManager();
		toolBar.add(new Separator());
		toolBar.add(collapseAllAction);		
		toolBar.add(toggleLinkingAction);
        toolBar.add(new Separator());
	}
	
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#updateActionBars()
	 */
	public void updateActionBars()
	{
		TestNavigator testNavigator = (TestNavigator)getNavigator();
		Object currentFrameInput = ((TreeFrame)testNavigator.getFrameList().getCurrentFrame()).getInput();
		toggleFolderAction.setEnabled((currentFrameInput == null) || (testNavigator.getInitialViewerInput() == currentFrameInput));
		toggleEObjectChildren.setEnabled((currentFrameInput == null) || (!(currentFrameInput instanceof EObject)));
		
		IStructuredSelection structuredSelection = (IStructuredSelection) getContext().getSelection();
		IStructuredSelection resourceStructuredSelection = getResourceSelection(structuredSelection);
		
		propertyDialogAction.setEnabled(structuredSelection.size() == 1);
		addBookmarkAction.selectionChanged(resourceStructuredSelection);
		deleteAction.selectionChanged(structuredSelection);
		addTaskAction.selectionChanged(structuredSelection);	
		refreshAction.selectionChanged(structuredSelection);
		openProjectAction.selectionChanged(structuredSelection);
		closeProjectAction.selectionChanged(structuredSelection);
		boolean isResourceTestNav = ((TestNavigator)getNavigator()).getCurrentViewIndex()==TestNavigator.VIEW_ID_RESOURCE;
		copyAction.selectionChanged(structuredSelection, isResourceTestNav);
		pasteAction.selectionChanged(structuredSelection, isResourceTestNav);
		moveAction.selectionChanged(structuredSelection, isResourceTestNav);
		renameAction.selectionChanged(structuredSelection);

	
		frameListActionGroup.updateActionBars();
		sorterActionGroup.updateActionBars();
		newActionGroup.updateActionBars();
        openWithActionGroup.updateActionBars();
		workingSetFilterActionGroup.updateActionBars();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#runDefaultAction(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void runDefaultAction(IStructuredSelection selection)
	{
		openAction.selectionChanged(selection);
		openAction.run();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#handleKeyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void handleKeyPressed(KeyEvent event)
	{
		if(event.character == SWT.DEL && (event.stateMask == SWT.None))
		{
			if(deleteAction.isEnabled())
				deleteAction.run();
				//- Swallow the event.
				event.doit = false;
		}

		if(event.keyCode == SWT.F3 && event.stateMask == SWT.None) {
			if(openAction.isEnabled()) {
				openAction.run();
				//- Swallow the event.
				event.doit = false;
			}
		}

		if(event.keyCode == SWT.F5 && event.stateMask == SWT.None) {
			if(refreshAction.isEnabled()) {
				refreshAction.run();
				//- Swallow the event.
				event.doit = false;
			}
		}
		if(event.keyCode == SWT.F2 && event.stateMask == SWT.None) {
			if(renameAction.isEnabled()) {
				renameAction.run();
			}
			//- Swallow the event.
			event.doit = false;		
		}
		
		super.handleKeyPressed(event);
	}

	/**
	 * Returns an structured selection with resource elements based on the given
	 * structured selection.
	 * @param structuredSelection
	 * @return IStructuredSelection
	 * @see TestNavigator#getFile(Object)
	 */
	protected IStructuredSelection getResourceSelection(IStructuredSelection structuredSelection)
	{
		TestNavigator testNavigator = (TestNavigator)getNavigator();
		List resources = new ArrayList();
		for(Iterator i = structuredSelection.iterator(); i.hasNext();)
		{
			Object object = i.next();
			if((object instanceof IResource) && (((IResource)object).getType() != IResource.FILE))
			{
				resources.add(object);
			}
			else
			{
				IFile file = testNavigator.getFile(object);
				if(file != null)
					resources.add(file);				
			}			
		}
		return new StructuredSelection(resources);
	}

	/**
	 * Returns the image descriptor from the eclipse UI plugin with the given relative 
	 * path.
	 * @return ImageDescriptor;
	 */
	protected ImageDescriptor getImageDescriptor(String relativePath)
	{
		String iconPath = "icons/full/"; //$NON-NLS-1$
		try
		{
			URL installURL = Platform.getBundle(PlatformUI.PLUGIN_ID).getEntry("/");//$NON-NLS-1$
			URL url = new URL(installURL, iconPath + relativePath);
			return ImageDescriptor.createFromURL(url);
		}
		catch(MalformedURLException e)
		{
			return ImageDescriptor.getMissingImageDescriptor();
		}
	}	
	
	/** returns the rename action defined for test navigator
	 * @return rename action
	 */
	public IAction getRenameAction() {
		return renameAction;
	}

	/** retruns the copy action defined for the test navigator view
	 * @return copy action
	 */
	public IAction getCopyAction() {
		return copyAction;
	}

	/** retruns the paste action defined for the test navigator view
	 * @return paste action
	 */
	public IAction getPasteAction() {
		return pasteAction;
	}

	/** retruns the refresh action defined for the test navigator view
	 * @return refresh action
	 */
	public IAction getRefreshAction() {
		return refreshAction;
	}
	
	/** retruns the move action defined for the test navigator view
	 * @return refresh action
	 */
	public IAction getMoveAction() {
		return moveAction;
	}
	
}
