/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DefaultTypeProvider.java,v 1.14 2007/03/01 19:35:41 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNodeListener;
import org.eclipse.hyades.test.ui.navigator.ITypeProvider;
import org.eclipse.hyades.test.ui.navigator.ITypeProviderContext;
import org.eclipse.hyades.test.ui.navigator.ITypeProviderProxyNode;
import org.eclipse.hyades.ui.util.IDisposable;

/** 
 * This type provider is used if nobody provides a better one for a given type.
 * It returns a resource based proxy nodes tree filtered on the given type.
 * @author jgout
 * @since 4.0
 */
public class DefaultTypeProvider implements ITypeProvider, IResourceChangeListener, IDisposable {
	
    /**
     * cache Map: project -> ITypeProviderProxyNode
     */
    private Map cache;
    private IProxyNodeListener refresher;
    private IFileProxyManager fileProxyManager;

    public DefaultTypeProvider() {
        cache = Collections.synchronizedMap(new HashMap()); 
        //- register this class as a resource change listener to the proxy node cache
        FileProxyNodeCache.getInstance().addResourceListener(this);
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.ITypeProvider#init(org.eclipse.hyades.test.ui.navigator.ITypeProviderContext)
     */
    public void init(ITypeProviderContext context) {
        fileProxyManager = context.getFileProxyManager();
		refresher = context.getProxyNodeListener();
    }
    
    public ITypeProviderProxyNode get(IProject project, String type) {
        if (!project.exists()) return null;
        ITypeProviderProxyNode proxy = (ITypeProviderProxyNode)cache.get(project);
        if(proxy == null) {
            //- proxy node is not yet created
            proxy = new DefaultTypeProviderProxyNode(project, fileProxyManager, type,  project);
            //- cache the proxy
            cache.put(project, proxy);
        }
        //- return proxy only if it has some children
        return proxy.getChildren().length > 0 ? proxy : null;
	}
   
    public void resourceChanged(IResourceChangeEvent event) {
        IResourceDelta resDelta = event.getDelta();
        //- get all projects affected by the current change
        IResourceDelta[] affectedProjects = resDelta.getAffectedChildren();
        for (int i = 0; i < affectedProjects.length; i++) {
        	Object lowestChanged = null;
            IResourceDelta delta = affectedProjects[i];
            IProject project = (IProject) delta.getResource();
            DefaultTypeProviderProxyNode proxy = (DefaultTypeProviderProxyNode) cache.get(project);
            // proxy may be null if we receive a resource change event before get() is invoked:
            // in this case, there are no proxies to update.
            if (delta.getKind() == IResourceDelta.REMOVED) {
            	cache.remove(project);
            } else if (delta.getKind() == IResourceDelta.CHANGED) {
            	if (proxy == null) {
            		//lowestChanged = project;
            	} else {
	            	int previousChildrenCount = proxy.getChildren().length;
		            lowestChanged = proxy.resourceChanged(delta);
		            if (lowestChanged == proxy) {
						int newChildrenCount = proxy.getChildren().length;
						if ((previousChildrenCount == 0 || newChildrenCount == 0) && previousChildrenCount != newChildrenCount) {
							lowestChanged = project;
						}
					}
            	}
	            if(lowestChanged != null) {
	            	refresher.nodeChanged(lowestChanged);
	            }
            }
        }
    }
    
    public void clear() {
        cache.clear();
    }

    public void dispose() {
        FileProxyNodeCache.getInstance().removeResourceListener(this);        
    }

}