/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EMFResourceProxyFactory.java,v 1.21 2010/05/07 13:03:06 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.core.internal.util.EMFResourceReader;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.navigator.FileProxyNode;
import org.eclipse.hyades.test.ui.navigator.HyadesProxyNodeFactory;
import org.eclipse.hyades.test.ui.navigator.IFileProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IPersistableFileProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.ui.IMemento;

/** 
 * <p>Generic factory for creating {@link IProxyNode}s from EMF resources
 * and recreating {@link FileProxyNode}s from {@link IMemento}s.</p>
 * 
 * <p>EMF resources are loaded from EMF resource files in the Eclipse workspace 
 * containing serialized TPTP models.</p>

 * <p>Supported TPTP models include:</p>
 * 
 * <ul>
 * <li>TPFTestSuite</li>
 * <li>TPFTestCase</li>
 * <li>TPFExecutionResult</li>
 * <li>TPFTestComponent</li>
 * <li>DPLDatapool</li>
 * <li>TPFDeployment</li>
 * <li>CFGArtifact</li>
 * <li>CFGLocation</li>
 * </ul>
 * 
 * <p>When creating a {@link IProxyNode}, this factory handles the following<sup>1</sup>:</p>
 * 
 * <ul>
 * <li>Loads the model(s) from the EMF resource into memory.</li>
 * <li>Creates a typed {@link IProxyNode} with the loaded model, if the EMF resource contains a single model.</li>
 * <li>Creates a typed {@link FileProxyNode} (or proxy tree) with the loaded resource, if the EMF resource contains two or more models.</li>
 * <li>Unloads the model(s) from the EMF resource from memory.</li>
 * </ul>
 * 
 * <p><sup>1</sup>For performance reasons, {@link DatapoolProxyNode}s are not created using the same process 
 * due to the large size of datapools.  Instead, a {@link DatapoolProxyNode} is created from reading the root 
 * properties from the serialized EMF model in the datapool file (*.datapool) using {@link EMFResourceReader}.</p>
 * 
 * <p>When recreating a {@link FileProxyNode}, this factory handles the following<sup>1</sup>:</p>
 * 
 * <ul>
 * <li>Loads the proxy state from the previously saved {@link IMemento}.</li>
 * <li>Creates a typed {@link FileProxyNode} with the loaded proxy state.</li>
 * </ul>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version May 7, 2010
 * @since   February 1, 2005
 */
public class EMFResourceProxyFactory implements IFileProxyFactory, IPersistableFileProxyFactory {
	
	/**
	 * Factory id.
	 */
	public final static String ID = "EMFResourceProxyFactory_4.3"; //$NON-NLS-1$
	
	/**
	 * <p>Creates a {@link IProxyNode} from a EMF resource loaded from the EMF resource file 
	 * in the Eclipse workspace containing one ore more serialized TPTP models.</p>
	 * 
	 * <p>Supported TPTP models include:</p>
     * 
     * <ul>
     * <li>TPFTestSuite</li>
     * <li>TPFTestCase</li>
     * <li>TPFExecutionResult</li>
     * <li>TPFTestComponent</li>
     * <li>DPLDatapool</li>
     * <li>TPFDeployment</li>
     * <li>CFGArtifact</li>
     * <li>CFGLocation</li>
     * </ul>
     * 
     * <p>This method returns <code>null</code> if the EMF resource file:</p>
     * 
     * <ul>
     * <li>does not exist in the Eclipse workspace, or</li>
     * <li>does not contain a serialized TPTP model, or</li>
     * <li>the serialized TPTP model(s) could not be read/loaded.</li>
     * </ul>
     *      
     * <p>This method handles the following<sup>1</sup>:</p>
     * 
     * <ul>
     * <li>Loads the model(s) from the EMF resource into memory.</li>
     * <li>Creates a typed {@link IProxyNode} with the loaded model, if the EMF resource contains a single model.</li>
     * <li>Creates a typed {@link FileProxyNode} with the loaded resource, if the EMF resource contains two or more models.</li>
     * <li>Unloads the model(s) from the EMF resource from memory.</li>
     * </ul>
     * 
     * <p><sup>1</sup>For performance reasons, {@link DatapoolProxyNode}s are not created using the same process 
     * due to the large size of datapools.  Instead, a {@link DatapoolProxyNode} is created from reading the root 
     * properties from the serialized EMF model in the datapool file (*.datapool) using {@link EMFResourceReader}.</p>
     * 
	 * @param file The EMF resource file in the Eclipse workspace containing one or more serialized TPTP models.
	 * @param parent An existing object (possibly another proxy node) which is the parent of the newly created {@link IProxyNode} in the proxy node hierarchy.
	 * @return The newly created {@link IProxyNode}, otherwise <code>null</code>.
	 * @see org.eclipse.hyades.test.ui.navigator.IFileProxyFactory#create(org.eclipse.core.resources.IFile, java.lang.Object)
	 */
	public IProxyNode create(IFile file, Object parent) {
		
		//Resolve the URI to the file in the Eclipse workspace:
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString(), false);
		
		//Creating a proxy node requires a valid URI:
		if(uri != null){

			//Resolve the extension of the file in the Eclipse workspace:
			String extension = file.getFileExtension();
			
			//For performance reasons, special case datapool files (*.datapool) due to their large size so they are NOT loaded into memory:
			if((extension != null) && (extension.trim().equalsIgnoreCase(ICommonConstants.DATAPOOL_FILE_EXTENSION))) {

				//Assumption: An EMF resource resolved from a datapool file (*.datapool) will contain only one model.
				//Resolve the datapool properties from reading the EMF resource in the datapool file (*.datapool):
				Map datapoolProperties = EMFResourceReader.getRootProperties(file);

				//Resolve the required datapool properties:
				String name = ((String)(datapoolProperties.get("name"))); //$NON-NLS-1$
				String id = ((String)(datapoolProperties.get("id"))); //$NON-NLS-1$

				//Create a datapool proxy node with the required datapool properties: 
				if((name != null) && (name.trim().length() > 0) && (id != null) && (id.trim().length() > 0)){
					return (new DatapoolProxyNode(name.trim(), ((String)(datapoolProperties.get("description"))), uri.appendFragment(id.trim()), parent)); //$NON-NLS-1$
				}
			}

			//Resolve the EMF resource for the file in the Eclipse workspace:
			Resource resource = null;

			//Create proxies for only files that exist:
			if (file.exists()) {

				try{
					resource = EMFUtil.getResource(new ResourceSetImpl(), uri, true);
				}
				catch (Exception e) {					
					//Ignore and return null since the resource could not be loaded (e.g. not completely serialized).
				}

				//Creating a proxy node requires a valid EMF resource:
				if(resource != null) {
					
					try {

						//Resolve the model(s) from the EMF resource:
						EObject[] eObjects = ((EObject[])(resource.getContents().toArray(new EObject[resource.getContents().size()]))); 				

						//Create a proxy node for the model in the EMF resource: 
						if (eObjects.length == 1) {
							return (HyadesProxyNodeFactory.getInstance().create(eObjects[0], parent));
						} 

						//Create a file proxy node for the multiple models in the EMF resource: 
						else if (eObjects.length > 1) {
							return (createFileProxy(file, resource));
						}
					}

					finally {

						//Unload the resource for the file:
						resource.unload();
					}
				} 
			} 
		}

		return null;
	}
	
	/** Create a proxy for the given file using the memento. This memento is the proxy state stored previously.
	 * 
	 * @param file the file to convert
	 * @param memento the state of the proxy previously saved by IPersistableProxyNode.saveState().
	 * @retrun the root of the proxy node tree
	 * @see org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode
	 */
	public IProxyNode recreate(IMemento memento, IFile file, Object parent) {
		//- try to retrieve one of the default hyades proxy nodes
		IProxyNode proxy = HyadesProxyNodeFactory.getInstance().recreate(memento, parent);
		if(proxy == null) {
            //- this means that this memento represents a file proxy node
			String nodeKind = memento.getString(TestUIConstants.TAG_NODE_KIND);
			IMemento[] mementoChildren = memento.getChildren(TestUIConstants.TAG_CHILD);
            for (int i = 0; i < mementoChildren.length; i++) {
                IMemento mementoChild = mementoChildren[i];
                //- inherit the TAG_URI_ROOT which is an extra data contained by the parent
                mementoChild.putString(TestUIConstants.TAG_URI_ROOT, memento.getString(TestUIConstants.TAG_URI_ROOT));               
            }
            if(nodeKind.equals(TestUIConstants.ARTIFACT_FILE_NODE)) {
				return new ArtifactFileProxyNode(file, memento);
			} else if(nodeKind.equals(TestUIConstants.DATAPOOL_FILE_NODE)) {
				return new DatapoolFileProxyNode(file, memento);
			} else if(nodeKind.equals(TestUIConstants.DEPLOY_FILE_NODE)) {
				return new DeploymentFileProxyNode(file, memento);
			} else if(nodeKind.equals(TestUIConstants.EXECUTION_FILE_NODE)) {
				return new ExecutionFileProxyNode(file, memento);
			} else if(nodeKind.equals(TestUIConstants.LOCATION_FILE_NODE)) {
				return new LocationFileProxyNode(file, memento);
			} else if(nodeKind.equals(TestUIConstants.TESTCOMPONENT_FILE_NODE)) {
				return new TestComponentFileProxyNode(file, memento);
			} else if(nodeKind.equals(TestUIConstants.TESTSUITE_FILE_NODE)) {
				return new TestSuiteFileProxyNode(file, memento);
			}
		} else {
			return proxy;
		}
		return null;
	}
	
	/** This method is called to build a node that contains multiple EMF objects as children.
	 *  The object created depends of the extension of the file (.datapool -> DatapoolFileProxyNode)
	 *  using as name the name of the file since if there are multiple objects inside, we can't
	 *  know which one should be used to name the node.
	 * @param file
	 * @return file level proxy node
	 */
	private FileProxyNode createFileProxy(IFile file, Resource resource) {
		
		String extension = file.getFileExtension();

		if(extension != null) {
			
			if(extension.equalsIgnoreCase(ICommonConstants.ARTIFACT_FILE_EXTENSION)) {
				return (new ArtifactFileProxyNode(file, resource));
			} 
			else if(extension.equalsIgnoreCase(ICommonConstants.DATAPOOL_FILE_EXTENSION)) {
				return (new DatapoolFileProxyNode(file, resource));
			} 
			else if(extension.equalsIgnoreCase(ICommonConstants.LOCATION_FILE_EXTENSION)) {
				return (new LocationFileProxyNode(file, resource));
			} 
			else if(extension.equalsIgnoreCase(ICommonConstants.DEPLOYMENT_FILE_EXTENSION)) {
				return (new DeploymentFileProxyNode(file, resource));
			} 
			else if(extension.equalsIgnoreCase(ICommonConstants.TEST_SUITE_FILE_EXTENSION)) {
				return (new TestSuiteFileProxyNode(file, resource));
			}  
			else if(extension.equalsIgnoreCase(ICommonConstants.TEST_COMPONENT_EXTENSION)) {
				return (new TestComponentFileProxyNode(file, resource));
			}  
			else if(extension.equalsIgnoreCase(ICommonConstants.EXECUTION_FILE_EXTENSION)) {
				return (new ExecutionFileProxyNode(file, resource));
			} 
		} 

		return null;
	}
}
