/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileFactoryManager.java,v 1.8 2010/02/18 14:49:14 bjerome Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.navigator.IFileProxyFactory;

/**
 * <p>File factory manager.</p>
 * 
 * <p>This class manages the <code>org.eclipse.hyades.test.ui.testNavigatorFileProxyFactory</code> extension point.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version February 4, 2010
 * @since   January 5, 2006
 */
public class FileFactoryManager {
	
	private Map<String, IFileProxyFactory> factoryIDToFactoryMap = null;
	private Map<String, List<IFileProxyFactory>> fileExtensionToFactoriesMap = null;
	private Map<String, Boolean> fileExtensionToCase = null;
	
	/**
	 * No-argument constructor.
	 */
	public FileFactoryManager() {
		
		factoryIDToFactoryMap = new HashMap<String, IFileProxyFactory>();
		fileExtensionToCase = new HashMap<String, Boolean>();

		HashMap<String, List<String>> fileExtensionToFactoryIDsMap = new HashMap<String, List<String>>();

		//Step 1: Resolve the org.eclipse.hyades.test.ui.testNavigatorFileProxyFactory extension point:
		IExtensionPoint extensionPoint = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + ".testNavigatorFileProxyFactory"); //$NON-NLS-1$
		
		if (extensionPoint != null) {
			
			IConfigurationElement[] configurationElements = extensionPoint.getConfigurationElements();
			
			for (int counter = 0; counter < configurationElements.length; counter++) {
				
				IConfigurationElement configurationElement = configurationElements[counter];
				
				if ("factory".equals(configurationElement.getName())) { //$NON-NLS-1$

					String factoryID = configurationElement.getAttribute("id"); //$NON-NLS-1$
					String factoryClass = configurationElement.getAttribute("class"); //$NON-NLS-1$

					if((factoryID == null) || (factoryID.trim().length() == 0)){
						UiPlugin.logError("Unable to create the factory without an ID (id=\"" + factoryID + "\" class=\"" + factoryClass + "\")."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					}
					else if((factoryClass == null) || (factoryClass.trim().length() == 0)){
						UiPlugin.logError("Unable to create the factory without a class (id=\"" + factoryID + "\" class=\"" + factoryClass + "\")."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					}
					else if(!factoryIDToFactoryMap.containsKey(factoryID)) {

						try {
							factoryIDToFactoryMap.put(factoryID, ((IFileProxyFactory)(configurationElement.createExecutableExtension("class"))));	//$NON-NLS-1$
						} 
						catch (CoreException e) {
							UiPlugin.logError("Unable to create the factory instance (id=\"" + factoryID + "\" class=\"" + factoryClass + "\").", e); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3
						}
					} 
					else {
						UiPlugin.logError("Unable to create the factory with a previously registered factory ID (id=\"" + factoryID + "\" class=\"" + factoryClass + "\")."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					}					
				} 
				else if("file".equals(configurationElement.getName())) { //$NON-NLS-1$
					
					String fileExtension = configurationElement.getAttribute("extension"); //$NON-NLS-1$
					String factoryID = configurationElement.getAttribute("factoryID"); //$NON-NLS-1$
					String caseSensitive_str = configurationElement.getAttribute("caseSensitive"); //$NON-NLS-1$
					boolean caseSensitive = (caseSensitive_str == null) ? true : Boolean.valueOf(caseSensitive_str);
					
					if((fileExtension == null) || (fileExtension.trim().length() == 0)) {
						UiPlugin.logError("Unable to create the file without an extension (extension=\"" + fileExtension + "\" factoryID=\"" + factoryID + "\")."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					}
					else if((factoryID == null) || (factoryID.trim().length() == 0)) {
						UiPlugin.logError("Unable to create the file without a factory ID (extension=\"" + fileExtension + "\" factoryID=\"" + factoryID + "\")."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					}
					else{
						if (!caseSensitive) {
							fileExtension = fileExtension.toLowerCase(); // ease a lot the work : we store all in lower case if we ignore the case
						}
						List<String> factoryIDs = fileExtensionToFactoryIDsMap.get(fileExtension);

						if(factoryIDs == null){

							factoryIDs = new ArrayList<String>();
							factoryIDs.add(factoryID);

							fileExtensionToFactoryIDsMap.put(fileExtension, factoryIDs);
							fileExtensionToCase.put(fileExtension, caseSensitive);
						}
						else if(!factoryIDs.contains(factoryID)){
							factoryIDs.add(factoryID);
						}
					}
				}
			}
		}
		
		//Step 2: Convert the factory IDs to factory instances for a file extension:
		fileExtensionToFactoriesMap = new HashMap<String, List<IFileProxyFactory>>();

		Iterator<String> fileExtensionFactoryIDsMapIterator = fileExtensionToFactoryIDsMap.keySet().iterator();
		
		while (fileExtensionFactoryIDsMapIterator.hasNext()) {
			
			String fileExtension = fileExtensionFactoryIDsMapIterator.next();
			fileExtension = toLowercaseIfNotCaseSensitive(fileExtension);
			List<String> factoryIDs = fileExtensionToFactoryIDsMap.get(fileExtension);
			
			List<IFileProxyFactory> proxyFactories = new ArrayList<IFileProxyFactory>(factoryIDs.size());
			Iterator<String> factoryIDsIterator = factoryIDs.iterator();
			
			while (factoryIDsIterator.hasNext()) {
				
				String factoryID = factoryIDsIterator.next();
				
				if(factoryIDToFactoryMap.containsKey(factoryID)){
					proxyFactories.add(factoryIDToFactoryMap.get(factoryID));					
				}
				else{
					UiPlugin.logError("Unable to resolve the factory instance (id=\"" + factoryID + "\")."); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
			
			fileExtensionToFactoriesMap.put(fileExtension, proxyFactories);
		}		
	}
	
	private String toLowercaseIfNotCaseSensitive(String extension) {
		if (extension == null) {
			return extension;
		}
		Boolean caseSensitive = fileExtensionToCase.get(extension.toLowerCase());
		if ((caseSensitive != null) && (!caseSensitive.booleanValue())) {
			return extension.toLowerCase();
		}
		return extension;
	}
	
	/** 
	 * <p>Resolves the list of zero or more {@link IFileProxyFactory}s registered for test assets with the given file extension.</p>
	 *  
	 * <p>Note, the file extension does not contain the dot separator character.</p>
	 * 
	 * @param fileExtension The file extension of the test assets.
	 * @return The list of {@link IFileProxyFactory}s registered for a given file extension.
	 */
	public List<IFileProxyFactory> getFactories(String fileExtension) {

		if(isRegistered(fileExtension)){
			fileExtension = toLowercaseIfNotCaseSensitive(fileExtension);
			return (fileExtensionToFactoriesMap.get(fileExtension));
		} 

		return (Collections.EMPTY_LIST);
	}
	
	/** 
	 * <p>Resolves if there is one or more {@link IFileProxyFactory}s registered for test assets with the given file extension.</p>
	 *  
	 * <p>Note, the file extension does not contain the dot separator character.</p>
	 * 
	 * @param fileExtension The file extension of the test assets.
	 * @return <code>true</code> if there is one or more {@link IFileProxyFactory}s registered for a given file extension, otherwise <code>false</code>.
	 */
	public boolean isRegistered(String fileExtension) {
		
		if((fileExtension != null) && (fileExtension.trim().length() > 0)){
			fileExtension = toLowercaseIfNotCaseSensitive(fileExtension);
			return (fileExtensionToFactoriesMap.containsKey(fileExtension));
		}
		
		return false;
	}

	/** 
	 * <p>Resolves the {@link IFileProxyFactory} registered with the given ID.</p>
	 * 
	 * @param id The ID of the registered file {@link IFileProxyFactory}.
	 * @return The {@link IFileProxyFactory} registered with the given ID, otherwise <code>null</code>.
	 */
	public IFileProxyFactory getFactory(String id) {

		if((id != null) && (id.trim().length() > 0) && (factoryIDToFactoryMap.containsKey(id))) {
			return (factoryIDToFactoryMap.get(id));
		} 

		return null;
	}
	
	/**
	 * check if extension is case sensitive or not (true by default if extension is null or not registered)
	 * @param extension
	 * @return true if extension is null, if extension is not registered or if extension is registered as case sensitive, false else
	 */
	public boolean isCaseSensitive(String extension) {
		if (extension == null) {
			return true;
		}
		Boolean caseSensitive = fileExtensionToCase.get(extension.toLowerCase());
		if (caseSensitive == null) {
			return true;
		}
		return caseSensitive.booleanValue();
	}
}
