/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileProxyManager.java,v 1.31 2008/10/08 10:51:30 bjerome Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.ReferenceRegistry;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.ReferenceTypeRegistry;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;

/**
 * FileProxyManager.java
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite 
 * @version October 8, 2008
 * @since   February 1, 2005
 */
public class FileProxyManager implements IFileProxyManager {
    
    /**
     * Interface used as value of the field 'parent' of file proxy. The parent of file proxy nodes should be accessed through the getParent() method.
     * @author jgout
     * @since 4.2
     */
    public interface IUnboundedParent {}
    
    //- Map: proxy -> parent (Object)
    private Map parents;
    //- Map: proxy -> parent, (provisional only to store the parent between both calls of getProxy and cacheProxy)
    private Map parentsToCache;
    private FileProxyNodeCache fileProxyNodeCache;
    
	public FileProxyManager() {
        parents = Collections.synchronizedMap(new HashMap());
        parentsToCache = Collections.synchronizedMap(new HashMap());
        fileProxyNodeCache = FileProxyNodeCache.getInstance();
	}
		
	/** 
     * Returns the proxy node corresponding to the given file.
	 * The returned proxy is not automatically cached. 
     * If the returned proxy node needs to be kept a call to cacheProxy() has to be done
     * in order to perform a caching of this proxy node.
	 * @param file the file to convert
	 * @param parent the parent of the proxy node in case of creation. This parameter may be
	 * <code>null</code>. In this case, the proxy node is not localized in the test navigator tree. 
     * A call to cacheProxy will do nothing. 
	 * @see FileProxyManager#cacheProxy(IFile, IProxyNode)
	 */	
    public IProxyNode getProxy(IFile file, Object parent) {
        IProxyNode proxy = fileProxyNodeCache.getProxy(file);
        if(parent != null) {
            parentsToCache.put(proxy, parent);
        }
        return proxy;
    }
        
	/**
	 * Put the specified proxy in the cache so further calls to getProxy(IFile, Object) return
	 * in an efficient way the same cached proxy. To preform properly the proxy node caching, 
     * it is necessary that the given proxy node was obtained by a call to whether getProxy() or updateProxy().  
	 * @param file a file
	 * @param newProxy the proxy associated to the given file. 
     * This proxy node should be the returned proxy node of a call to getProxy. 
	 */
	public void cacheProxy(IFile file, IProxyNode newProxy) {
        if(newProxy != null) {
            Object parent = parentsToCache.remove(newProxy);
            if(parent != null) {
                parents.put(newProxy, parent);
            }
        }
    }


	/** 
     * Returns whether the given file is already in the local proxy data base. 
	 * @param file
	 * @return true if the given file exists as proxy and false otherwise
	 */
	public boolean isaCachedFile(IFile file) {
		return fileProxyNodeCache.isCached(file);
	}

	/** 
     * Computes a new proxy using registered file proxy factories.
	 * This method updates the cached proxy for the given file 
     * If the returned proxy node needs to be kept, a call to cacheProxy() has to be done
     * in order to perform a caching of this proxy node.
	 * @param file the file from which the proxy is derived.
	 * @param parent the parent node to pass to the factory.
	 * @return the proxy alias associated to the given file.
	 */
	public IProxyNode updateProxy(IFile file, Object parent) {
		cleanUpOutdatedReferences(file);
		IProxyNode proxy = fileProxyNodeCache.createProxy(file);
        if(parent != null) {
            parentsToCache.put(proxy, parent);
        }
		return proxy;
	}
	
	// static final declaration to speed up process, keep them because often used
	private static final ReferenceTypeRegistry factory = ReferenceTypeRegistry.getInstance();
	private static final ReferenceRegistry registry =  ReferenceRegistry.getInstance();
	
	private void cleanUpOutdatedReferences(IFile file) {		
		Set allType = registry.getReferenceTypes(file);
		if ((allType == null) || (allType.isEmpty())) {
			return; // no reference => no need to continue
		}
		List toBeDeleted = new ArrayList();
		for (Iterator it = allType.iterator(); it.hasNext(); ) {
			String refType = (String)it.next();
			String oppRef = factory.getOppositeReferenceType(refType);
			if (factory.isExplicit(refType)) {			
				// if reference is explicit and opposite one is implicit, kill opposite reference (no update on target)
				// if reference is explicit and opposite one is explicit, a listener should be there to update target, so no need to delete ref
				// if reference is implicit, do not touch it else it won't be rebuild
				Collection refs = registry.getReferences(file, refType);			
				for (Iterator it2 = refs.iterator(); it2.hasNext();) {
					IProxyNode target = (IProxyNode) it2.next();
					IFile targetfile = (IFile)target.getUnderlyingResource();
					if (!factory.isExplicit(oppRef)) {
						registry.removeReference(targetfile, file);
					}
					toBeDeleted.add(targetfile); 
				}
			}
		}
		// now, clean up all explicit references (could not do it before to avoid concurrent access exception
		for (Iterator it = toBeDeleted.iterator(); it.hasNext(); ) {
			IFile targetFile = (IFile)(it.next());
			registry.removeReference(file, targetFile);
		}
		toBeDeleted.clear();
	}
	
	/** Returns the proxy which has the given uid. This search starts from the given proxy node and cross its complete sub tree of proxies.
	 * 
	 * @param proxy root of the proxy tree where the search is done. 
	 * @param uid the uid of the searched proxy.
	 * @return a proxy node which has the given uid or <code>null</code> if there is no such proxy found in the complete given tree of proxy.
	 */
	public IProxyNode findProxyByID(IProxyNode proxy, String uid) {
		return fileProxyNodeCache.findProxyByID(proxy, uid);
	}

	/** 
     * Removes the proxy entry associated to the given file.
	 * @param file
	 */
	public IProxyNode uncacheProxy(IFile file) {
        IProxyNode removedProxy = fileProxyNodeCache.remove(file);
        //- since this proxy will no longer exist remove its parent as well.
        parents.remove(removedProxy);
        parentsToCache.remove(removedProxy);
        return removedProxy;
	}
		
	/** 
     * Retrieves the node associated to the given object. 
	 * This parameter is adapted to a IProxy in order to retrieve the node
	 * representing it.
	 * 
	 * @param object An object adaptable to IProxy.
	 * @return the proxy associated to the given object or <code>null</code> if such proxy node does not exist.
	 */
	public IProxyNode getCorrespondingProxy(Object object) {
		return fileProxyNodeCache.getCorrespondingProxy(object);
	}
	
	/** 
     * Retrieves the node using its underlying resource and its identifier. 
	 * @param fileName name of the underlying resource of the searched proxy
	 * @param identifier identifier of the searched proxy
	 * @return a proxy node or <code>null</code> if not found.
	 * @deprecated use {@link #getCorrespondingProxy(new Proxy(fileName, String))} instead.
	 */
	public IProxyNode getCorrespondingProxy(String fileName, String identifier) {
		return getCorrespondingProxy(new Proxy(fileName, identifier));
	}
    
    /**
     * Returns the parent of the given proxy node.
     * @param proxy a proxy node
     * @return the parent of the given proxy node or <code>null</code> if this proxy node has not been cached or if the given proxy is null.
     * @since 4.2
     */
    public Object getParent(IProxyNode proxy) {
    	if (proxy != null) {
			Object parent = proxy.getParent();
			if (parent != null && parent instanceof IUnboundedParent) {
				return parents.get(proxy);
			}
			return parent;
		}
    	return null;
    }

    public void dispose() {
    	if (fileProxyNodeCache != null) {
    		fileProxyNodeCache.dispose();
    		fileProxyNodeCache = null;
    	}
    }
	
}