/*******************************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileProxyMarkerPersister.java,v 1.14 2010/03/30 15:48:03 bjerome Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.io.StringReader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.WorkspaceJob;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.ILock;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.MultiRule;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.ui.XMLMemento;

/**
 * <p>Implementation of the {@link IFileProxyPersister} interface for saving {@link IPersistableProxyNode}s 
 * to markers and loading markers into {@link IProxyNode}s.</p>
 * 
 * <p>If the serialized proxy is larger than the marker size limit (65535 bytes), the {@link FileProxyMetadataPersister}
 * is used to save/load the proxy to/from a {@link XMLMemento}.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Julien Canches
 * @author  Paul Slauenwhite
 * @version March 30, 2010
 * @since   September 28, 2006
 * @see     IFileProxyPersister
 * @see     IMarker
 * @see     FileProxyMetadataPersister
 */
public class FileProxyMarkerPersister implements IFileProxyPersister {

	private FileProxyMetadataPersister fileProxyMetadataPersister = new FileProxyMetadataPersister();

	private static MarkerJob markerJob = null;

	private static final String MARKER_PROXYSTATE = "org.eclipse.hyades.test.ui.proxyStateMarker"; //$NON-NLS-1$

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.navigator.proxy.IFileProxyPersister#loadProxy(org.eclipse.core.resources.IFile)
	 */
	public IProxyNode loadProxy(IFile file) {

		try {

			//Step 1: Check the marker job for queued markers:
			if(markerJob != null){

				MarkerRecord markerRecord = markerJob.getScheduledMarker(file);

				if((markerRecord != null) && (markerRecord.getProxyState() != null)){
					return (FileProxyMetadataPersister.deserializeProxyNode(file, markerRecord.getProxyFactoryId(), XMLMemento.createReadRoot(new StringReader(markerRecord.getProxyState()))));
				}
			}

			//Step 2: Check the markers for the file:
			IMarker[] markers = file.findMarkers(MARKER_PROXYSTATE, false, IResource.DEPTH_ZERO);

			if(markers.length > 0) {

				//Note: Default to the first marker in the array.

				//Check if the file has not changed since the last save:
				if(file.getModificationStamp() == Long.parseLong(markers[0].getAttribute(FileProxyMetadataPersister.TAG_LAST_SAVE_STAMP, "0"))){ //$NON-NLS-1$

					String proxyFactoryId = markers[0].getAttribute(FileProxyMetadataPersister.TAG_FACTORY_ID, null);

					if (FileProxyNodeCache.NULL_PROXY.getFactoryID().equals(proxyFactoryId)){
						return (FileProxyNodeCache.NULL_PROXY);
					}

					String proxyState = markers[0].getAttribute(FileProxyMetadataPersister.TAG_PROXY_STATE, null);

					if (proxyState != null) {
						return (FileProxyMetadataPersister.deserializeProxyNode(file, proxyFactoryId, XMLMemento.createReadRoot(new StringReader(proxyState))));
					}

					//Fall back to the FileProxyMetadataPersister:
					return (fileProxyMetadataPersister.loadProxy(file));
				}
			} 
		} 
		catch (Exception e) {

			UiPlugin.logError(e);

			//Remove existing markers:
			try {
				file.deleteMarkers(MARKER_PROXYSTATE, false, IResource.DEPTH_ZERO);
			} 
			catch (Exception ee) {
				//Ignore and return null.
			}            
		}   

		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.navigator.proxy.IFileProxyPersister#saveProxy(org.eclipse.core.resources.IFile, org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode)
	 */
	public void saveProxy(IFile file, IPersistableProxyNode proxyNode) throws Exception{

		//Resolve the proxy state:			
		XMLMemento xmlMemento = XMLMemento.createWriteRoot(FileProxyMetadataPersister.PROXY_STATE_MEMENTO_TYPE);

		proxyNode.saveState(xmlMemento);

		StringWriter proxyStateWriter = new StringWriter();

		xmlMemento.save(proxyStateWriter);

		String proxyState = proxyStateWriter.toString();

		//Determine if the serialized proxy is within the marker size limit (65535 bytes):
		boolean isValidProxySize = true;

		try {
			isValidProxySize = (proxyState.getBytes("UTF-8").length <= 65535); //$NON-NLS-1$
		} 
		catch (UnsupportedEncodingException u) {
			isValidProxySize = false;
		}

		//Schedule the marker job:
		if (markerJob != null) {

			//If the current marker job is not back to the Job.NONE state, create a new one:
			if ((markerJob.getState() != Job.RUNNING) && (markerJob.cancel())) {

				//If the current marker job has changed form the Job.NONE state (for example, now running), create a new one:
				if (markerJob.scheduleMarker(file, proxyNode.getFactoryID(), (isValidProxySize ? proxyState : null))) {
					markerJob.schedule();
				} 
				else {
					markerJob = null;
				}
			} 
			else {
				markerJob = null;
			}        	
		}

		if (markerJob == null) {

			markerJob = new MarkerJob();
			markerJob.scheduleMarker(file, proxyNode.getFactoryID(), (isValidProxySize ? proxyState : null)); 
			markerJob.schedule();    		
		}       

		//Fall back to the FileProxyMetadataPersister:
		if (!isValidProxySize) {
			fileProxyMetadataPersister.saveProxy(file, proxyNode);
		}
	}

	/**
	 * <p>A job for creating/updating markers in a single workspace operation.</p>
	 * 
	 * <p>Additional marker create/update operations may be scheduled when the job 
	 * is in the {@link Job#NONE} (not scheduled) state.</p>
	 * 
	 * 
	 * @author  Jerome Gout
	 * @author  Jerome Bozier
	 * @author  Julien Canches
	 * @author  Paul Slauenwhite
	 * @version March 19, 2010
	 * @since   September 28, 2006
	 * @see     WorkspaceJob
	 */
	private static class MarkerJob extends WorkspaceJob {

		private List<MarkerRecord> scheduledMarkers = new ArrayList<MarkerRecord>();

		/**
	     * Reentrant, deadlock detecting, and deadlock recovering lock to guard against multiple 
	     * threads modifying/reading the list of scheduled markers.
	     */
	    private final static ILock LOCK = Job.getJobManager().newLock();
	    
		public MarkerJob() {

			super("MarkerJob"); //$NON-NLS-1$

			setPriority(Job.SHORT);
			setSystem(true);
		}

		public MarkerRecord getScheduledMarker(IFile file) {
			
			try {

				//Acquire a lock to guard against multiple threads modifying/reading the list of scheduled markers:
				LOCK.acquire();

				Iterator<MarkerRecord> scheduledMarkersIterator = scheduledMarkers.iterator();

				while (scheduledMarkersIterator.hasNext()) {

					MarkerRecord markerRecord = scheduledMarkersIterator.next();
					
					if(markerRecord.getFile().equals(file)){
						return markerRecord;
					}
				}
			} 
			finally {

				//Release the lock:
				LOCK.release();
			}
			
			return null;
		}

		/**
		 * <p>Schedule a marker create/update operation when the job is in the {@link Job#NONE} (not scheduled) state.</p>
		 * 
		 * @param file The file to create the marker. 
		 * @param proxyFactoryId The ID of the proxy factory to create the proxy.
		 * @param proxyState The state of the proxy.
		 * @return <code>true</code> if the marker create/update operation was scheduled, otherwise <code>false</code>.
		 */
		public boolean scheduleMarker(IFile file, String proxyFactoryId, String proxyState) {

			if (getState() == NONE) {

				MarkerRecord markerRecord = new MarkerRecord(file, proxyFactoryId, proxyState);
				ISchedulingRule newRule = file;
				ISchedulingRule currentRule = getRule();

				if (currentRule instanceof MultiRule) {

					ISchedulingRule[] currentChildRules = ((MultiRule)(currentRule)).getChildren();
					ISchedulingRule[] newChildRules = new ISchedulingRule[currentChildRules.length + 1];

					System.arraycopy(currentChildRules, 0, newChildRules, 0, currentChildRules.length);

					newChildRules[currentChildRules.length] = file;

					newRule = new MultiRule(newChildRules);
				} 
				else if (currentRule != null){
					newRule = new MultiRule(new ISchedulingRule[]{currentRule, file});
				}

				try{
					setRule(newRule);
				} 
				catch (IllegalArgumentException e) {
					scheduledMarkers.remove(markerRecord);
				}

				scheduledMarkers.add(markerRecord);

				return true;
			}

			return false;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.core.resources.WorkspaceJob#runInWorkspace(org.eclipse.core.runtime.IProgressMonitor)
		 */
		public IStatus runInWorkspace(IProgressMonitor monitor) {

			try {

				//Acquire a lock to guard against multiple threads modifying/reading the list of scheduled markers:
				LOCK.acquire();

				//Note: The progress monitor's cancel request is not honored since we use the 
				//cancellation mechanism to only cancel jobs that have not started running since 
				//we must complete all running jobs.
				monitor.beginTask("", scheduledMarkers.size()); //$NON-NLS-1$

				try {

					MultiStatus multiStatus = new MultiStatus(UiPlugin.getID(), 0,  "Marker job multi-status", null); //$NON-NLS-1$
					Iterator<MarkerRecord> scheduledMarkersIterator = scheduledMarkers.iterator();

					while (scheduledMarkersIterator.hasNext()) {

						MarkerRecord markerRecord = scheduledMarkersIterator.next();

						if (markerRecord.getFile().isAccessible()) { 

							IStatus status = markerRecord.serialize();

							if (status.getSeverity() != IStatus.OK) {
								multiStatus.add(status);
							}
						}

						monitor.worked(1);
					}

					return multiStatus;
				} 
				finally {

					scheduledMarkers.clear();

					monitor.done();
				}
			} 
			finally {

				//Release the lock:
				LOCK.release();
			}
		}
	}

	/**
	 * <p>A marker record.</p>
	 * 
	 * 
	 * @author  Jerome Gout
	 * @author  Jerome Bozier
	 * @author  Paul Slauenwhite
	 * @version March 19, 2010
	 * @since   September 28, 2006
	 * @see     IMarker
	 */
	private static class MarkerRecord {

		private IFile file = null;
		private String proxyFactoryId = null;
		private String proxyState = null;
		private long lastFileModificationStamp = -1;

		/**
		 * </p>Constructor.</p>
		 * 
		 * @param file The file to create the marker. 
		 * @param proxyFactoryId The ID of the proxy factory to create the proxy.
		 * @param proxyState The state of the proxy.
		 */
		public MarkerRecord(IFile file, String proxyFactoryId, String proxyState) {

			this.file = file;
			this.proxyFactoryId = proxyFactoryId;
			this.proxyState = proxyState;
			this.lastFileModificationStamp = file.getModificationStamp();
		}

		public IStatus serialize() {

			try {

				//Remove existing markers:
				file.deleteMarkers(MARKER_PROXYSTATE, false, IResource.DEPTH_ZERO);

				//Create the new marker:
				IMarker marker = file.createMarker(MARKER_PROXYSTATE);
				marker.setAttribute(FileProxyMetadataPersister.TAG_LAST_SAVE_STAMP, String.valueOf(lastFileModificationStamp));
				marker.setAttribute(FileProxyMetadataPersister.TAG_FACTORY_ID, proxyFactoryId);

				if (proxyState != null) {
					marker.setAttribute(FileProxyMetadataPersister.TAG_PROXY_STATE, proxyState);
				}

				return (Status.OK_STATUS);
			} 
			catch (Throwable e) {

				//Log error and return error status:
				UiPlugin.logError(e);

				return (new Status(IStatus.ERROR, UiPlugin.getID(), "Unable to create/update a marker for file '" + file.getName() + "'", e)); //$NON-NLS-1$ //$NON-NLS-2$
			}   
		}

		public IFile getFile() {
			return file;
		}

		public String getProxyFactoryId() {
			return proxyFactoryId;
		}

		public String getProxyState() {
			return proxyState;
		}
	}
}
