/*******************************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileProxyMetadataPersister.java,v 1.9 2010/05/27 20:00:05 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.navigator.IFileProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IPersistableFileProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode;
import org.eclipse.hyades.test.ui.navigator.IPersistableTypedElementProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITypedElementProxyFactory;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.XMLMemento;

/**
 * <p>Implementation of the {@link IFileProxyPersister} interface for saving {@link IPersistableProxyNode}s 
 * to mementos and loading mementos into {@link IProxyNode}s.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Julien Canches
 * @author  Paul Slauenwhite
 * @version May 27, 2010
 * @since   September 28, 2006
 * @see     IFileProxyPersister
 * @see     IMemento
 */
public class FileProxyMetadataPersister implements IFileProxyPersister {

	private IPath storageArea = null;

	public static final String TAG_FACTORY_ID = "factoryID"; //$NON-NLS-1$
	public static final String TAG_PROXY_STATE = "proxyState"; //$NON-NLS-1$
	public static final String TAG_LAST_SAVE_STAMP = "lastSaveStamp"; //$NON-NLS-1$
	public static final String PROXY_STATE_MEMENTO_TYPE = "proxyStateMemento"; //$NON-NLS-1$
	
	public FileProxyMetadataPersister(){
		storageArea = Platform.getStateLocation(Platform.getBundle(UiPlugin.getID()));
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.navigator.proxy.IFileProxyPersister#loadProxy(org.eclipse.core.resources.IFile)
	 */
	public IProxyNode loadProxy(IFile file) {
		
		Reader reader = null;
		
		try{
			
			File mementoFile = new File(storageArea.append(file.getFullPath()).toOSString());

			if (mementoFile.canRead()) {

				reader = new InputStreamReader(new GZIPInputStream(new BufferedInputStream(new FileInputStream(mementoFile))));
				
				XMLMemento memento = XMLMemento.createReadRoot(reader);
				String lastSaveTimeStamp = memento.getString(TAG_LAST_SAVE_STAMP);

				//Check if the file has not changed since the last save:
				if(file.getModificationStamp() == (lastSaveTimeStamp != null ? Long.parseLong(lastSaveTimeStamp) : 0)){

					String proxyFactoryId = memento.getString(TAG_FACTORY_ID);
					
					if (FileProxyNodeCache.NULL_PROXY.getFactoryID().equals(proxyFactoryId)){
						return FileProxyNodeCache.NULL_PROXY;
					}
					
					IMemento proxyState = memento.getChild(TAG_PROXY_STATE);
					
					if (proxyState != null) {
						return (deserializeProxyNode(file, proxyFactoryId, proxyState));
					}
				}
			}
		} 
		catch (Exception e) {

			//Log error and return null:
			UiPlugin.logError(e);       
		} 
		finally{
			
			if(reader != null){
				
				try {
					reader.close();
				} 
				catch (IOException i) {
					//Ignore since reader cannot be closed.
				}
			}
		}

		return null;
	}

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.internal.navigator.proxy.IFileProxyPersister#saveProxy(org.eclipse.core.resources.IFile, org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode)
     */
    public void saveProxy(IFile file, IPersistableProxyNode proxyNode) throws Exception{
		
    	Writer writer = null;
    	
    	try{
    		
    		File mementoFile = new File(storageArea.append(file.getFullPath()).toOSString());
    		mementoFile.getParentFile().mkdirs();

    		writer = new OutputStreamWriter(new GZIPOutputStream(new BufferedOutputStream(new FileOutputStream(mementoFile))));

    		XMLMemento xmlMemento = XMLMemento.createWriteRoot(PROXY_STATE_MEMENTO_TYPE);
    		xmlMemento.putString(TAG_LAST_SAVE_STAMP, String.valueOf(file.getModificationStamp()));
    		xmlMemento.putString(TAG_FACTORY_ID, proxyNode.getFactoryID());
    		
    		proxyNode.saveState(xmlMemento.createChild(TAG_PROXY_STATE));
    		
    		xmlMemento.save(writer);
    	} 
    	finally {

    		if(writer != null){

    			try {
    				writer.close();
    			} 
    			catch (IOException i) {
    				//Ignore since writer cannot be closed.
    			}
    		}
    	}
	}
    
    public static IProxyNode deserializeProxyNode(IFile file, String proxyFactoryId, IMemento memento) {

		try{

			//Update memento with the file path used to build the URI of the EObjectProxyNode:
			memento.putString(TestUIConstants.TAG_URI_ROOT, file.getFullPath().toString());

			//Resolve the file proxy factory:
			IFileProxyFactory fileProxyFactory = TestNavigator.getFileFactoryManager().getFactory(proxyFactoryId);

			if(fileProxyFactory != null) {

				if(fileProxyFactory instanceof IPersistableFileProxyFactory) {

					IProxyNode proxyNode = ((IPersistableFileProxyFactory)(fileProxyFactory)).recreate(memento, file, FileProxyNodeCache.UNBOUNDED_PARENT);

					Assert.isLegal(file.equals(proxyNode.getAdapter(IFile.class)), "Invalid proxy '" + proxyNode.getClass().getName() + "' returned by factory '" + fileProxyFactory.getClass().getName() + "' for file '" + file.getFullPath().toString() + "'."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$

					return proxyNode;
				} 
				else {

					//Log error and return null:
					UiPlugin.logError("The factory id '" + proxyFactoryId + "' is not an instance of IPersistableFileProxyFactory");  //$NON-NLS-1$ //$NON-NLS-2$
				}
			} 
			else {

				ITypedElementProxyFactory typedElementProxyFactory = TypedElementFactoryManager.getInstance().getFactoryFromID(proxyFactoryId);

				if(typedElementProxyFactory != null) {

					if(typedElementProxyFactory instanceof IPersistableTypedElementProxyFactory) {						
						return ((IPersistableTypedElementProxyFactory)(typedElementProxyFactory)).recreate(memento, FileProxyNodeCache.UNBOUNDED_PARENT);
					} 
					else {

						//Log error and return null:
						UiPlugin.logError("The factory id '" + proxyFactoryId + "' is not an instance of IPersistableTypedElementProxyFactory");  //$NON-NLS-1$ //$NON-NLS-2$
					}
				}
			}
		}
		catch (Exception e) {
			// if exception occur, we will return null and proxy will be recreated instead of beeing loaded
		}

		return null;
	}
}
