/********************************************************************** 
 * Copyright (c) 2006, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id$ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.jobs.ILock;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigatorMessages;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.ReferenceRegistry;
import org.eclipse.hyades.test.ui.internal.util.TestUIUtilities;
import org.eclipse.hyades.test.ui.navigator.FileProxyNode;
import org.eclipse.hyades.test.ui.navigator.IFileProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxy;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;

/**
 * <p>File {@link IProxyNode} cache manager.</p>
 * 
 * <p>This class is a singleton and is in charge of storing all file 
 * {@link IProxyNode}s for all instances of {@link FileProxyManager}.</p>
 * 
 * <p>Note: In the event that a method returns <code>null</code> or 
 * {@link ErrorProxy} (see {@link #createProxy(IFile)} and 
 * {@link #getProxy(IFile)}), a proxy for the {@link IFile} will not be 
 * cached so the proxy can be recreated (e.g. refresh).</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version April 12, 2010
 * @since   January 5, 2006
 */
public class FileProxyNodeCache {

	/**
	 * <p>Null proxy associated to files without a proxy.</p>
	 * 
	 * @see NullProxy
	 */
	protected final static NullProxy NULL_PROXY = new NullProxy();

	/**
	 * <p>Unbounded parent representing a parent to a file proxy.</p>
	 * 
	 * @see UnboundedParent
	 */
	protected final static FileProxyManager.IUnboundedParent UNBOUNDED_PARENT = new UnboundedParent();

	/**
	 * {@link Map} of {@link IFile}s to {@link IProxyNode}s.
	 */
	private Map<IFile, IProxyNode> fileProxyNodes = null;

	/**
	 * The resource change listener.
	 */
	private ResourceChangeListener resourceChangeListener = null;

	/**
	 * The default file proxy persister.
	 */
	private IFileProxyPersister defaultPersister = null;

	/**
	 * Set of {@link IResourceChangeListener}s.
	 */
	private Set<IResourceChangeListener> resourceListeners = null; 
	
	/**
	 * Singleton instance of this class.
	 */
	private static FileProxyNodeCache instance = null;

	/**
	 * Number of references to the singleton instance of this class.
	 */
	private static int referencesCount = 0;
	
	/**
	 * Reentrant, deadlock detecting, and deadlock recovering lock to 
	 * guard against multiple threads creating/cache the same proxy more than once.
	 */
	private final static ILock LOCK = Job.getJobManager().newLock();

	/**
	 * Empty {@link IProxyNode} array.
	 */
	private final static IProxyNode[] NO_CHILDREN = new IProxyNode[0];
	
	/**
	 * Privatized constructor.
	 */
	private FileProxyNodeCache() {

		fileProxyNodes = new HashMap<IFile, IProxyNode>(); 
		resourceChangeListener = new ResourceChangeListener();
		defaultPersister = new FileProxyMarkerPersister();
		resourceListeners = Collections.synchronizedSet(new HashSet<IResourceChangeListener>());
	}

	/**
	 * <p>Resolves the singleton instance of this class.</p>
	 * 
	 * <p>For each reference to the singleton instance of this class
	 * that is resolved by invoking this method, a invocation to the 
	 * {@link #dispose()} method is required before the singleton 
	 * instance of this class is disposed.
	 * 
	 * @return The singleton instance of this class. 
	 */
	public static FileProxyNodeCache getInstance() {

		if(instance == null) {
			instance = new FileProxyNodeCache();
		}

		referencesCount++;

		return instance;
	}

	/**
	 * <p>Clears all of the {@link IProxyNode}(s) from the cache.</p>
	 * 
	 * <p><b>Note:</b> Although invoking this method is efficient, 
	 * opening the Test Navigator will cause all {@link IProxyNode}(s)
	 * to be rebuilt, which <i>may</i> be inefficient.</p>
	 */
	public void clearCache() {
		fileProxyNodes.clear();
	}
	
	/**
	 * Disposes the singleton instance of this class when the number 
	 * of references to the singleton instance of this class is 1.
	 */
	public void dispose() {
		
		if (--referencesCount == 0) {
		
			fileProxyNodes.clear();
			resourceChangeListener.dispose();
			defaultPersister = null;
			resourceListeners.clear();
			instance = null;
		}
	}
	
	/**
	 * <p>Registers a resource change listener with the cache.</p>
	 * 
	 * <p>Components that manage {@link IProxyNode}s should register a resource
	 * change listener to handle properly resource changes when notified of 
	 * resource changes.</p>
	 * 
	 * <p>Note: If a resource change listener is registered, do NOT
	 * register a resource change listener with the workspace resource services 
	 * (see {@link IWorkspace#addResourceChangeListener(IResourceChangeListener)}).</p>
	 * 
	 * @param listener The resource change listener to register.
	 */
	public void addResourceListener(IResourceChangeListener listener) {
		resourceListeners.add(listener);
	}

	/**
	 * <p>Unregisters a resource change listener from the cache.</p>
	 * 
	 * @param listener The resource change listener to unregister.
	 */
	public void removeResourceListener(IResourceChangeListener listener) {
		resourceListeners.remove(listener);
	}
	
	/**
	 * <p>Creates a new {@link IProxyNode} for the {@link IFile} using a proxy a registered 
	 * {@link IProxyNode} factory for the file extension (see the 
	 * <code>org.eclipse.hyades.test.ui.testNavigatorFileProxyFactory</code> 
	 * extension point).</p>
	 * 
	 * <p>If the {@link IFile} is <code>null</code> or does not exist, <code>null</code> is 
	 * returned.</p>
	 * 
	 * <p>If an existing proxy is cached, it is first removed from the cache.</p>
	 * 
	 * <p>If a {@link IProxyNode} for the file using a proxy could not be created due to no 
	 * registered {@link IProxyNode} factories for the file extension, <code>null</code>
	 * is returned.  If a {@link IProxyNode} for the file using a proxy could not be created 
	 * due to all of the registered {@link IProxyNode} factories for the file extension throwing 
	 * errors, a {@link ErrorProxy} is returned.</p>
	 * 
	 * <p>Note: In the event that this method returns <code>null</code> or {@link ErrorProxy}, 
	 * a proxy for the {@link IFile} will not be cached so the proxy can be recreated 
	 * (e.g. refresh).</p>
	 * 
	 * @param file The {@link IFile} used to create the new {@link IProxyNode}.  
	 * @return The new {@link IProxyNode} for the file using a proxy a registered {@link IProxyNode} factory for the file extension, otherwise <code>null</code> or {@link ErrorProxy}. 
	 */
	public IProxyNode createProxy(IFile file) {

		if((file != null) && (file.exists())){
			
			IProxyNode proxyNode = null;
			
			try {
	
				//Acquire a lock to guard against multiple threads creating/cache the same proxy more than once:
				LOCK.acquire();
	
				fileProxyNodes.remove(file);
	
				proxyNode = internalCreateProxy(file);				
			}
			catch (Throwable t) {

				//Log the exception:
				UiPlugin.logError("Unable to create proxy for file '" + file.getFullPath().toString() + "'.", t); //$NON-NLS-1$ //$NON-NLS-2$
				
				//Create an error proxy:
				proxyNode = new ErrorProxy(file);
			}
			finally {
	
				//Release the lock:
				LOCK.release();
			}
			
			if (proxyNode != NULL_PROXY){
				return proxyNode;
			}    		
		}
		
		return null;
	}
	
	/**
	 * <p>Resolves the {@link IProxyNode} for the {@link IFile} from the cache.</p>
	 * 
	 * <p>If the {@link IFile} is <code>null</code>, <code>null</code> is returned.  A  
	 * {@link IProxyNode} can be resolved without the file existing.</p>
	 * 
	 * <p>If a {@link IProxyNode} for the {@link IFile} does not exist in the cache, 
	 * it is loaded from disk (e.g. marker file).  If a {@link IProxyNode} for the {@link IFile}
	 * cannot be loaded from disk (e.g. marker file), it is created for the {@link IFile} 
	 * using a registered {@link IProxyNode} factory for the file extension (see the 
	 * <code>org.eclipse.hyades.test.ui.testNavigatorFileProxyFactory</code> 
	 * extension point).</p>
	 *    
	 * <p>If a {@link IProxyNode} for the file using a proxy could not be created due to no 
	 * registered {@link IProxyNode} factories for the file extension, <code>null</code>
	 * is returned.  If a {@link IProxyNode} for the file using a proxy could not be created 
	 * due to all of the registered {@link IProxyNode} factories for the file extension throwing 
	 * errors, a {@link ErrorProxy} is returned.</p>
	 * 
	 * <p>Note: In the event that this method returns <code>null</code> or {@link ErrorProxy}, 
	 * a proxy for the {@link IFile} will not be cached so the proxy can be recreated 
	 * (e.g. refresh).</p>
	 * 
	 * @param file The {@link IFile} used to resolve the {@link IProxyNode}.  
	 * @return The {@link IProxyNode} resolved from the cache, disk, or created, otherwise <code>null</code> or {@link ErrorProxy}. 
	 */
	public IProxyNode getProxy(IFile file) {
		
		//Resolve the proxy only if the file is not null:
		if(file != null){

			//Resolve the proxy from the cache if the file is in the cache:
			//Note: A proxy may exist in the cache after a file has been deleted and does not exist.
			IProxyNode proxyNode = fileProxyNodes.get(file);
			
			//Resolve, load, or create the proxy only if the file exists:
			if ((proxyNode == null) && (file.exists())){
			
				try {
	
					//Acquire a lock to guard against multiple threads creating/cache the same proxy more than once:
					LOCK.acquire();
	
					//Resolve the proxy from the cache if the file is in the cache since it may have been added while acquiring the lock:
 					proxyNode = fileProxyNodes.get(file);
					
 					//Load or create the proxy:
 					if (proxyNode == null) {
					
 						//Load the proxy from disk (e.g. marker file):
						proxyNode = defaultPersister.loadProxy(file);
						
						//If the proxy exists, cache the proxy:
						if (proxyNode != null) {
							fileProxyNodes.put(file, proxyNode);
						} else {
							//Create the proxy using a registered {@link IProxyNode} factory for the file extension:
							proxyNode = internalCreateProxy(file);
						}
					}
				} 
				catch (Throwable t) {
					
					//Log the exception:
					UiPlugin.logError("Unable to create proxy for file '" + file.getFullPath().toString() + "'.", t); //$NON-NLS-1$ //$NON-NLS-2$

					//Create an error proxy:
					proxyNode = new ErrorProxy(file);
				} 
				finally {
	
					//Release the lock:
					LOCK.release();
				}
			}
	
			if (proxyNode != NULL_PROXY){
				return proxyNode;
			}    		
		}
		
		return null;
	}

	/**
	 * <p>Resolves the {@link IProxyNode} for the {@link Object} from the cache.</p>
	 * 
	 * <p>The {@link Object} is first adapted to a {@link IProxy}, the {@link IProxyNode} is resolved
	 * for the {@link IFile} associated with the {@link IProxy} from the cache (see 
	 * {@link #getProxy(IFile)}), and the {@link IProxyNode} with the same ID as the {@link IProxyNode} 
	 * or one its children is resolved (see {@link #findProxyByID(IProxyNode, String)}).</p>
	 *  
	 * @param object The {@link Object} used to resolve the {@link IProxyNode}.  
	 * @return The {@link IProxyNode} resolved from the cache, disk, or created, otherwise <code>null</code>. 
	 * @see #getProxy(IFile)
	 * @see #findProxyByID(IProxyNode, String)
	 */
	public IProxyNode getCorrespondingProxy(Object object) {

		if(object != null){

			IProxy proxy = null;
			
			if (object instanceof IProxy) {
				proxy = ((IProxy)(object));
			} 
			else if (object instanceof IAdaptable) {
				proxy = ((IProxy)(((IAdaptable)(object)).getAdapter(IProxy.class)));
			} 
			else {
				proxy = ((IProxy)(Platform.getAdapterManager().getAdapter(object, IProxy.class)));
			}

			if (proxy != null) {
				
				IResource resource = proxy.getUnderlyingResource();
				
				if (resource instanceof IFile) {
					
					IProxyNode proxyNode = getProxy(((IFile)(resource)));
					
					if (proxyNode != null) {
						return (findProxyByID(proxyNode, proxy.getIdentifier()));
					}
				}
			}
		}
		
		return null;
	}
	
	/**
	 * <p>Resolves the {@link IProxyNode} for the {@link IProxyNode} with the same ID as the {@link IProxyNode} 
	 * or one its decedents.</p>
	 * 
	 * <p>If the ID is <code>null</code>, empty, or whitespace, the {@link IProxyNode} is returned.</p>
	 *  
	 * @param proxyNode The {@link IProxyNode} used to resolve the {@link IProxyNode} with the same ID as the {@link IProxyNode} or one its decedents.
	 * @param id The ID of the {@link IProxyNode}.
	 * @return The {@link IProxyNode} or one its descendants with the same ID, otherwise <code>null</code>. 
	 */
	public IProxyNode findProxyByID(IProxyNode proxyNode, String id) {
		
		if(proxyNode != null){
			
			if((id != null) && (id.trim().length() > 0)){
	
				if(proxyNode.getIdentifier().equals(id)){
					return proxyNode;
				}
				else {
					
					//Search the children:
					IProxyNode[] children = proxyNode.getChildren();
					
					for (int counter = 0; counter < children.length; counter++) {
					
						proxyNode = findProxyByID(children[counter], id);
						
						if(proxyNode != null) {
							return proxyNode;
						}
					}
				}
			}
			else{
				return proxyNode;
			}
		}
		
		return null;
	}
	
	/**
	 * Resolves if the a {@link IProxyNode} exists in the cache for the {@link IFile}.
	 * 
	 * @return <code>true</code> if the a {@link IProxyNode} exists in the cache for the {@link IFile}, otherwise <code>false</code>.
	 */
	public boolean isCached(IFile file) {        
		return (getCachedProxy(file) != null);
	}

	/**
	 * Resolves the a {@link IProxyNode} from the cache for the {@link IFile}.
	 * 
	 * @return The {@link IProxyNode} resolved from the cache, otherwise <code>null</code>. 
	 */
	public IProxyNode getCachedProxy(IFile file) {
		
		IProxyNode proxyNode = fileProxyNodes.get(file);

		if (proxyNode != NULL_PROXY){
			return proxyNode;
		}    		

		return null;
	}

	/**
	 * Removes the a {@link IProxyNode} from the cache for the {@link IFile}.
	 * 
	 * @return The {@link IProxyNode} removed from the cache, otherwise <code>null</code>. 
	 */
	public IProxyNode remove(IFile file) {
		
		IProxyNode proxyNode = fileProxyNodes.remove(file);
		
		if(proxyNode != NULL_PROXY){
			return proxyNode;
		}
		
		return null;
	} 

	/**
	 * <p>Creates, validates, saves, caches, and returns a {@link IProxyNode} for the 
	 * file by invoking the registered proxy factory for the file type.</p>
	 * 
	 * <p>If there are no registered proxy factories for the file type, a 
	 * {@link NullProxy} is saved, cached, and returned.</p>
	 * 
	 * 
	 * @param  file The file to create the proxy. 
	 * @return The proxy for the file, {@link NullProxy}, or <code>null</code>.
	 * @see    FileFactoryManager#getFactories(String)
	 * @see    IProxyNode#getAdapter(Class)
	 * @throws Throwable An error occurs when creating the proxy.
	 */
	private IProxyNode internalCreateProxy(IFile file) throws Throwable {

		IProxyNode proxyNode = null;

		//Resolve the list of registered file proxy factories for the file type:
		List<IFileProxyFactory> fileProxyFactories = TestNavigator.getFileFactoryManager().getFactories(file.getFileExtension());

		//If there are one or more registered file proxy factories for the file type, attempt to create the proxy:
		if(fileProxyFactories.size() > 0){

			//Resolve the iterator for the list of registered file proxy factories for the file type:
			Iterator<IFileProxyFactory> fileProxyFactoriesIterator = fileProxyFactories.iterator();

			//Iterate list of registered file proxy factories:
			while(fileProxyFactoriesIterator.hasNext()) {

				IFileProxyFactory factory = fileProxyFactoriesIterator.next();

				//Attempt to create the proxy using the file and an unbounded parent marker with the file proxy factory:
				try {
					proxyNode = factory.create(file, UNBOUNDED_PARENT);
				} 
				catch (Throwable t) {
					
					//Re-throw the exception if no other file proxy factories are registered for the file type:
					if(!fileProxyFactoriesIterator.hasNext()){
						throw t;
					}
				}

				//If the proxy was created successfully, validate the newly created proxy with the file:
				if (proxyNode != null) {

					//Validate the proxy with the file:
					Assert.isLegal(file.equals(proxyNode.getAdapter(IFile.class)), "Invalid proxy '" + proxyNode.getClass().getName() + "' returned by factory '" + factory.getClass().getName() + "' for file '" + file.getFullPath().toString() + "'."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$

					break;
				}
			}			
		}

		//If there are no registered file proxy factories for the file type, use the NullProxy:
		else{
			proxyNode = NULL_PROXY;
		}

		//If the proxy exists, save and cache the proxy with the file:
		if (proxyNode != null) {

			//If the proxy is persistable, save the proxy state in the file marker:
			if (proxyNode instanceof IPersistableProxyNode) {
				defaultPersister.saveProxy(file, ((IPersistableProxyNode)(proxyNode)));
			}

			//Cache the proxy with the file ONLY after the proxy is successfully created/validated/saved:
			fileProxyNodes.put(file, proxyNode);
		}

		return proxyNode;
	}

	/**
	 * <p>Null proxy associated to files without a proxy.</p>
	 * 
	 * <p>This proxy is required to increase the performance of the {@link FileProxyNodeCache} 
	 * to avoid multiple proxy requests by acting as a marker to indicate that no factory returned 
	 * a proxy for a given file.</p> 
	 * 
	 * <p>Note: Null proxies are only used within the {@link org.eclipse.hyades.test.ui.internal.navigator.proxy} 
	 * package and should never be displayed in the UI.</p> 
	 * 
	 * 
	 * @author  Jerome Gout
	 * @author  Jerome Bozier
	 * @author  Paul Slauenwhite
	 * @version October 22, 2009
	 * @since   January 5, 2006
	 */
	protected static class NullProxy implements IPersistableProxyNode{

		private final static String FACTORY_ID = (UiPlugin.PLUGIN_ID + ".nullFactory"); //$NON-NLS-1$

		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode#getFactoryID()
		 */
		public String getFactoryID() {
			return FACTORY_ID;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode#saveState(org.eclipse.ui.IMemento)
		 */
		public boolean saveState(IMemento memento) {
			return true;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getChildren()
		 */
		public IProxyNode[] getChildren() {
			return NO_CHILDREN;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getImage()
		 */
		public Image getImage() {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getParent()
		 */
		public Object getParent() {
			return UNBOUNDED_PARENT;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getText()
		 */
		public String getText() {
			return "<null proxy>"; //$NON-NLS-1$
		}

		/* (non-Javadoc)
		 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
		 */
		public Object getAdapter(Class adapter) {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.navigator.IProxy#getIdentifier()
		 */
		public String getIdentifier() {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.navigator.IProxy#getUnderlyingResource()
		 */
		public IResource getUnderlyingResource() {
			return null;
		}
	}
	
	/**
	 * <p>Unbounded parent representing a parent to a file proxy.</p>
	 * 
	 * <p>Note: Unbounded parents are only used within the {@link org.eclipse.hyades.test.ui.internal.navigator.proxy} 
	 * package.</p> 
	 * 
	 * 
	 * @author  Paul Slauenwhite
	 * @version October 22, 2009
	 * @since   October 22, 2009
	 */
	private static class UnboundedParent implements FileProxyManager.IUnboundedParent{
		//No-operations.
	}
	
	/**
	 * <p>Resource change listener for the {@link FileProxyNodeCache}.</p>
	 * 
	 * <p>Components that manage {@link IProxyNode}s should register a resource
	 * change listener to handle properly resource changes when notified of 
	 * resource changes.</p>
	 * 
	 * <p>Note: If a resource change listener is registered, do NOT
	 * register a resource change listener with the workspace resource services 
	 * (see {@link IWorkspace#addResourceChangeListener(IResourceChangeListener)}).</p>
	 * 
	 * 
	 * @author  Jerome Gout
	 * @author  Jerome Bozier
	 * @author  Paul Slauenwhite
	 * @version October 22, 2009
	 * @since   January 5, 2006
	 */
	private class ResourceChangeListener implements IResourceChangeListener, IResourceDeltaVisitor {

		/**
		 * <p>Registers the resource change listener with the workspace resource services 
		 * (see {@link IWorkspace#addResourceChangeListener(IResourceChangeListener)}).</p>
		 */
		public ResourceChangeListener() {
			ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		}

		/**
		 * <p>Unregisters the resource change listener from the workspace resource services 
		 * (see {@link IWorkspace#addResourceChangeListener(IResourceChangeListener)}).</p>
		 */
		public void dispose() {
			ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		}

		/* (non-Javadoc)
		 * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org.eclipse.core.resources.IResourceChangeEvent)
		 */
		public void resourceChanged(IResourceChangeEvent event) {
			
			IResourceDelta delta = event.getDelta();
			
			if(delta != null){
				
				//Delegate to resource change listeners registered in the cache:
				IResourceChangeListener[] copy;
				
				synchronized(resourceListeners) {
					copy = ((IResourceChangeListener[])(resourceListeners.toArray(new IResourceChangeListener[resourceListeners.size()])));
				}
				
				for (int counter = 0; counter < copy.length; counter++) {
					
					try {
						copy[counter].resourceChanged(event);
					} 
					catch (Throwable e) {
						UiPlugin.logError(e);
					}                
				}
				
				//Delegate to the cache:
				try {
					delta.accept(this);
				} 
				catch (CoreException e) {
					UiPlugin.logError(e);
				}
			}
		}

		/* (non-Javadoc)
		 * @see org.eclipse.core.resources.IResourceDeltaVisitor#visit(org.eclipse.core.resources.IResourceDelta)
		 */
		public boolean visit(IResourceDelta delta) throws CoreException {
			
			if (delta.getKind() == IResourceDelta.CHANGED) {
				
				if (delta.getResource().getType() == IResource.PROJECT) {
					
					if (((delta.getFlags()) & (IResourceDelta.OPEN)) != 0) {

						IProject project = ((IProject)(delta.getResource()));
						
						if (!project.isOpen()) {
							
							Iterator<IFile> filesIterator = fileProxyNodes.keySet().iterator();
							
							while(filesIterator.hasNext()) {
								
								if (project.equals(filesIterator.next().getProject())) {
									filesIterator.remove();
								}
							}
							
							return false; 
						} 
						else {		
						
							//Create the proxy nodes when a project is opened since proxy nodes are deleted when projects are closed:
							TestUIUtilities.createProxyNodes(project);
 						}
					}
				}

				return (delta.getResource().getType() != IResource.FILE);
			}
			
			if (delta.getKind() == IResourceDelta.REMOVED) {
				
				IResource resource = delta.getResource();
				
				if (resource.getType() == IResource.FILE) {
					
					remove(((IFile)(resource)));
					
					ReferenceRegistry.getInstance().removeReferences(((IFile)(resource)));
				} 
				else if(resource.getType() == IResource.PROJECT){
					ReferenceRegistry.getInstance().removeProjectReferenceRegistry(((IProject)(resource)));
					return true; // child have to be checked to delete their references
				} 
				else {
					return true;
				}
			}
			
			return false;
		}
	}

	/**
	 * <p>Error proxy used when a proxy cannot be created successfully.</p>
	 * 
	 * 
	 * @author  Paul Slauenwhite
	 * @version October 22, 2009
	 * @since   October 22, 2009
	 */
	private class ErrorProxy extends FileProxyNode{

		public ErrorProxy(IFile file){
			super(file);
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.navigator.FileProxyNode#getText()
		 */
		public String getText() {
			return (super.getText() + " [" + TestNavigatorMessages.PROXY_LOADING_ERROR + "]"); //$NON-NLS-1$ //$NON-NLS-2$
		}

		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.navigator.FileProxyNode#getImage()
		 */
		public Image getImage() {
			return (PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJS_WARN_TSK));
		}

		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getChildren()
		 */
		public IProxyNode[] getChildren() {
			return NO_CHILDREN;
		}
	}	
}
