/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestAssetGroupProxyManager.java,v 1.15 2010/05/19 11:16:17 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IMarkerDelta;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.jobs.ILock;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxyNodeListener;
import org.eclipse.hyades.test.ui.navigator.ITypeProviderContext;

/**
 * <p>This class manage the extensions of the 
 * <code>org.eclipse.hyades.test.ui.testNavigatorTestAssetGroupProxy</code> extension point.</p>
 * 
 * <p>This class manage all test asset groups created using a cache.</p>
 * 
 * 
 * @author     Jerome Gout
 * @author 	   Jerome Bozier
 * @author     Paul Slauenwhite
 * @version    May 19, 2010
 * @since      March 18, 2005
 */
public class TestAssetGroupProxyManager implements IResourceChangeListener {
	
    static class TestAssetInfo {
        private String name;
        private boolean isFlat;
        private String imageKey;
        private boolean caseSensitive;
        
        public TestAssetInfo(String name, boolean isFlat, String imageKey,boolean caseSensitive) {
            this.name = name;
            this.isFlat = isFlat;
            this.imageKey = imageKey;
            this.caseSensitive = caseSensitive;
        }
        
        /**
         * @return if case sensitive
         */
        public boolean isCaseSensitive() {
        	return caseSensitive;
        }
        
        /**
         * @return Returns the imageKey.
         */
        public String getImageKey() {
            return imageKey;
        }
        /**
         * @return Returns the isFlat.
         */
        public boolean isFlat() {
            return isFlat;
        }
        /**
         * @return Returns the name.
         */
        public String getName() {
            return name;
        }
    }
    
    /**
     * string -> TestAssetInfo
     */
	private Map extensions;
    /**
     * project -> List(ITestAssetGroupProxyNode)
     */
    private Map cache;
    private IProxyNodeListener refresher;
	
    /**
     * Reentrant, deadlock detecting, and deadlock recovering lock to guard against multiple 
     * threads creating/cache the same proxy more than once.
     */
    private final static ILock LOCK = Job.getJobManager().newLock();
    
	public TestAssetGroupProxyManager(ITypeProviderContext context) {
		
		extensions = new HashMap();
        cache = new HashMap();
        refresher = context.getProxyNodeListener();
		
        registerGroupProxies();
        
        //Listen to resource changes through proxy node cache:
        FileProxyNodeCache.getInstance().addResourceListener(this);
	}
	
	private void registerGroupProxies(){

		IExtensionPoint extensionPointt = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + ".testNavigatorTestAssetGroupProxy"); //$NON-NLS-1$
		
		if (extensionPointt != null) {
			
			IConfigurationElement[] configurationElements = extensionPointt.getConfigurationElements();
			
			for (int counter = 0; counter < configurationElements.length; counter++) {
				
				IConfigurationElement configurationElement = configurationElements[counter];
				
				if("testAsset".equals(configurationElement.getName())) { //$NON-NLS-1$
					
					String extension = configurationElement.getAttribute("extension"); //$NON-NLS-1$

					if((extension != null) && (extension.trim().length() > 0)){
					
						//Default is true:
						boolean isFlat = true;
						String flat = configurationElement.getAttribute("flat");//$NON-NLS-1$
						
						if(flat != null){
							isFlat = Boolean.parseBoolean(flat);
						}
						
						String icon = configurationElement.getAttribute("icon"); //$NON-NLS-1$

						//Optional:
						if(icon != null){
							UiPlugin.getDefault().putImage(Platform.getBundle(configurationElement.getDeclaringExtension().getNamespaceIdentifier()), icon);
						}

						//Default is true:
						boolean isCaseSensitive = true;
						String caseSensitive = configurationElement.getAttribute("caseSensitive");//$NON-NLS-1$
						
						if(caseSensitive != null){
							isCaseSensitive = Boolean.parseBoolean(caseSensitive);
						}
						
						//If not case sensitive, default to lower case:
						if (!isCaseSensitive) {
							extension = extension.toLowerCase(); 
						}
						
						addExtension(extension, new TestAssetInfo(configurationElement.getAttribute("name"), isFlat, icon, isCaseSensitive)); //$NON-NLS-1$
					}
				}
			}
		}
	}
	
	private void addExtension(String extension, TestAssetInfo info) {
		if(!extensions.containsKey(extension)) {
			extensions.put(extension, info);
		} else {
			UiPlugin.logInfo("extension: "+extension+" already registered"); //$NON-NLS-1$ //$NON-NLS-2$			
		}
	}
	
	public Collection getExtensions() {
		return extensions.keySet();
	}
    
    public ITestAssetGroupProxyNode getTestAssetGroup(IProject project, String extension) {
    	ITestAssetGroupProxyNode group = cacheGet(project, extension);
    	if (group == null) {
    		TestAssetInfo info = (TestAssetInfo)extensions.get(extension);
    		if (info == null) { // may have be stored in lower case if not case sensitive
    			info = (TestAssetInfo)extensions.get(extension.toLowerCase());
    			if (info != null) { 
    				if (info.isCaseSensitive()) {
    					info = null; // case sensitive, so invalid info
    				} else {
    					extension = extension.toLowerCase();
    				}
    			}
    		}
	        if(extensions.containsKey(extension)) {
	        	//- this extension has a test asset group registered to display those files

	            //- build the proxy node with the right parameters
	            try {

	    			//Acquire a lock to guard against multiple threads creating/cache the same proxy more than once:
		    		LOCK.acquire();
	            	
		    		// the group may be have been computed in another thread while we
		    		// were acquiring the lock.
		    		group = cacheGet(project, extension);
		    		if (group == null) {
		    			group = new TestAssetGroupProxyNode(project, extension, info.getName(), info.getImageKey(), info.isFlat, project,info.isCaseSensitive());
		    			cachePut(project, extension, group);
		    		}
	            } 
	            finally {

	            	//Release the lock:
		    		LOCK.release();
	            }
	        }
    	}
    	
    	if(group != null && group.getChildren().length > 0) {
    		return group;
    	}
        return null;
    }

    private void cachePut(IProject project, String extension, ITestAssetGroupProxyNode testAsset) {
    	Map map = (Map) cache.get(project);
    	if (map == null) {
    		map = new HashMap();
    		cache.put(project, map);
    	}
    	map.put(extension, testAsset);
    }
    
    private ITestAssetGroupProxyNode cacheGet(IProject project, String extension) {
    	Map map = (Map) cache.get(project);
    	if (map != null) {
    		ITestAssetGroupProxyNode group = (ITestAssetGroupProxyNode) map.get(extension);
    		if (group != null) return group;
    	}
    	return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org.eclipse.core.resources.IResourceChangeEvent)
     */
    public void resourceChanged(IResourceChangeEvent event) {
        Object lowestChanged = null;
        IResourceDelta resDelta = event.getDelta();
        //- get all projects affected by the current change
        IResourceDelta[] affectedProjects = resDelta.getAffectedChildren();
        affectedProjects = filter(affectedProjects);
        for (int i = 0; i < affectedProjects.length; i++) {
            IResourceDelta delta = affectedProjects[i];
            IProject project = (IProject)delta.getResource();
            //- given a project we need to update all test asset group proxy nodes created in this project
            Map groups = (Map)cache.get(project);
            if(groups != null) {
                for (Iterator it = extensions.keySet().iterator(); it.hasNext();) {
                	String ext = (String)it.next();
                    TestAssetGroupProxyNode proxy = (TestAssetGroupProxyNode) groups.get(ext);
                    if (proxy != null) {
	                    IProxyNode lowestProxy = proxy.resourceChanged(delta);
	                    if(proxy.getChildren().length == 0) {
	                        //- this means that the group no longer contains children
	                        lowestChanged = project;
	                        lowestProxy = null;
	                    }
	                    if(lowestProxy != null) {
	                        //- there was a change in this test asset group proxy node tree
	                        if(lowestChanged == null) {
	                            //- this was the first change
	                            lowestChanged = lowestProxy;
	                        } else {
	                            //- the second one ... we need to refresh the owner of those nodes (the project)
	                            lowestChanged = project;
	                        }
	                    }
                    }
                }
            }
        }
        //- refresh only the lowest node
        if(lowestChanged != null) {
            refresher.nodeChanged(lowestChanged);
        }
    }

    public void dispose() {
        FileProxyNodeCache.getInstance().removeResourceListener(this);
    }

    private IResourceDelta[] filter(IResourceDelta[] deltas) {

    	if ((deltas == null) || (deltas.length == 0)) {
    		return deltas;
    	}

    	final Set<IResourceDelta> resourceDeltas = new HashSet<IResourceDelta>(deltas.length);

    	for (int i=0; i<deltas.length; i++) {

    		try {
    			final IResourceDelta projectDelta = deltas[i]; 
    			projectDelta.accept(new IResourceDeltaVisitor(){

    				/* (non-Javadoc)
    				 * @see org.eclipse.core.resources.IResourceDeltaVisitor#visit(org.eclipse.core.resources.IResourceDelta)
    				 */
    				public boolean visit(IResourceDelta delta) {

    					IResource res = delta.getResource();

    					if( res.getType() != IResource.FILE ){
    						return true;
    					}

    					switch( delta.getKind() ) {
    					case IResourceDelta.CHANGED:
    						
    						int flags = delta.getFlags();
    						
    						if ((flags & IResourceDelta.MARKERS) != 0) {
    						
    							IMarkerDelta markdelta[] = delta.getMarkerDeltas();
    							
    							for (int j=0; j<markdelta.length; j++) {
    							
    								if (markdelta[j].isSubtypeOf(IMarker.PROBLEM)) {
    									resourceDeltas.add(projectDelta); // problem marker appear on navigator
    									return false;
    								}
    							}
    						
    							break;
    						}
    						
    						resourceDeltas.add(projectDelta);
    						break;
    					case IResourceDelta.ADDED :
    					case IResourceDelta.REMOVED :
    						resourceDeltas.add(projectDelta);
    						break;
    					}
    					return false;
    				}
    			});
    		} 
    		catch (CoreException e) {
    			UiPlugin.logError(e);
    		}
    	}

    	return ((IResourceDelta[])(resourceDeltas.toArray(new IResourceDelta[resourceDeltas.size()])));
    }
}
