/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestProvider.java,v 1.14 2010/05/06 12:28:31 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxy;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxyNodeListener;
import org.eclipse.hyades.test.ui.navigator.ITestCaseProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITestSuiteProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITypeProviderContext;
import org.eclipse.hyades.test.ui.navigator.ITypeProviderProxyNode;
import org.eclipse.hyades.ui.extension.IAssociationConstants;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.internal.extension.AssociationMappingRegistry;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.LabelProviderChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.model.IWorkbenchAdapter;

/**
 * <p>Label and content provider for representing a test workspace.</p>
 * 
 * <p>This provider providers a snapshot (not updated when the 
 * underlying workspace changes) of all tests in the workspace rendered
 * in a logical view.</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Julien Canches
 * @version May 6, 2010
 * @since   April 7, 2005
 */
public class TestProvider extends LabelProvider implements ITreeContentProvider, IProxyNodeListener {
	
	protected FileProxyManager fileProxyManager;
    private IFilter filter;
    protected TypeProviderManager manager;
    protected IProxyNodeListener proxyNodeListener;
	private Map projectToTypeProviders = new HashMap();
    
    public TestProvider(IFilter filter, final boolean allowUpdate) {
        this.fileProxyManager = new FileProxyManager() {
			public IProxyNode getProxy(IFile file, Object parent) {
				String ext = file.getFileExtension();
				if (ICommonConstants.TEST_SUITE_FILE_EXTENSION.equals(ext) ||
					"java".equals(ext.toLowerCase())) { //$NON-NLS-1$
					return super.getProxy(file, parent);
				}
				return null;
			}
        	
        };
        this.filter = filter;
        if (allowUpdate) {
        	this.proxyNodeListener = new IProxyNodeListener() {
				public void nodeChanged(final Object node) {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							TestProvider.this.nodeChanged(node);
						}
					});
				}
        	};
        } else {
        	this.proxyNodeListener = new IProxyNodeListener() {
				public void nodeChanged(Object node) {
					// NOP
				}
			};
        }
        this.manager = new TypeProviderManager(new ITypeProviderContext() {
			public IProxyNodeListener getProxyNodeListener() {
				return proxyNodeListener;
			}
			public IFileProxyManager getFileProxyManager() {
				return fileProxyManager;
			}
			public boolean isStaticView() {
				return !allowUpdate;
			}
		});
    }

    public void addListener(IProxyNodeListener listener) {
    	addListenerObject(listener);
    }
    
    public void removeListener(IProxyNodeListener listener) {
    	removeListenerObject(listener);
    }
    
    private List getTypeProviderProxyNodes(IProject project) {
        List result = (List) projectToTypeProviders.get(project);
        if (result == null) {
        	result = new ArrayList();
            AssociationMappingRegistry registry = (AssociationMappingRegistry) TestUIExtension.getTestSuiteMappingRegistry();
            IAssociationMapping associationMapping = registry.getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS);
            String[] types = associationMapping.getTypes();
            for (int i = 0; i < types.length; i++) {
                if (filter.satisfies(types[i])) {
                    IAssociationDescriptor descriptor = associationMapping.getDefaultAssociationDescriptor(types[i]);
                    if (descriptor != null) {
                        ITypeProviderProxyNode proxy = manager.getTypeProviderProxyNode(project, types[i]);
                        if (proxy != null) {
                            result.add(proxy);
                        }
                    }
                }
            }
            projectToTypeProviders.put(project, result);
        }
        return result;
    }

    public Object[] getChildren(Object parentElement) {
        if (parentElement instanceof IWorkspaceRoot) {
            try {
                IContainer container = (IContainer) parentElement;
                IResource[] members = container.members();
                List children = new ArrayList(members.length);
                for (int i = 0; i < members.length; i++) {
                    if ((members[i] instanceof IProject) && (members[i].isAccessible()) && (!TestNavigator.getFiltersManager().filter(members[i]))) {
                        children.add(members[i]);
                    }
                }
                return children.toArray();
            } catch (CoreException e) {
                UiPlugin.logError(e);
                return new Object[0];
            }
        } else if (parentElement instanceof IProject) {
            List proxies = filterEmptyTypeProviderProxyNodes(getTypeProviderProxyNodes((IProject) parentElement));
            if (proxies.size() == 1) {
                // Skip this level
                return getChildren(proxies.get(0));
            }
            return proxies.toArray();
        } else if (parentElement instanceof ITestSuiteProxyNode && filter.satisfiesChildren((ITestSuiteProxyNode)parentElement)) {
            List testCases = new ArrayList();
            collectTestCases(testCases, (ITestSuiteProxyNode) parentElement);
            return testCases.toArray();
        } else if (parentElement instanceof IProxyNode && filter.satisfiesChildren((IProxyNode)parentElement)) {
        	List proxies = filterProxyNodes(((IProxyNode)parentElement).getChildren());
            return proxies.toArray();
        } else {
            return new Object[0];
        }
    }
    
    private List filterProxyNodes(IProxyNode[] proxies) {
    	ArrayList ret = new ArrayList(proxies.length);
    	for (int i = 0; i < proxies.length; i++) {
			IProxyNode pn = proxies[i];
			if (filter.satisfies(pn) && (pn instanceof ITestSuiteProxyNode || getChildren(pn).length > 0)) {
				ret.add(pn);
			}
		}
    	return ret;
    }
    
    private List filterEmptyTypeProviderProxyNodes(List proxies) {
    	ArrayList ret = new ArrayList(proxies.size());
    	Iterator it = proxies.iterator();
    	while (it.hasNext()) {
			IProxyNode pn = (IProxyNode) it.next();
			if (pn.getChildren().length > 0) {
				ret.add(pn);
			}
		}
    	return ret;
    }

    /**
     * @param parentElement
     * @return
     */
    private void collectTestCases(List list, IProxyNode proxyNode) {
        IProxyNode[] children = proxyNode.getChildren();
        for (int i = 0; i < children.length; i++) {
            if (children[i] instanceof ITestCaseProxyNode) {
                if (filter.satisfies(((ITestCaseProxyNode) children[i]).getType())) {
                    list.add(children[i]);
                }
            } else {
                collectTestCases(list, children[i]);
            }
        }
    }

    public Object getParent(Object element) {
        if (element instanceof IResource) {
            return ((IResource) element).getParent();
        } else if (element instanceof ITestCaseProxyNode) {
            return getParentTestSuiteProxyNode((ITestCaseProxyNode) element);
        } else if (element instanceof IProxyNode) {
            Object parent = fileProxyManager.getParent((IProxyNode) element);
            if (parent instanceof ITypeProviderProxyNode) {
            	ITypeProviderProxyNode dtpParent = (ITypeProviderProxyNode) parent;
                IProject project = (IProject) dtpParent.getUnderlyingResource();
                if (filterEmptyTypeProviderProxyNodes(getTypeProviderProxyNodes(project)).size() == 1) {
                    return project;
                }
            }
            return parent;
        }
        return null;
    }

    /**
     * @param node
     * @return
     */
    private Object getParentTestSuiteProxyNode(IProxyNode node) {
        Object parent = fileProxyManager.getParent(node);
        if (parent == null) {
            return null;
        } else if (parent instanceof ITestSuiteProxyNode) {
            return parent;
        } else if (parent instanceof IProxyNode) {
            return getParentTestSuiteProxyNode((IProxyNode) parent);
        } else {
            return null;
        }
    }

    public boolean hasChildren(Object element) {
    	if (element instanceof IProject) {
    		// Optimization to avoid digging into a whole project, just to
    		// determine if it has children. We assume that it has, and will
    		// delay the real discovery of the project content until the specific
    		// content is requested by the user.
    		return true;
    	}
        return getChildren(element).length > 0;
    }

    public Object[] getElements(Object inputElement) {
        return getChildren(inputElement);
    }

    public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
        // Not implemented: the input may not change
    }

    public Image getImage(Object element) {
        if (element instanceof IContainer) {
            ImageDescriptor descriptor;
            IWorkbenchAdapter workbenchAdapter = getWorkbenchAdapter(element);
            if (workbenchAdapter != null) {
                descriptor = workbenchAdapter.getImageDescriptor(element);
                if (descriptor != null) return descriptor.createImage();
            }
        } else if (element instanceof IProxyNode) {
            return ((IProxyNode) element).getImage();
        }
        return null;
    }

    public String getText(Object element) {
        if (element instanceof IContainer) {
            IWorkbenchAdapter workbenchAdapter = getWorkbenchAdapter(element);
            if (workbenchAdapter != null) {
                return workbenchAdapter.getLabel(element);
            }
        } else if (element instanceof IProxyNode) {
            return ((IProxyNode) element).getText();
        }
        return null;
    }

    private IWorkbenchAdapter getWorkbenchAdapter(Object object) {
        IWorkbenchAdapter workbenchAdapter = (IWorkbenchAdapter) Platform.getAdapterManager().getAdapter(object, IWorkbenchAdapter.class);
        if (workbenchAdapter == this) return null;
        return workbenchAdapter;
    }

    /**
     * @return Returns the fileProxyManager.
     */
    public FileProxyManager getFileProxyManager() {
        return fileProxyManager;
    }

	public void dispose() {
		super.dispose();
		manager.dispose();
		fileProxyManager.dispose();
	}
	
	public IProxyNode getCorrespondingProxy(TPFTest test) {
		IProxy proxy = (IProxy) test.getAdapter(IProxy.class);
		if (proxy != null) {
			IResource res = proxy.getUnderlyingResource();
			if (res instanceof IFile) {
				// Ensure that the project content is loaded
				getTypeProviderProxyNodes(res.getProject());
			}
		}
		return fileProxyManager.getCorrespondingProxy(test);
	}

	public void nodeChanged(Object node) {
		Object[] listeners = getListeners();
		for (int i = 0; i < listeners.length; i++) {
			if (listeners[i] instanceof ILabelProviderListener) {
				try {
					((ILabelProviderListener)listeners[i]).labelProviderChanged(new LabelProviderChangedEvent(TestProvider.this, node));
				} catch (Throwable t) {
					UiPlugin.logError(t);
				}
			} else if (listeners[i] instanceof IProxyNodeListener) {
				try {
					((IProxyNodeListener)listeners[i]).nodeChanged(node);
				} catch (Throwable t) {
					UiPlugin.logError(t);
				}
			}
		}
	}
   
}