/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TypeProviderManager.java,v 1.11 2010/05/21 15:28:57 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.async.ISynchronizedAccess;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.async.JobSynchronizedAccess;
import org.eclipse.hyades.test.ui.navigator.ITypeProvider;
import org.eclipse.hyades.test.ui.navigator.ITypeProviderContext;
import org.eclipse.hyades.test.ui.navigator.ITypeProviderProxyNode;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;

/** 
 * <p>Manager for <code>testNavigatorTypeProvider</code> extension point.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Paul Slauenwhite
 * @version May 21, 2010
 * @since   March 18, 2005
 */
public class TypeProviderManager {

    static interface ITypeProviderInfo extends ISynchronizedAccess {

        /**
         * @return Returns the imageKey.
         */
        public String getImageKey();

        /**
         * @return Returns the name.
         */
        public String getName();

        /**
         * @return Returns the provider.
         */
        public ITypeProvider getProvider();

    }
    
    class ExtensionTypeProviderInfo extends JobSynchronizedAccess implements ITypeProviderInfo {
        private String name;
        private String imageKey;
        /**
         * This is the place of the provider itself, but it can contain the IConfigurationElement due to lazy load of object.
         */
        private Object provider;
        
        public ExtensionTypeProviderInfo(String name, String imageKey, Object provider) {
            this.name = name;
            this.imageKey = imageKey;
            this.provider = provider;
        }

        /* (non-Javadoc)
         * @see org.eclipse.hyades.test.ui.internal.navigator.proxy.ITypeProviderInfo#getImageKey()
         */
        public String getImageKey() {
            return imageKey;
        }
        /* (non-Javadoc)
         * @see org.eclipse.hyades.test.ui.internal.navigator.proxy.ITypeProviderInfo#getName()
         */
        public String getName() {
            return name;
        }
        /* (non-Javadoc)
         * @see org.eclipse.hyades.test.ui.internal.navigator.proxy.ITypeProviderInfo#getProvider()
         */
        public ITypeProvider getProvider() {
            if(provider instanceof ITypeProvider) {
                return (ITypeProvider)provider;
            } else {
                ITypeProvider p = null;
                try {
                    p = (ITypeProvider)((IConfigurationElement)provider).createExecutableExtension("class"); //$NON-NLS-1$
                } catch (CoreException e) {
                    UiPlugin.logError(e);
                    return null;
                }
                p.init(context);
				provider = p;
                return p;
            }
        }
    }
    
    class DefaultTypeProviderInfo extends JobSynchronizedAccess implements ITypeProviderInfo {
        private DefaultTypeProvider defaultTypeProvider;
        private String name;
        
        public DefaultTypeProviderInfo(String type) {
            defaultTypeProvider = new DefaultTypeProvider();
            defaultTypeProvider.init(context);
            name = TestNavigator.getRegisteredTypeName(type);
        }

        public String getImageKey() {
            //- default folder icon should be used for default type provider node.
            return null;
        }

        public String getName() {
            return name;
        }

        public ITypeProvider getProvider() {
            return defaultTypeProvider;
        }
    }
    
    
	/**
	 * Map to cache the type providers.
     * type -> ExtensionTypeProviderInfo
	 */
	private Map providers;
	
	private ITypeProviderContext context;

	public TypeProviderManager(ITypeProviderContext context) {
        
		providers = Collections.synchronizedMap(new HashMap());
        this.context = context;
        
        IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + ".testNavigatorTypeProvider"); //$NON-NLS-1$

        if (extPoint != null) {
            
        	IConfigurationElement[] members = extPoint.getConfigurationElements();
            
            for (int i = 0; i < members.length; i++) {
                
            	IConfigurationElement element = members[i];
                
                if("provider".equals(element.getName())) { //$NON-NLS-1$
                    
                    String name = element.getAttribute("label"); //$NON-NLS-1$
                    
                    //Backward compatibility with previous attribute name:
                    if(name == null){
                    	name = element.getAttribute("name"); //$NON-NLS-1$
                    }
                    
                    String imageKey = element.getAttribute("icon"); //$NON-NLS-1$
                    
                    if(imageKey != null) {
                    	
                    	ImageRegistry imageRegistry = UiPlugin.getInstance().getImageRegistry();

                    	if(imageRegistry.get(imageKey) == null){

                    		ImageDescriptor imageDescriptor = UIUtil.getImageDescriptorFromPlugin(Platform.getBundle(element.getDeclaringExtension().getNamespaceIdentifier()), imageKey);
                    		
                    		if(imageDescriptor != null) {
                    			imageRegistry.put(imageKey, imageDescriptor);
                    		}
                    	}
                    }
                    
                    addProvider(element.getAttribute("type"), new ExtensionTypeProviderInfo(name, imageKey, element)); //$NON-NLS-1$
                }
            }
        }
    }
    
    private void addProvider(String type, ITypeProviderInfo info) {
        if(!providers.containsKey(type)) {
            providers.put(type, info);
        } else {
            UiPlugin.logInfo("Provider for type: "+type+" already registered"); //$NON-NLS-1$ //$NON-NLS-2$            
        }
    }
	
	/**
     * Returns a type provider proxy for the given project and type.<br> 
     * The returned proxy node is the root node in the navigator just beneath the project one. 
     * It contains all elements of the given type as children.
	 * @param project a project
	 * @param type a TPTP test type of element (such as "<code>org.eclipse.hyades.test.java.junit.testSuite</code>").
	 * @return a type provider proxy node. 
	 */
	public ITypeProviderProxyNode getTypeProviderProxyNode(IProject project, String type) {
		synchronized(providers) {
	    	if(!providers.containsKey(type)) {
	    		//- since no provider registered for the given type, add the default one for the given type in the map
	            addProvider(type, new DefaultTypeProviderInfo(type));
	    	}
		}
        //- By construction, the internal map of providers has a provider for the given type.
    	ITypeProviderInfo info = (ITypeProviderInfo)providers.get(type);
    	//- deleguate to client
    	ITypeProvider provider = info.getProvider();
    	try {
    		if (info.acquireLock(project)) {
    			return provider.get(project, type);
    		}
    		return null;
    	} finally {
    		info.releaseLock(project);
    	}
	}

    /** Returns the name registered in the extension for the given type of element.
     * @param type the type element for which the name has been registered.
     * @return The name of the type provider node to used or <code>null</code> 
     * if no registration for the given type or if the label to use for this type provider should be one given 
     * in the declaration of the type (using <code>org.eclipse.hyads.ui.typeDescriptions</code> extension point). 
     */
    public String getNameFor(String type) {
        if(providers.containsKey(type)) {
            return ((ITypeProviderInfo)providers.get(type)).getName();
        }
        return null;
    }

    /** Returns the image key registered in the extension for the given type of element.
     * @param type the type element for which the image key has been registered.
     * @return The image key of the type provider node to used or <code>null</code> 
     * if no registration for the given type or if the icon to use for this type provider should be the default folder icon. 
     */
    public String getImageKeyFor(String type) {
        if(providers.containsKey(type)) {
            return ((ITypeProviderInfo)providers.get(type)).getImageKey();
        }
        return null;
    }

    public void dispose() {
        //- dispose all registered type providers.
        for (Iterator it = providers.values().iterator(); it.hasNext();) {
            ITypeProviderInfo info = (ITypeProviderInfo) it.next();
            ITypeProvider provider = info.getProvider();
            //TODO: when ITypeProvider will extend IDisposable the following cast is unnecessary.
            if (provider instanceof IDisposable) {
                ((IDisposable)provider).dispose();
            }            
        }
    }
}
