/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TypedElementFactoryManager.java,v 1.8 2008/03/20 19:49:54 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.HashMap;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.navigator.DefaultTypedElementProxyFactory;
import org.eclipse.hyades.test.ui.navigator.ITypedElementProxyFactory;

/**
 * @author jgout
 * @version February 26, 2008
 * @since 3.2
 */
public class TypedElementFactoryManager {
	
	private static TypedElementFactoryManager instance;
	private HashMap factories;
	private HashMap types;
	
	public static TypedElementFactoryManager getInstance() {
		if(instance == null) {
			instance = new TypedElementFactoryManager();
		}
		return instance;
	}
	
	private TypedElementFactoryManager() {
		factories = new HashMap();
		types = new HashMap();
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + ".testNavigatorTypedElementProxyFactory"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) {
				IConfigurationElement element = members[i];
				if ("factory".equals(element.getName())) { //$NON-NLS-1$
					String id = element.getAttribute("id"); //$NON-NLS-1$
					addFactory(id, element);
				} else if("typedElement".equals(element.getName())) { //$NON-NLS-1$
					//- get extension of files we want to make visible
					String type = element.getAttribute("type"); //$NON-NLS-1$
					String factoryID = element.getAttribute("factoryID"); //$NON-NLS-1$
					if(type != null && type.length() != 0) {
						addType(type, factoryID);
					}
				}
			}
		}
		//- backward compatibility .. this following statements should be removed in the future version
		extPoint = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + ".testNavigatorTypedElementFactory"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) {
				IConfigurationElement element = members[i];
				if ("typedElement".equals(element.getName())) { //$NON-NLS-1$
					//- get extension of files we want to make visible
					String typeName = element.getAttribute("type"); //$NON-NLS-1$
					if(typeName != null && typeName.length() != 0) {
						//- build the factoryID for the name of the factory class given in the extension
						String id = element.getAttribute("factory"); //$NON-NLS-1$
						addFactory(id, element);
						addType(typeName, id);
					}
				}
			}
		}
	}
	
	private void addFactory(String id, IConfigurationElement element) {
		if(!factories.containsKey(id)) {
			//- if the given type is not yet in the internal map, add it
			factories.put(id, element);
		} else {
			UiPlugin.logInfo("ID: "+id+" already used in a previously registered factory"); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}
	
	private void addType(String type, String factoryID) {
		if(!types.containsKey(type)) {
			types.put(type, factoryID);
		} else {
			UiPlugin.logInfo("type: "+type+" already registered"); //$NON-NLS-1$ //$NON-NLS-2$			
		}
	}
	
	public ITypedElementProxyFactory getFactory(String typeName) {
		if(types.containsKey(typeName)) {
			return getFactoryFromID((String)types.get(typeName));
		} else {
				return new DefaultTypedElementProxyFactory();
		}
	}

	/** Returns the factory associated to the given id. 
	 * The factory should have been registered using the following extension point:  <code>org.eclipse.hyades.test.ui.testNavigatorTypedElementProxyFactory</code> 
	 * @param factoryID the id of a registered factory.
	 * @return the typed element proxy factory which has been registered with the given id.
	 */
	public ITypedElementProxyFactory getFactoryFromID(String factoryID) {
		Object value = factories.get(factoryID);
		if(value == null) return null;
		if (value instanceof IConfigurationElement) {
			//- if the factory has not yet been loaded
			IConfigurationElement element = (IConfigurationElement) value;
			ITypedElementProxyFactory factory = null;
			try {
				if("factory".equals(element.getName())) { //$NON-NLS-1$
					factory = (ITypedElementProxyFactory)element.createExecutableExtension("class"); //$NON-NLS-1$
				} else {
					//- old extension point
					factory = (ITypedElementProxyFactory)element.createExecutableExtension("factory"); //$NON-NLS-1$
				}
			} catch (CoreException e) {
				UiPlugin.logError(e);
				//- problem building the instance of factory using reflection
				factory = new DefaultTypedElementProxyFactory();
			}
			factories.put(factoryID, factory);
			return factory;
		} else {
			//- the factory has been loaded yet
			if (value instanceof ITypedElementProxyFactory) {
				return (ITypedElementProxyFactory) value;
			} else {
				UiPlugin.logError("internal error"); //$NON-NLS-1$
				//- problem with stored factory (this should never appear)
				return null;
			}
		}
	}

}
