/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JobSynchronizedAccess.java,v 1.3 2007/04/26 18:50:09 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy.async;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;

/**
 * An implementation of ISynchronizedAccess specially optimized for running in jobs.
 * The platform job manager is used to acquire and release locks, and a job waiting
 * for lock acquisition may let another non-blocked job meanwhile.
 * @author jcanches
 */
public class JobSynchronizedAccess implements ISynchronizedAccess {

	/**
	 * The list of rules that this instance currently owns. Must be protected against concurrent
	 * access because two jobs may acquire/release a lock for two different resources at the
	 * same time. Use an array list as small as the maximum number of background jobs the
	 * test navigator may launch, plus one (the UI thread), since there should be not more
	 * than this number of threads acquiring a lock at any time.
	 */
	private Collection rules = Collections.synchronizedList(new ArrayList(TestNavigator.BACKGROUND_JOB_POOL_SIZE + 1));
	
	/**
	 * A rule that ensures exclusive access to a given resource, when it has the same owner.
	 * In other words: a given owner can only have one job running within a ResourceRule
	 * block for a given resource.
	 * @author jcanches
	 */
	private static class ResourceRule implements ISchedulingRule {

		private JobSynchronizedAccess owner;
		private Object resource;

		public ResourceRule(JobSynchronizedAccess owner, Object resource) {
			this.owner = owner;
			this.resource = resource;
		}
		
		public boolean contains(ISchedulingRule rule) {
			if (rule == this) return true;
			// Any rule than can be contained in the workspace rule is also accepted
			return ResourcesPlugin.getWorkspace().getRoot().contains(rule);
		}

		public boolean isConflicting(ISchedulingRule rule) {
			if (rule instanceof ResourceRule) {
				ResourceRule oRule = (ResourceRule) rule;
				return this.owner == oRule.owner && this.resource.equals(oRule.resource);
			}
			return false;
		}
	}
	
	public boolean acquireLock(Object resource) {
		ResourceRule rule = new ResourceRule(this, resource);
		Platform.getJobManager().beginRule(rule, new NullProgressMonitor());
		rules.add(rule);
		return true;
	}

	private ResourceRule removeRule(Object resource) {
		for (Iterator it = rules.iterator(); it.hasNext();) {
			ResourceRule rule = (ResourceRule) it.next();
			if (rule.resource.equals(resource)) {
				it.remove();
				return rule;
			}
		}
		return null;
	}
	
	public void releaseLock(Object resource) {
		ResourceRule rule = removeRule(resource);
		if (rule != null) {
			Platform.getJobManager().endRule(rule);
		}
	}

}
