/*******************************************************************************
 * Copyright (c) 2006, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProxiesRequest.java,v 1.4 2009/04/16 13:20:16 bjerome Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy.async;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.ProxyRequestManager;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * A proxies request is a job that performs the computation, for a list
 * of keys, of a list of matching proxy nodes. Assuming that the proxies computation is
 * a long-running operation, the request can be queried for results at any time, even if
 * the result for the given key has not been computed yet. In this case, the request
 * returns a placeholder proxy.
 * The caller has the choice of waiting until the computation is completed, or not waiting 
 * and being notified when the computation progresses, or a mix of the two (i.e. wait for a
 * limited time, then switch to a notification method).
 * @author jcanches
 * @author jbozier
 * @since 4.3
 * @version April 15, 2009
 */
public abstract class ProxiesRequest extends TestNavigatorJob implements IDisposable {

	private IProxiesRequestListener listener;
	private List requests;
	private Map results = Collections.synchronizedMap(new HashMap());
	private boolean cancelled;
	private long lastNotificationTime;
	private long notificationInterval;
	private boolean notificationNeeded;
	private State state = new State();
	private boolean issueCancelNotification = false;

	protected ProxiesRequest(String name, Collection keys, TestNavigator testNavigator) {
		super(name, testNavigator);
		ProxyRequestManager.instance().startRequest(this);
		this.requests = new ArrayList(keys);
		for (Iterator it = keys.iterator(); it.hasNext();) {
			Object key = it.next();
			IProxyNode proxy = getPlaceHolder(key);
			results.put(key, proxy);
		}
	}

	public IProxyNode getProxy(Object key) {
		return (IProxyNode)results.get(key);
	}

	/**
	 * Waits until this request is executed, or until the specified time-out is reached.
	 * @param timeOut A time, in milliseconds, that this method should not exceed.
	 * @param listener A listener that will be notified, if the request exceeds the time.
	 * @param notificationInterval The pace, in milliseconds, at which the method
	 * {@link IProxiesRequestListener#proxiesComputed(ProxiesRequest)} should be
	 * invoked. Only used if the request exceeds the time allocated.
	 * @return Whether the request could be completed in time or not. If not, the listener will
	 * be used to get asynchronous update of the state of the request.
	 * @throws OperationCanceledException If the method dispose() has already been called.
	 */
	public boolean wait(int timeOut, IProxiesRequestListener listener, int notificationInterval) throws OperationCanceledException {
		if (cancelled) throw new OperationCanceledException();
		synchronized(state) {
			if (state.getState() != State.COMPLETED) {
				try {
					state.wait(timeOut);
				} catch (InterruptedException e) {
					// OK
				}
			}
		}
		switch(state.getState()) {
		case State.RUNNING:
		case State.INITIALIZED: {
			this.notificationInterval = notificationInterval;
			this.listener = listener;
			this.lastNotificationTime = System.currentTimeMillis();
			return false;
		}
		case State.COMPLETED: {
			this.listener = listener;
			fireNotification(state);
			return true;
		}
		case State.CANCELLED: {
			throw new OperationCanceledException();
		}
		default: {
			throw new RuntimeException("Illegal internal state"); //$NON-NLS-1$
		}
		}
	}
	
	public boolean isPending() {
		int s = state.getState();
		return s == State.INITIALIZED || s == State.RUNNING;
	}

	/**
	 * Performs the computation. This method will trigger notification once a listener
	 * is registered. This method should be run in a separate thread.
	 */
	public IStatus run(IProgressMonitor monitor) {
		monitor.beginTask(getTaskName(), requests.size());
		synchronized(state) {
			state.set(State.RUNNING);
		}
		Thread.currentThread().setPriority(Thread.MIN_PRIORITY + 2);
		try {
			for (Iterator it = requests.iterator(); !cancelled && it.hasNext();) {
				Object key = it.next();
				monitor.subTask(getSubTaskName(key));
				_computeProxy(key);
				if (monitor.isCanceled()) {
					cancelled = true;
				} else {
					monitor.worked(1);
				}
			}
		} finally {
			synchronized(state) {
				state.set(cancelled ? State.CANCELLED : State.COMPLETED);
				fireNotification(state);
				state.notifyAll();
			}
			monitor.done();
		}
		return new Status(IStatus.OK, UiPlugin.PLUGIN_ID, 0, "", null); //$NON-NLS-1$
	}
		
	private void _computeProxy(Object key) {
		IProxyNode proxy = computeProxy(key);
		results.put(key, proxy);
		ProxyRequestManager.instance().endRequest(this);
		fireNotification(key);
	}
	
	abstract protected IProxyNode getPlaceHolder(Object key);
	
	abstract protected IProxyNode computeProxy(Object key);
	
	abstract protected String getTaskName();
	
	abstract protected String getSubTaskName(Object key);
	
	private void fireNotification(Object key) {
		if (listener == null) return;
		try {
			if (key instanceof State) {
				if (state.getState() == State.CANCELLED) {
					if (issueCancelNotification) {
						listener.computationCancelled(this);
					}
				} else {
					if (notificationNeeded) {
						listener.proxiesComputed(this);
					}
					listener.computationCompleted(this);
				}
			} else if (!cancelled) {
				listener.proxyComputed(this, key);
				long now = System.currentTimeMillis();
				if (lastNotificationTime + notificationInterval < now) {
					listener.proxiesComputed(this);
					notificationNeeded = false;
					lastNotificationTime = now;
				} else {
					notificationNeeded = true;
				}
			}
		} catch (Throwable t) {
			UiPlugin.logError(t);
		}
	}
	
	public boolean cancel(int timeOut) {
		cancelled = true;
		// Wait a few seconds for the running thread to complete
		synchronized(state) {
			if (state.getState() == State.RUNNING) {
				try {
					state.wait(timeOut);
				} catch (InterruptedException e) {
					// OK
				}
				if (state.getState() == State.RUNNING) {
					issueCancelNotification = true;
					return false;
				}
			}
			return true;
		}
	}
	
	public void dispose() {
		cancel(1);
	}
	
	private static class State {
		
		public final static int INITIALIZED = 0;
		public final static int RUNNING = 1;
		public final static int CANCELLED = 2;
		public final static int COMPLETED = 3;
		
		private int state = INITIALIZED;
		
		public void set(int state) {
			this.state = state;
		}
		
		public int getState() {
			return state;
		}
		
	}
	
}
