/********************************************************************** 
 * Copyright (c) 2008, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ProjectReferenceRegistry.java,v 1.3 2009/05/28 16:12:58 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy.reference;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ISaveContext;
import org.eclipse.core.resources.ISaveParticipant;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyNodeCache;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.WorkbenchException;
import org.eclipse.ui.XMLMemento;

/**
 * This class is used to store proxy references for a given project.
 * Its persistency is made using a save participant.
 * 
 * @author jgout
 * @author jbozier
 * @version May 27, 2009
 * @since 4.3
 */
public class ProjectReferenceRegistry implements ISaveParticipant {
	
	public static final String TAG_REFERENCE = "reference"; //$NON-NLS-1$
	public static final String TAG_REFERENCER = "referencer"; //$NON-NLS-1$
	public static final String TAG_REF_TYPE = "refType"; //$NON-NLS-1$
	
	private Map references;
	private boolean hasChanged;
	private IProject project;
	
	class Reference {
		private String type;
		private IFile referenced;

		public Reference(String typeRef, IFile referenced) {
			this.type = typeRef;
			this.referenced = referenced;
		}

		public IFile getReferenced() {
			return referenced;
		}

		public String getType() {
			return type;
		}
		
		public boolean equals(Object arg0) {
			if(arg0 == this) return true;
			if (arg0 instanceof Reference) {
				Reference ref = (Reference) arg0;
				if (ref != null) {
					if(type != null && referenced != null) {
						return type.equals(ref.type) && referenced.equals(ref.referenced);
					} else if((type == null && ref.type == null) || (referenced == null && ref.referenced == null)) {
						return true;
					}
				}
			}
			return false;
		}

		public int hashCode() {
			return referenced != null ? referenced.hashCode() : -1;
		}
	}
	
	public ProjectReferenceRegistry(IProject project, IReferenceRegistrySavedState savedState) {
		this.references = new HashMap();
		this.hasChanged = false;
		this.project = project;
		if (savedState != null) {
			IPath refsFilePath = savedState.lookup(getSavedStateFileKey());
			if (refsFilePath != null) {
				File dependenciesFile = UiPlugin.getDefault().getStateLocation().append(refsFilePath).toFile();
				try {
					load(new GZIPInputStream(new BufferedInputStream(new FileInputStream(dependenciesFile))));
				} catch (Throwable t) {
					//- Oops, let's forget anything and start with a clean registry
					clear();
					UiPlugin.logWarning("Unable to load proxy node references file for project " + this.project.getName(), t); //$NON-NLS-1$
				}
			}
		}
	}

	private void addReference(IFile referencer, Reference ref) {
		Assert.isNotNull(ref);
		Collection refs = (Collection) references.get(referencer);
		if(refs == null) {
			//- this proxy has not yet references
			refs = new HashSet();
			references.put(referencer, refs);
		}
		refs.add(ref);
		hasChanged = true;
	}
	
	public void addReference(IFile referencer, String refType, IFile referenced) {
		addReference(referencer, new Reference(refType, referenced));
	}
	
	public void addBidirectionalReference(IFile referencer, String refType, IFile referenced, String oppositeRefType) {
		addReference(referencer, refType, referenced);
		addReference(referenced, oppositeRefType, referencer);
	}

	public void removeReference(IFile referencer, IFile referenced) {
		Assert.isNotNull(referenced);
		Collection refs = (Collection) references.get(referencer);
		if(refs != null) {
			for (Iterator it = refs.iterator(); it.hasNext();) {
				Reference ref = (Reference) it.next();
				if(referenced.equals(ref.getReferenced())) {
					it.remove();
					hasChanged = true;
				}
			}
		}
	}

	public void removeReferences(IFile referencer) {
		references.remove(referencer);
	}
	
	/**
	 * Returns the list of referenced proxy node associated to the given referencer.
	 * Only references matching with the given type are considered
	 * @param referencer
	 * @param refType
	 * @return
	 */
	public Collection getReferences(IFile referencer, String refType) {
		Assert.isNotNull(refType,"type should be not null"); //$NON-NLS-1$
		Collection refs = (Collection) references.get(referencer);
		if(refs != null) {
			List res = new LinkedList();
			for (Iterator it = refs.iterator(); it.hasNext();) {
				Reference ref = (Reference) it.next();
				if(refType.equals(ref.getType())) {
					IProxyNode proxy = FileProxyNodeCache.getInstance().getProxy(ref.getReferenced());
					if(proxy == null) {
					//	it.remove();
					// bugzilla 236422 : don't delete the reference, so that we can keep reference on items inside closed projects
					} else {
						res.add(proxy);
					}
				}
			}
			return res;
		} else {
			return Collections.EMPTY_LIST;
		}
	}
	
	/**
	 * Returns all references of type {@link Reference} registered for the given referencer.
	 * This method is used to persistence.
	 * @param referencer
	 * @return all references of type {@link Reference} registered for the given referencer.
	 */
	private Collection getAllReferences(IFile referencer) {
		Collection list = (Collection) references.get(referencer);
		return list != null ? list : Collections.EMPTY_LIST;
	}
	
	public Set getReferencerFiles() {
		Set res = new HashSet();
		Set keys = references.keySet();
		if (keys != null) {
			for (Iterator it = keys.iterator(); it.hasNext();) {
				res.add((IFile) it.next());
			}
		}
		return res;
	}
	
	public Set getReferenceTypes(IFile referencer) {
		Set res = new HashSet();
		Collection refs = (Collection) references.get(referencer);
		if(refs != null) {
			for (Iterator it = refs.iterator(); it.hasNext();) {
				Reference ref = (Reference) it.next();
				res.add(ref.getType());
			}
		}
		return res;
	}
	
	/**
	 * Clears the registry
	 */
	public void clear() {
		references.clear();
		hasChanged = false;
	}

	/**
	 * Returns the File to use for saving and restoring the last built state for the given project.
	 */
	private File getPersistenceFile(int saveNumber) {
		if (!project.exists()) return null;
		String depsFileName = project.getName() + ".references-" + Integer.toString(saveNumber); //$NON-NLS-1$
		return UiPlugin.getDefault().getStateLocation().append(depsFileName).toFile();
	}
	
	private IPath getSavedStateFileKey() {
		return new Path("references").append(project.getName()); //$NON-NLS-1$
	}

	public void doneSaving(ISaveContext context) {
		if(hasChanged) {
			hasChanged = false;
			//- cleanup previous saved file
			File depsFile = getPersistenceFile(context.getPreviousSaveNumber());
			if (depsFile.exists()) {
				depsFile.delete();
			}
		}
	}

	public void prepareToSave(ISaveContext context) throws CoreException {
		//- nop
	}

	public void rollback(ISaveContext context) {
		//- Remove the file we generated
		File file = getPersistenceFile(context.getSaveNumber());
		if (file.exists()) {
			file.delete();
		}
	}

	public void saving(ISaveContext context) throws CoreException {
		//- if context.getKind() == ISaveContext.PROJECT_SAVE then only 
		//- the instance of this class that has this.project == context.getProject is called
		if(hasChanged && project.exists()) {
			int saveNumber = context.getSaveNumber();
			File depsFile = getPersistenceFile(saveNumber);
			try {
				save(depsFile);
				context.map(getSavedStateFileKey(), new Path(depsFile.getName()));
				context.needSaveNumber();
			} catch (IOException e) {
				UiPlugin.logError("Unexpected exception during references saving operation.", e); //$NON-NLS-1$
			}			
		}
	}

	private void load(InputStream stream) throws WorkbenchException, IOException {
		InputStreamReader reader = new InputStreamReader(stream);
		IMemento memento = XMLMemento.createReadRoot(reader);
		reader.close();
		IMemento [] referencers = memento.getChildren(TAG_REFERENCER);
		for (int i = 0; i < referencers.length; i++) {
			String underlyingResName = referencers[i].getString(TestUIConstants.TAG_UNDERLYING_RESOURCE);
			IFile refFile = ResourcesPlugin.getWorkspace().getRoot().getFile(Path.fromPortableString(underlyingResName));
			if(refFile != null) {
				loadProxyReferences(refFile, referencers[i]);
			}
		}
	}

	private void loadProxyReferences(IFile file, IMemento memento) {
		//- retrieving references
		IMemento [] refStates = memento.getChildren(TAG_REFERENCE);
		IFile refFile;
		for (int j = 0; j < refStates.length; j++) {
			IMemento refMemento = refStates[j];
			String underlyingResName = refMemento.getString(TestUIConstants.TAG_UNDERLYING_RESOURCE);
			String refType = refMemento.getString(TAG_REF_TYPE);
			refFile = ResourcesPlugin.getWorkspace().getRoot().getFile(Path.fromPortableString(underlyingResName));
			if(refFile.exists()) {
				addReference(file, new Reference(refType, refFile));
			} else {
				//- be sure that next save will purge this dead entry
				hasChanged = true;
			}
		} 
	}

	private void save(File depsFile) throws IOException {
		OutputStreamWriter writer = new OutputStreamWriter(new GZIPOutputStream(new BufferedOutputStream(new FileOutputStream(depsFile))));
		IMemento memento = XMLMemento.createWriteRoot("references"); //$NON-NLS-1$
		for (Iterator it = references.keySet().iterator(); it.hasNext();) {
			saveProxyReferences((IFile) it.next(), memento);
		}
		((XMLMemento) memento).save(writer);
		writer.close();
	}

	private void saveProxyReferences(IFile file, IMemento memento) {
		//- save the referencer proxy itself
		IMemento referencerMemento = memento.createChild(TAG_REFERENCER);
		referencerMemento.putString(TestUIConstants.TAG_UNDERLYING_RESOURCE, file.getFullPath().toPortableString());
		//- save references as well
		for (Iterator it = getAllReferences(file).iterator(); it.hasNext();) {
			Reference ref = (Reference) it.next();
			IMemento referenceMemento = referencerMemento.createChild(TAG_REFERENCE);
			referenceMemento.putString(TAG_REF_TYPE, ref.getType());
			referenceMemento.putString(TestUIConstants.TAG_UNDERLYING_RESOURCE, ref.getReferenced().getFullPath().toPortableString());
		}

	}
}
