/********************************************************************** 
 * Copyright (c) 2008, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ReferenceRegistry.java,v 1.2 2009/09/29 20:10:33 bjerome Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy.reference;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ISaveContext;
import org.eclipse.core.resources.ISaveParticipant;
import org.eclipse.core.resources.ISavedState;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.util.TestCaseInvocationDeleter;

/**
 * @author Jerome Gout
 * @author Jerome Bozier
 * @version September 28, 2009
 */
public class ReferenceRegistry implements ISaveParticipant {
	
	private static ReferenceRegistry instance;
	private Map projectRegistries;
	private IReferenceRegistrySavedState internalSavedState;
	
	private class SavedState implements IReferenceRegistrySavedState {
		private ISavedState savedState;
		public SavedState(ISavedState state) {
			savedState = state;
		}
		public IPath lookup(IPath path) {
			return savedState.lookup(path);
		}
	}
	
	public static ReferenceRegistry getInstance() {
		if(instance == null) {
			instance = new ReferenceRegistry();
		}
		return instance;
	}
	
	private ReferenceRegistry() {
		 projectRegistries = new HashMap();
		 try {
	        ISavedState lastState = ResourcesPlugin.getWorkspace().addSaveParticipant(UiPlugin.getDefault(), this);
	        this.internalSavedState = (lastState != null) ? new SavedState(lastState) : null;
		} catch (CoreException e) {
			//- unable to retrieve the previous saved state, start with empty state.
		}
		TestCaseInvocationDeleter.getInstance(); // just to register it
	}
	
	private ProjectReferenceRegistry getRegistry(IFile referencer) {
		IProject project = referencer.getProject();
		if (project.isAccessible()) {
			return getProjectReferenceRegistry(project);
		}
		return null;
	}

	public ProjectReferenceRegistry getProjectReferenceRegistry(IProject project) {
		ProjectReferenceRegistry registry =	(ProjectReferenceRegistry)this.projectRegistries.get(project);
		if (registry == null) {
			//- Not initialized yet
			registry = new ProjectReferenceRegistry(project, this.internalSavedState);
			this.projectRegistries.put(project, registry);
		}
		return registry;
	}

	public void removeProjectReferenceRegistry(IProject project) {
		projectRegistries.remove(project); 
	}
	
	public void addReference(IFile referencer, String refType, IFile referenced) {
		ProjectReferenceRegistry registry = getRegistry(referencer);
		if (registry != null) {
			registry.addReference(referencer, refType, referenced);
		}
	}
		
	public void addBidirectionalReference(IFile referencer, String refType, IFile referenced, String oppositeRefType) {
		ProjectReferenceRegistry registryReferencer = getRegistry(referencer);
		ProjectReferenceRegistry registryReferenced = getRegistry(referenced);
		if (registryReferencer != null) {
			registryReferencer.addReference(referencer, refType, referenced);
		}
		if (registryReferenced != null) {
			registryReferenced.addReference(referenced, oppositeRefType, referencer);
		}
	}
	
	public void removeReference(IFile referencer, IFile referenced) {
		ProjectReferenceRegistry registry = getRegistry(referencer);
		if (registry != null) {
			registry.removeReference(referencer, referenced);
		}
	}
	
	public Collection getReferences(IFile referencer, String refType) {
		ProjectReferenceRegistry registry = getRegistry(referencer);
		if (registry != null) {
			return registry.getReferences(referencer, refType);
		}
		return Collections.EMPTY_LIST;
	}
	
	public Set getReferenceTypes(IFile referencer) {
		ProjectReferenceRegistry registry = getRegistry(referencer);
		if (registry != null) {
			return registry.getReferenceTypes(referencer);
		}
		return Collections.EMPTY_SET;
	}
	
	//- Persistence methods, delegation to project focused registries
	
	public void doneSaving(ISaveContext context) {
		if (context.getKind() == ISaveContext.PROJECT_SAVE) {
			IProject project = context.getProject();
			ProjectReferenceRegistry registry =	(ProjectReferenceRegistry)projectRegistries.get(project);
			if (registry != null) {
				registry.doneSaving(context);
			}
		} else {
			Iterator it = projectRegistries.values().iterator();
			while (it.hasNext()) {
				ProjectReferenceRegistry registry = (ProjectReferenceRegistry) it.next();
				registry.doneSaving(context);
			}
		}			
	}

	public void prepareToSave(ISaveContext context) throws CoreException {
		if (context.getKind() == ISaveContext.PROJECT_SAVE) {
			IProject project = context.getProject();
			ProjectReferenceRegistry registry =	(ProjectReferenceRegistry)projectRegistries.get(project);
			if (registry != null) {
				registry.prepareToSave(context);
			}
		} else {
			Iterator it = projectRegistries.values().iterator();
			while (it.hasNext()) {
				ProjectReferenceRegistry registry = (ProjectReferenceRegistry) it.next();
				registry.prepareToSave(context);
			}
		}	
	}

	public void rollback(ISaveContext context) {
		if (context.getKind() == ISaveContext.PROJECT_SAVE) {
			IProject project = context.getProject();
			ProjectReferenceRegistry registry =	(ProjectReferenceRegistry)projectRegistries.get(project);
			if (registry != null) {
				registry.rollback(context);
			}
		} else {
			Iterator it = projectRegistries.values().iterator();
			while (it.hasNext()) {
				ProjectReferenceRegistry registry = (ProjectReferenceRegistry) it.next();
				registry.rollback(context);
			}
		}
	}

	public void saving(ISaveContext context) throws CoreException {
		if (context.getKind() == ISaveContext.PROJECT_SAVE) {
			IProject project = context.getProject();
			ProjectReferenceRegistry registry =	(ProjectReferenceRegistry)projectRegistries.get(project);
			if (registry != null) {
				registry.saving(context);
			}
		} else {
			Iterator it = projectRegistries.values().iterator();
			while (it.hasNext()) {
				ProjectReferenceRegistry registry = (ProjectReferenceRegistry) it.next();
				registry.saving(context);
			}
		}
	}

	public void removeReferences(IFile referencer) {
		ProjectReferenceRegistry registry = getRegistry(referencer);
		if (registry != null) {
			registry.removeReferences(referencer);
		}
	}

}
