/********************************************************************** 
 * Copyright (c) 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ReferenceTypeRegistry.java,v 1.1 2008/03/20 19:49:53 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy.reference;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.ui.UiPlugin;

/**
 * Registry to manage reference types (<code>ReferenceType</code>).
 * @author jgout
 * @version February 26, 2008
 * @since 4.3
 */
public class ReferenceTypeRegistry {

	private static ReferenceTypeRegistry instance;
	private Map types;
	
	static class ReferenceType {

		private String name;
		private boolean owns;
		private boolean isExplicit;
		private String oppositeId;

		public ReferenceType(String name, boolean owns, boolean isExplicit, String oppositeId) {
			this.name = name;
			this.owns = owns;
			this.isExplicit = isExplicit;
			this.oppositeId = oppositeId;
		}

		public boolean owns() {
			return owns;
		}

		public boolean isExplicit() {
			return isExplicit;
		}

		public String getOppositeId() {
			return oppositeId;
		}

		public String getName() {
			return name;
		}
	}
	
	private ReferenceTypeRegistry() {
		types = new HashMap();
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + ".testNavigatorReferenceTypes"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) {
				IConfigurationElement element = members[i];
				if ("type".equals(element.getName())) { //$NON-NLS-1$
					String id = element.getAttribute("id"); //$NON-NLS-1$
					String name = element.getAttribute("name"); //$NON-NLS-1$
					boolean owns = Boolean.valueOf(element.getAttribute("owns")).booleanValue(); //$NON-NLS-1$
					boolean isExplicit = Boolean.valueOf(element.getAttribute("isExplicit")).booleanValue(); //$NON-NLS-1$
					String oppositeId = element.getAttribute("oppositeReferenceTypeId"); //$NON-NLS-1$
					types.put(id, new ReferenceType(name, owns, isExplicit, oppositeId));
				}
			}
		}
	}
	
	public static ReferenceTypeRegistry getInstance() {
		if(instance == null) {
			instance = new ReferenceTypeRegistry();
		}
		return instance;
	}
	
	/**
	 * Tests whether the reference type (given through its id) owns.
	 * @param refTypeId the id of the reference type
	 * @return <code>true</code> if the reference type (given through its id) owns or <code>false</code> otherwise.
	 */
	public boolean owns(String refTypeId) {
		ReferenceType type = (ReferenceType)types.get(refTypeId);
		if(type != null) {
			return type.owns();
		}
		UiPlugin.logWarning("Unable to find reference type: "+refTypeId+" default setting is applied."); //$NON-NLS-1$  //$NON-NLS-2$
		return false;
	}
	
	/**
	 * Tests whether the reference type (given through its id) is explicit.
	 * @param refTypeId the id of the reference type
	 * @return <code>true</code> if the reference type (given through its id) is explicit or <code>false</code> otherwise.
	 */
	public boolean isExplicit(String refTypeId) {
		if (refTypeId != null) {
			ReferenceType type = (ReferenceType) types.get(refTypeId);
			if (type != null) {
				return type.isExplicit();
			}
			UiPlugin.logWarning("Unable to find reference type: " + refTypeId + " default setting is applied."); //$NON-NLS-1$ //$NON-NLS-2$
		}		
		return false;
	}
	
	/**
	 * Returns the human readable name of the given reference type. 
	 * @param refTypeId the id of the reference type.
	 * @return the human readable name of the given reference type. 
	 */
	public String getName(String refTypeId) {
		ReferenceType type = (ReferenceType)types.get(refTypeId);
		if(type != null) {
			return type.getName();
		}
		UiPlugin.logWarning("Unable to find reference type: "+refTypeId+" default setting is applied."); //$NON-NLS-1$ //$NON-NLS-2$
		return "Unknown reference"; //$NON-NLS-1$
	}
	
	/**
	 * Returns the opposite reference type id of a given one if it has been defined or <code>null</code> otherwise.
	 * @param refTypeId the reference type id 
	 * @return the opposite reference type id of a given one if it has been defined or <code>null</code> otherwise.
	 */
	public String getOppositeReferenceType(String refTypeId) {
		ReferenceType type = (ReferenceType)types.get(refTypeId);
		if(type != null) {
			return type.getOppositeId();
		}
		UiPlugin.logWarning("Unable to find reference type: "+refTypeId+" default setting is applied."); //$NON-NLS-1$ //$NON-NLS-2$
		return "Unknown reference"; //$NON-NLS-1$
	}
	
}
