/*******************************************************************************
 * Copyright (c) 2008, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * Change that deletes an existing file.
 * 
 * 
 * @author  Jerome Bozier
 * @version November 20, 2009
 * @since   December 23, 2008
 */
public class DeleteFileChange extends PhysicalDeleteChange {

	private IFile file;
	private String name;
	
	public DeleteFileChange(IFile file){
		Assert.isNotNull(file, "file"); //$NON-NLS-1$
		this.file = file;
		updateAssociationMapping(file);
	}
	
	public DeleteFileChange(IFile file, String name){
		this(file);
		this.name = name;
		updateAssociationMapping(file);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ltk.core.refactoring.Change#isValid(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException {
		RefactoringStatus status = new RefactoringStatus();
		if (!file.exists()) {
			status.addError(RefactoringMessages.NO_FILE);
		}
		if (file.isReadOnly()) {
			status.addWarning(RefactoringMessages.READ_ONLY_FILE);
		}
		return status;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ltk.core.refactoring.Change#perform(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public Change perform(IProgressMonitor pm) throws CoreException {
		
		try {
			pm.beginTask(RefactoringMessages.DELETING_FILE, 1); 
			Change undo = null;
			if(file.exists()){
				file.refreshLocal(0, null);
				if (file.isAccessible() && (!file.isReadOnly())) {				
					try {
						TrashBox.instance().trashResource(file);
						undo = TrashBox.instance().createUntrashChange(file);
					} catch (Exception e) {
						UiPlugin.logError("error while putting file into trashbox : "+file.getName());
						UiPlugin.logError(e);
					}
					file.delete(true, true, pm);
				}
			}
			return undo;
		} 
		finally {
			pm.done();
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ltk.core.refactoring.Change#getName()
	 */
	public String getName() {
		if(name != null) {
			return name;
		} else {
			return NLS.bind(RefactoringMessages.DELETE_FILE, file.getFullPath().toString());
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ltk.core.refactoring.Change#getModifiedElement()
	 */
	public Object getModifiedElement() {
		return file;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ltk.core.refactoring.Change#getAffectedObjects()
	 */
	public Object[] getAffectedObjects() {
		Object [] res = new Object[1];
		res[0] = file;
		return res;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ltk.core.refactoring.Change#initializeValidationData(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void initializeValidationData(IProgressMonitor pm) {
		//No-operation.
	}
}

